/* The GPL applies to this program.
  In addition, as a special exception, the copyright holders give
  permission to link the code of portions of this program with the
  OpenSSL library under certain conditions as described in each
  individual source file, and distribute linked combinations
  including the two.
  You must obey the GNU General Public License in all respects
  for all of the code used other than OpenSSL.  If you modify
  file(s) with this exception, you may extend this exception to your
  version of the file(s), but you are not obligated to do so.  If you
  do not wish to do so, delete this exception statement from your
  version.  If you delete this exception statement from all source
  files in the program, then also delete it here.
*/

#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#define _LINUX_TIME_H 1	/* to get things compile on kernel 2.6.x */
#include <linux/videodev.h>

#include "pwc-ioctl.h"

char *device = "/dev/video0";

#define SET_PAN		0
#define SET_TILT	1

void error_exit(char *what_ioctl)
{
	fprintf(stderr, "Error while doing ioctl %s: %s\n", what_ioctl, strerror(errno));

	/* commented out: some versions of the driver seem to return
	 * unexpected errors */
	/* exit(1); */
}

void check_device(int *fd)
{
	if (*fd == -1)
	{
		/* open device */
		*fd = open(device, O_RDWR);
		if (*fd == -1)
		{
			fprintf(stderr, "Error while accessing device %s: %s\n", device, strerror(errno));
			exit(1);
		}
	}
}

void not_supported(char *what)
{
	printf("%s is not supported by the combination\n", what);
	printf("of your webcam and the driver.\n");
}

void dump_current_settings(int fd)
{
	struct video_capability vcap;
	struct video_window vwin;
	struct video_picture vpic;
	struct pwc_probe pwcp;
	int dummy;
	struct pwc_whitebalance pwcwb;
	struct pwc_leds pwcl;
	struct pwc_mpt_range pmr;
	struct pwc_mpt_angles pma;
	struct pwc_serial ps;

	/* get name */
	if (ioctl(fd, VIDIOCGCAP, &vcap) == -1)
		error_exit("VIDIOCGCAP");
	printf("Current device: %s\n", vcap.name);

	/* verify that it IS a Philips Webcam */
	if (ioctl(fd, VIDIOCPWCPROBE, &pwcp) == -1)
		error_exit("VIDIOCPWCPROBE");
	if (strcmp(vcap.name, pwcp.name) != 0)
		printf("Warning: this might not be a Philips compatible webcam!\n");
	printf("VIDIOCPWCPROBE returns: %s - %d\n", pwcp.name, pwcp.type);

	if (ioctl(fd, VIDIOCPWCGSERIAL, &ps) == -1)
		error_exit("VIDIOCPWCGSERIAL");
	printf("Serial number: %s\n", ps.serial);

	/* get resolution/framerate */
	if (ioctl(fd, VIDIOCGWIN, &vwin) == -1)
		error_exit("VIDIOCGWIN");
	printf("Resolution (x, y): %d, %d\n", vwin.width, vwin.height);
	printf("Offset: %d, %d\n", vwin.x, vwin.y);
	if (vwin.flags & PWC_FPS_FRMASK)
		printf("Framerate: %d\n", (vwin.flags & PWC_FPS_FRMASK) >> PWC_FPS_SHIFT);

	/* color (etc.) settings */
	if (ioctl(fd, VIDIOCGPICT, &vpic) == -1)
		error_exit("VIDIOCGPICT");
	printf("Brightness: %d\n", vpic.brightness);
	printf("Hue: %d\n", vpic.hue);
	printf("Colour: %d\n", vpic.colour);
	printf("Contrast: %d\n", vpic.contrast);
	printf("Whiteness: %d\n", vpic.whiteness);
	printf("Palette: ");
	switch(vpic.palette) {
	case VIDEO_PALETTE_GREY:
		printf("Linear intensity grey scale (255 is brightest).\n");
		break;
	case VIDEO_PALETTE_HI240:
		printf("The BT848 8bit colour cube.\n");
		break;
	case VIDEO_PALETTE_RGB565:
		printf("RGB565 packed into 16 bit words.\n");
		break;
	case VIDEO_PALETTE_RGB555:
		printf("RGV555 packed into 16 bit words, top bit undefined.\n");
		break;
	case VIDEO_PALETTE_RGB24:
		printf("RGB888 packed into 24bit words.\n");
		break;
	case VIDEO_PALETTE_RGB32:
		printf("RGB888 packed into the low 3 bytes of 32bit words. The top 8bits are undefined.\n");
		break;
	case VIDEO_PALETTE_YUV422:
		printf("Video style YUV422 - 8bits packed 4bits Y 2bits U 2bits V\n");
		break;
	case VIDEO_PALETTE_YUYV:
		printf("Describe me\n");
		break;
	case VIDEO_PALETTE_UYVY:
		printf("Describe me\n");
		break;
	case VIDEO_PALETTE_YUV420:
		printf("YUV420 capture\n");
		break;
	case VIDEO_PALETTE_YUV411:
		printf("YUV411 capture\n");
		break;
	case VIDEO_PALETTE_RAW:
		printf("RAW capture (BT848)\n");
		break;
	case VIDEO_PALETTE_YUV422P:
		printf("YUV 4:2:2 Planar\n");
		break;
	case VIDEO_PALETTE_YUV411P:
		printf("YUV 4:1:1 Planar\n");
		break;
	case VIDEO_PALETTE_YUV420P:
		printf("YUV 4:2:0 Planar\n");
		break;
	case VIDEO_PALETTE_YUV410P:
		printf("YUV 4:1:0 Planar\n");
		break;
	default:
		printf("Unknown! (%d)\n", vpic.palette);
	}

	if (ioctl(fd, VIDIOCPWCGCQUAL, &dummy) == -1)
		error_exit("VIDIOCPWCGCQUAL");
	printf("Compression preference: %d\n", dummy);

	if (ioctl(fd, VIDIOCPWCGAGC, &dummy) == -1)
		error_exit("VIDIOCPWCGAGC");
	printf("Automatic gain control: %d\n", dummy);

	if (ioctl(fd, VIDIOCPWCGAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCGAWB");
	printf("Whitebalance mode: ");
	if (pwcwb.mode == PWC_WB_AUTO)
		printf("auto\n");
	else if (pwcwb.mode == PWC_WB_MANUAL)
		printf("manual (red: %d, blue: %d)\n", pwcwb.manual_red, pwcwb.manual_blue);
	else if (pwcwb.mode == PWC_WB_INDOOR)
		printf("indoor\n");
	else if (pwcwb.mode == PWC_WB_OUTDOOR)
		printf("outdoor\n");
	else if (pwcwb.mode == PWC_WB_FL)
		printf("artificial lightning ('fl')\n");
	else
		printf("unknown!\n");

	if (ioctl(fd, VIDIOCPWCGLED, &pwcl) != -1)
	{
		printf("Led ON time: %d\n", pwcl.led_on);
		printf("Led OFF time: %d\n", pwcl.led_off);
	}
	else
	{
		not_supported("Blinking of LED");
	}

	if (ioctl(fd, VIDIOCPWCGCONTOUR, &dummy) == -1)
		error_exit("VIDIOCPWCGCONTOUR");
	printf("Sharpness: %d\n", dummy);

	if (ioctl(fd, VIDIOCPWCGBACKLIGHT, &dummy) == -1)
		error_exit("VIDIOCPWCGBACKLIGHT");
	printf("Backlight compensation mode: ");
	if (dummy == 0) printf("off\n"); else printf("on\n");

	if (ioctl(fd, VIDIOCPWCGFLICKER, &dummy) != -1)
	{
		printf("Anti-flicker mode: ");
		if (dummy == 0) printf("off\n"); else printf("on\n");
	}
	else
	{
		not_supported("Anti-flicker mode");
	}

	if (ioctl(fd, VIDIOCPWCGDYNNOISE, &dummy) != -1)
	{
		printf("Noise reduction mode: %d ", dummy);
		if (dummy == 0) printf("(none)");
		else if (dummy == 3) printf("(high)");
		printf("\n");
	}
	else
	{
		not_supported("Noise reduction mode");
	}

	if (ioctl(fd, VIDIOCPWCMPTGRANGE, &pmr) == -1)
	{
		not_supported("Pan/tilt range");
	}
	else
	{
		printf("Pan min. : %d, max.: %d\n", pmr.pan_min, pmr.pan_max);
		printf("Tilt min.: %d, max.: %d\n", pmr.tilt_min, pmr.tilt_max);
	}

	pma.absolute=1;
	if (ioctl(fd, VIDIOCPWCMPTGANGLE, &pma) == -1)
	{
		not_supported("Get pan/tilt position");
	}
	else
	{
		printf("Pan  (degrees * 100): %d\n", pma.pan);
		printf("Tilt (degrees * 100): %d\n", pma.tilt);
	}
}

void query_pan_tilt_status(fd)
{
	struct pwc_mpt_status pms;

	if (ioctl(fd, VIDIOCPWCMPTSTATUS, &pms) == -1)
		error_exit("VIDIOCPWCMPTSTATUS");

	printf("Status: %d\n", pms.status);
	printf("Time pan: %d\n", pms.time_pan);
	printf("Time tilt: %d\n", pms.time_tilt);
}

void reset_pan_tilt(int fd, int what)
{
	if (ioctl(fd, VIDIOCPWCMPTRESET, &what) == -1)
		error_exit("VIDIOCPWCMPTRESET");
}

void set_pan_or_tilt(int fd, char what, int value)
{
	struct pwc_mpt_angles pma;

	pma.absolute=1;
	if (ioctl(fd, VIDIOCPWCMPTGANGLE, &pma) == -1)
		error_exit("VIDIOCPWCMPTGANGLE");

	if (what == SET_PAN)
		pma.pan = value;
	else if (what == SET_TILT)
		pma.tilt = value;

	if (ioctl(fd, VIDIOCPWCMPTSANGLE, &pma) == -1)
		error_exit("VIDIOCPWCMPTSANGLE");
}

void set_framerate(int fd, int framerate)
{
	struct video_window vwin;

	/* get resolution/framerate */
	if (ioctl(fd, VIDIOCGWIN, &vwin) == -1)
		error_exit("VIDIOCGWIN");

	if (vwin.flags & PWC_FPS_FRMASK)
	{
		/* set new framerate */
		vwin.flags &= ~PWC_FPS_FRMASK;
		vwin.flags |= (framerate << PWC_FPS_SHIFT);
   
		if (ioctl(fd, VIDIOCSWIN, &vwin) == -1)
			error_exit("VIDIOCSWIN");
	}
	else
	{
		fprintf(stderr, "This device doesn't support setting the framerate.\n");
		exit(1);
	}
}

void flash_settings(int fd)
{
	if (ioctl(fd, VIDIOCPWCSUSER) == -1)
		error_exit("VIDIOCPWCSUSER");
}

void restore_settings(int fd)
{
	if (ioctl(fd, VIDIOCPWCRUSER) == -1)
		error_exit("VIDIOCPWCRUSER");
}

void restore_factory_settings(int fd)
{
	if (ioctl(fd, VIDIOCPWCFACTORY) == -1)
		error_exit("VIDIOCPWCFACTORY");
}

void set_compression_preference(int fd, int pref)
{
	if (ioctl(fd, VIDIOCPWCSCQUAL, &pref) == -1)
		error_exit("VIDIOCPWCSCQUAL");
}

void set_automatic_gain_control(int fd, int pref)
{
	if (ioctl(fd, VIDIOCPWCSAGC, &pref) == -1)
		error_exit("VIDIOCPWCSAGC");
}

void set_shutter_speed(int fd, int pref)
{
	if (ioctl(fd, VIDIOCPWCSSHUTTER, &pref) == -1)
		error_exit("VIDIOCPWCSSHUTTER");
}

void set_automatic_white_balance_mode(int fd, char *mode)
{
	struct pwc_whitebalance pwcwb;

	if (ioctl(fd, VIDIOCPWCGAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCGAWB");

	if (strcasecmp(mode, "auto") == 0)
		pwcwb.mode = PWC_WB_AUTO;
	else if (strcasecmp(mode, "manual") == 0)
		pwcwb.mode = PWC_WB_MANUAL;
	else if (strcasecmp(mode, "indoor") == 0)
		pwcwb.mode = PWC_WB_INDOOR;
	else if (strcasecmp(mode, "outdoor") == 0)
		pwcwb.mode = PWC_WB_OUTDOOR;
	else if (strcasecmp(mode, "fl") == 0)
		pwcwb.mode = PWC_WB_FL;
	else
	{
		fprintf(stderr, "'%s' is not a known white balance mode.\n", mode);
		exit(1);
	}

	if (ioctl(fd, VIDIOCPWCSAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCSAWB");
}

void set_automatic_white_balance_mode_red(int fd, int val)
{
	struct pwc_whitebalance pwcwb;

	if (ioctl(fd, VIDIOCPWCGAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCGAWB");

	pwcwb.manual_red = val;

	if (ioctl(fd, VIDIOCPWCSAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCSAWB");
}

void set_automatic_white_balance_mode_blue(int fd, int val)
{
	struct pwc_whitebalance pwcwb;

	if (ioctl(fd, VIDIOCPWCGAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCGAWB");

	pwcwb.manual_blue = val;

	if (ioctl(fd, VIDIOCPWCSAWB, &pwcwb) == -1)
		error_exit("VIDIOCPWCSAWB");
}

void set_automatic_white_balance_speed(int fd, int val)
{
	struct pwc_wb_speed pwcwbs;

	if (ioctl(fd, VIDIOCPWCGAWBSPEED, &pwcwbs) == -1)
		error_exit("VIDIOCPWCGAWBSPEED");

	pwcwbs.control_speed = val;

	if (ioctl(fd, VIDIOCPWCSAWBSPEED, &pwcwbs) == -1)
		error_exit("VIDIOCPWCSAWBSPEED");
}

void set_automatic_white_balance_delay(int fd, int val)
{
	struct pwc_wb_speed pwcwbs;

	if (ioctl(fd, VIDIOCPWCGAWBSPEED, &pwcwbs) == -1)
		error_exit("VIDIOCPWCGAWBSPEED");

	pwcwbs.control_delay = val;

	if (ioctl(fd, VIDIOCPWCSAWBSPEED, &pwcwbs) == -1)
		error_exit("VIDIOCPWCSAWBSPEED");
}

void set_led_on_time(int fd, int val)
{
	struct pwc_leds pwcl;

	if (ioctl(fd, VIDIOCPWCGLED, &pwcl) == -1)
		error_exit("VIDIOCPWCGLED");

	pwcl.led_on = val;

	if (ioctl(fd, VIDIOCPWCSLED, &pwcl) == -1)
		error_exit("VIDIOCPWCSLED");
}

void set_led_off_time(int fd, int val)
{
	struct pwc_leds pwcl;

	if (ioctl(fd, VIDIOCPWCGLED, &pwcl) == -1)
		error_exit("VIDIOCPWCGLED");

	pwcl.led_off = val;

	if (ioctl(fd, VIDIOCPWCSLED, &pwcl) == -1)
		error_exit("VIDIOCPWCSLED");
}

void set_sharpness(int fd, int val)
{
	if (ioctl(fd, VIDIOCPWCSCONTOUR, &val) == -1)
		error_exit("VIDIOCPWCSCONTOUR");
}

void set_backlight_compensation(int fd, int val)
{
	if (ioctl(fd, VIDIOCPWCSBACKLIGHT, &val) == -1)
		error_exit("VIDIOCPWCSBACKLIGHT");
}

void set_antiflicker_mode(int fd, int val)
{
	if (ioctl(fd, VIDIOCPWCSFLICKER, &val) == -1)
		error_exit("VIDIOCPWCSFLICKER");
}

void set_noise_reduction(int fd, int val)
{
	if (ioctl(fd, VIDIOCPWCSDYNNOISE, &val) == -1)
		error_exit("VIDIOCPWCSDYNNOISE");
}

void usage(void)
{
	fprintf(stderr, "-d device  device to use (default: /dev/video0)\n");
	fprintf(stderr, "-p	dump current settings\n");
	fprintf(stderr, "-f x	set framerate (0...63)\n");
	fprintf(stderr, "-b	store settings in nonvolatile RAM\n");
	fprintf(stderr, "-r	restore settings from nonvolatile RAM\n");
	fprintf(stderr, "-x	restore factory settings\n");
	fprintf(stderr, "-c x	set compression preference (0-3)\n");
	fprintf(stderr, "-g x	set automatic gain control (0...65535)\n");
	fprintf(stderr, "-s x	set shutter speed (1...65535)\n");
	fprintf(stderr, "-w auto/manual/indoor/outdoor/fl\n");
	fprintf(stderr, "	set automatic white balance mode\n");
	fprintf(stderr, "-a x	red-balance (only if -w manual) (0...65535)\n");
	fprintf(stderr, "-e x	blue-balance (idem) (0...65535)\n");
	fprintf(stderr, "-i x	speed of automatic white balance (1...65535)\n");
	fprintf(stderr, "-j x	delay for automatic white balance (1...65535)\n");
	fprintf(stderr, "-k x	set led on-time (0...25500ms\n");
	fprintf(stderr, "-l x	set led off-time\n");
	fprintf(stderr, "-m x	set electronic sharpness (0...65535)\n");
	fprintf(stderr, "-n x	set backlight compensation (0=0ff, other=on)\n");
	fprintf(stderr, "-o x	set antiflicker mode (0=0ff, other=on)\n");
	fprintf(stderr, "-q x	set noise reduction mode (0=none...3=high)\n");
	fprintf(stderr, "-t x	reset pan(bit 0) and/or tilt(bit 1)\n");
	fprintf(stderr, "-u	query pan/tilt status\n");
	fprintf(stderr, "-y x	set pan position\n");
	fprintf(stderr, "-z x	set tilt position\n");
	fprintf(stderr, "-h	this help\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "With this tool, you can only set settings specific to the Philips WebCams.\n");
	fprintf(stderr, "For something more generic, see dov4l:\n");
	fprintf(stderr, "http://www.vanheusden.com/dov4l/\n");
}

int main(int argc, char *argv[])
{
	int fd = -1;
	int c;
	int loop;

	fprintf(stderr, "setpwc v" VERSION ", (C) 2003-2005 by folkert@vanheusden.com\n");

	for(loop=1; loop<argc; loop++)
	{
		if (strcmp(argv[loop], "-d") == 0)
		{
			device = argv[loop + 1];
			break;
		}
	}

        while((c = getopt(argc, argv, "d:hf:pbrxc:g:s:w:a:e:i:j:k:l:m:n:o:q:t:uv:y:z:")) != -1)
	{
		switch(c) {
		case 'd':
			device = optarg;
			break;
		case 'h':
			usage();
			return 0;
		case 'f':
			check_device(&fd);
			set_framerate(fd, atoi(optarg));
			break;
		case 'p':
			check_device(&fd);
			dump_current_settings(fd);
			break;
		case 'b':
			check_device(&fd);
			flash_settings(fd);
			break;
		case 'r':
			check_device(&fd);
			restore_settings(fd);
			break;
		case 'x':
			check_device(&fd);
			restore_factory_settings(fd);
			break;
		case 'c':
			check_device(&fd);
			set_compression_preference(fd, atoi(optarg));
			break;
		case 'g':
			check_device(&fd);
			set_automatic_gain_control(fd, atoi(optarg));
			break;
		case 's':
			check_device(&fd);
			set_shutter_speed(fd, atoi(optarg));
			break;
		case 'w':
			check_device(&fd);
			set_automatic_white_balance_mode(fd, optarg);
			break;
		case 'a':
			check_device(&fd);
			set_automatic_white_balance_mode_red(fd, atoi(optarg));
			break;
		case 'e':
			check_device(&fd);
			set_automatic_white_balance_mode_blue(fd, atoi(optarg));
			break;
		case 'i':
			check_device(&fd);
			set_automatic_white_balance_speed(fd, atoi(optarg));
			break;
		case 'j':
			check_device(&fd);
			set_automatic_white_balance_delay(fd, atoi(optarg));
			break;
		case 'k':
			check_device(&fd);
			set_led_on_time(fd, atoi(optarg));
			break;
		case 'l':
			check_device(&fd);
			set_led_off_time(fd, atoi(optarg));
			break;
		case 'm':
			check_device(&fd);
			set_sharpness(fd, atoi(optarg));
			break;
		case 'n':
			check_device(&fd);
			set_backlight_compensation(fd, atoi(optarg));
			break;
		case 'o':
			check_device(&fd);
			set_antiflicker_mode(fd, atoi(optarg));
			break;
		case 'q':
			check_device(&fd);
			set_noise_reduction(fd, atoi(optarg));
			break;
		case 't':
			check_device(&fd);
			reset_pan_tilt(fd, atoi(optarg));
			break;
		case 'u':
			check_device(&fd);
			query_pan_tilt_status(fd);
			break;
		case 'y':
			check_device(&fd);
			set_pan_or_tilt(fd, SET_PAN, atoi(optarg));
			break;
		case 'z':
			check_device(&fd);
			set_pan_or_tilt(fd, SET_TILT, atoi(optarg));
			break;
		default:
			fprintf(stderr, "Internal error: unexpected option (%c %s).\n", c, optarg);
			break;
		}
	}

	if (fd != -1)
		close(fd);

	return 0;
}
