#include <gtk/gtk.h>
#include <glade/glade.h>

#include "silcincludes.h"
#include "silcclient.h"

#include "buddylist.h"
#include "ui_buddylist.h"
#include "log.h"
#include "support.h" /* for CONNECTED/DISCONNECTED */
#include "config.h"
#include "callbacks.h"
#include "keys.h"

#include "common.h"

extern GladeXML *xmlmain;
extern SilkyBuddy *buddies; /* from buddylist.c */
extern silkyStruct *silky; /* from main.c */

/**
 * gui_set_up_buddylist_treeview
 *
 * This creates the treeview for the Buddy List window.
 *
 * @Returns: nothing
 **/
void gui_set_up_buddylist_treeview() {
  GtkWidget *treeview;
  GtkCellRenderer *renderer_text;
  GtkCellRenderer *renderer_pixbuf;
  GtkListStore *liststore;
  GtkTreeViewColumn *column;

  debug("creating buddy list treeview");

  /* create store */
  liststore = gtk_list_store_new (
				  3,
				  G_TYPE_STRING,  /* text column */
				  G_TYPE_STRING,
				  G_TYPE_POINTER /* silkybuddy pointer */
                          );
  treeview = glade_xml_get_widget(xmlmain, "buddylist_treeview");
  if (!treeview) {
    debug("can't find the treeview");
    return;
  }


  /* set the model for treeview, overwrites the old one if present */
  gtk_tree_view_set_model (GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(liststore) );

  renderer_text = gtk_cell_renderer_text_new();
  renderer_pixbuf = gtk_cell_renderer_pixbuf_new();

  /* status info */
  column = gtk_tree_view_column_new();
  gtk_tree_view_column_pack_start(column, renderer_pixbuf, FALSE);
  gtk_tree_view_column_add_attribute(column, renderer_pixbuf, "stock-id", 0);

  /* nickname */
  gtk_tree_view_column_pack_start(column, renderer_text, FALSE);
  gtk_tree_view_column_add_attribute(column, renderer_text, "text", 1);
  gtk_tree_view_column_set_title(column, _("Buddy List") );

  gtk_tree_view_append_column (GTK_TREE_VIEW(treeview), column);


  debug("created");
}




/**
 * refresh_gui_buddylist:
 *
 * This refreshes the Buddy List window.
 *
 * @Returns: nothing
 **/
void refresh_gui_buddylist() {
  GtkListStore *liststore;
  SilkyBuddy *b;
  GtkTreeIter iter;

  debug("refreshing the buddy list UI");

  /* if no columns in the treeview found */
  if( !gtk_tree_view_get_column(GTK_TREE_VIEW(glade_xml_get_widget(xmlmain, "buddylist_treeview")), 0) ) {
    debug("buddy list treeview not initialized, initializing now");
    gui_set_up_buddylist_treeview();
  }

 liststore = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(glade_xml_get_widget(xmlmain, "buddylist_treeview"))));

 /* clear the list */
 gtk_list_store_clear(liststore);


 /* append all buddies */
 for( b = buddies; b; b = b->next ) {
   debug("setting icon for '%s' (%s)", b->nickname, b->fingerprint);

   gtk_list_store_append(liststore, &iter); /* new line */

   /* ICON column */
   if (!CONNECTED) {
     debug("set unknown icon as we are not connected");
     gtk_list_store_set(liststore, &iter, 0, GTK_STOCK_DIALOG_QUESTION, -1); /* column 0, status image */
   } else {
     switch (b->mode) {

     case SILC_UMODE_DETACHED:
       gtk_list_store_set(liststore, &iter, 0, GTK_STOCK_DIALOG_WARNING, -1); /* column 0, status image */
       break;

     case SILC_UMODE_GONE:
       gtk_list_store_set(liststore, &iter, 0, GTK_STOCK_DIALOG_INFO, -1); /* column 0, status image */
       break;

     default:
       /* all other modes use the network icon */
       gtk_list_store_set(liststore, &iter, 0, GTK_STOCK_JUMP_TO, -1); /* column 0, status image */
       break;
     }
   }

   /* store pointer to the buddy for later use */
   gtk_list_store_set(liststore, &iter, 2, b, -1); /* SilkyBuddy */

   /* nickname */
   gtk_list_store_set(liststore, &iter, 1, b->nickname, -1);

 }

 debug("ready");
}



/**
 * on_buddylist_activate:
 * @widget: unused
 * @userdata: unused
 *
 * Refreshes and shows the "window_buddylist" widget.
 *
 * Returns: TRUE
 **/

gboolean on_buddylist_activate (GtkWidget *widget, gpointer userdata) {
  debug("showing buddylist");

  refresh_gui_buddylist();

  /* I18N This is a window title */
  gtk_window_set_title(GTK_WINDOW(glade_xml_get_widget (xmlmain, "window_buddylist")), g_strdup_printf("Silky v%s : %s ", SILKY_VERSION, _("Buddy List") ));

  gtk_widget_show(glade_xml_get_widget(xmlmain, "window_buddylist"));
  return TRUE;
}


/*
  Called as a UI callback when user wants to remove a buddy.
  Asks the buddy to be removed, and updates the UI.
*/
gboolean ui_buddy_remove (GtkWidget *wid, gpointer fingerprint) {
  debug("asking to remove a buddy with fingerprint '%s'", fingerprint);
  buddy_remove_by_fp(fingerprint);

  refresh_gui_buddylist(); /* update the UI */

  return TRUE;
}




/*
  Called from GTK when user selects an user from the buddylist.
  Shows a popup list if right-mouse-button was pressed
*/
gboolean buddy_selected (GtkTreeView *treeview, GdkEventButton *event) {
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;

  SilkyBuddy *b;

  GtkMenu *menu;
  GtkWidget *menuitem;

  debug("selected buddy");

  /* right click */
  if (event && event->button && event->button == 3) {

    selection = gtk_tree_view_get_selection(treeview);

    /* popup the menu only if something was selected */
    if (gtk_tree_selection_get_selected (selection, &model, &iter))
      {
	gtk_tree_model_get (model, &iter, 2, &b, -1);
	debug("selected nickname '%s', fingerprint '%s'", b->nickname, b->fingerprint);

	/* create the menu */
	menu = GTK_MENU(gtk_menu_new());

	if (b->client_id) {
	  debug("client_id found");
	} else {
	  debug("client_id not yet cached");
	}


	/* ADD MENU items and signal handlers*/

	/* WHOIS */

	menuitem = gtk_menu_item_new_with_label(_("Who is"));

	if (silky->conn) {
	  if (b->client_id) {
	    g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_whois), b->client_id);
	  } else if (b->fingerprint) {
	    const gchar *pubkeyfile = gen_keyfile_path(b->fingerprint, "client");
	    g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_whois_pubkey), (gpointer *)pubkeyfile);
	  }
	  gtk_widget_set_sensitive(menuitem, TRUE);
	} else {
	  gtk_widget_set_sensitive(menuitem, FALSE);
	}


	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	if (!silky->conn) { /* disable item if we are not connected */
	  gtk_widget_set_sensitive(menuitem, FALSE);
	}
	gtk_widget_show (menuitem);


	/* MSG */
	menuitem = gtk_menu_item_new_with_label(_("Send message"));

	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	if (!silky->conn || !b->client_id) {
	  gtk_widget_set_sensitive(menuitem, FALSE);
	} else {
	  gtk_widget_set_sensitive(menuitem, TRUE);
	  g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_query), b->client_id);
	}
	gtk_widget_show (menuitem);


	/* a separator here */
	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show(menuitem);

	/* remove from buddylist */
	menuitem = gtk_menu_item_new_with_label(_("Remove Buddy"));
	g_signal_connect(menuitem, "activate", G_CALLBACK(ui_buddy_remove), b->fingerprint);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show (menuitem);

	/* show the popup menu */
	gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
			event->button, event->time);
      }
  }

  return FALSE; /* tell GTK to use selector */


}
