%{

/*
  A scanner for slat permission mappings.
  Copyright (C) 2005 The MITRE Corporation

  Author: John D. Ramsdell

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <obstack.h>
#include <slat/xmalloc.h>
#include "slat.h"
#include "parser.h"
#include "scanner.h"

/* The space for identifiers is allocated using obstack */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack stack[1];

static void
scanner_init(void)
{
  obstack_init(stack);
}

static char *
alloc(const char *name)
{
  return (char *)obstack_copy0(stack, name, strlen(name));
}

/* Free all space allocated for identifiers. */

void
scanner_free(void)
{
  obstack_free(stack, 0);
}

/* For error reporting... */

static int lineno = 1;

static int ignore_until(int);

%}

letter          [A-Za-z]
digit           [0-9]
ID              {letter}({letter}|{digit}|_)*
SEMI            SENSITIVITY|sensitivity|CATEGORY|category|LEVEL|level
CURL            DOMINANCE|dominance

%%
[ \t\f]+	{ /* delete whitespace */ }
"\n"            { lineno++; }
"#"		{ if (ignore_until('\n') == EOF) return YY_NULL; }

COMMON|common	{ yylval.common = 1; return(COMMON); }

CLASS|class	{ yylval.common = 0; return(CLASS); }

^{SEMI}		{ if (ignore_until(';') == EOF) return YY_NULL; }
^{CURL}		{ if (ignore_until('}') == EOF) return YY_NULL; }

{ID}		{ yylval.id = alloc(yytext);  return IDENTIFIER; }

":"|"{"|"}"	{ return(yytext[0]); }

.               { return UNUSED; }

%%

/* Ignore until the given character is found. */

static int
ignore_until(int last)
{
  for (;;) {
    int c = input();
    if (c == EOF)
      return c;
    else if (c == '\n')
      lineno++;
    if (c == last)
      return c;
  }
}

static const char *filename = "-";

void
setfile(const char *path)
{
  scanner_init();
  if (path)
    filename = path;
}

int
yyerror(const char *msg)
{
  return fprintf(stderr, "%s:%d: %s at token %s\n",
		 filename, lineno, msg, yytext);
}
