/*
  Construct a labeled transition system from a binary policy and a
  file that associates information flow directions with permissions.
  Copyright (C) 2005 The MITRE Corporation

  Author: John D. Ramsdell

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <obstack.h>
#include <sepol/symtab.h>
#include <sepol/policydb.h>
#include <sepol/conditional.h>
#include <slat/xmalloc.h>
#include <slat/symbol.h>
#include <slat/pprint.h>
#include <slat/formula.h>
#include "scanner.h"
#include "slat.h"

extern int yyparse(void);

#ifdef PACKAGE
const char package[] = PACKAGE;
#else
const char *package = (const char *)0;
#endif

#ifdef VERSION
const char version[] = VERSION;
#else
const char version[] = "Version information not available";
#endif

/* Data structures that hold flow information. */

typedef struct {
  perm_datum_t *datum;
  flow_dir_t dir;		/* flows for permission */
} perm_flow_t;

typedef struct {
  class_datum_t *datum;
  symtab_t perms;		/* char * -> perm_flow_t */
} class_flow_t;

typedef struct {
  common_datum_t *datum;
  symtab_t perms;		/* char * -> perm_flow_t */
} common_flow_t;

typedef struct {
  policydb_t policy[1];
  common_flow_t *common;
  class_flow_t *class;
} flowdb_t;

/* flowdb contains the results of reading the policy and the flow
   information. */
static flowdb_t flowdb[1];

/* Arena style allocation */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack stack[1];

static void *obmalloc(size_t n)
{
  return obstack_alloc(stack, n);
}

/* Symbols in formulas.  In this usage, the space for each character
   string has been allocated as result of reading the policy, and a
   symbol contains only space for a pointer to the character
   string. */

struct symbol {
  const char *sym;
};

symbol_t
make_symbol(const char *sym)
{
  symbol_t s = (symbol_t)obmalloc(sizeof(struct symbol));
  s->sym = sym;
  return s;
}

const char *
symbol_name(symbol_t s)
{
  return s->sym;
}

/* Sets of symbols are implemented by a sorted list of symbols. */

static int			/* Set membership */
member(const char *sym, symbol_list_t list)
{
  while (list) {
    int cmp = strcmp(sym, symbol_name(symbol_list_head(list)));
    if (!cmp)
      return 1;
    if (cmp < 0)
      return 0;
    list = symbol_list_tail(list);
  }
  return 0;
}

static symbol_list_t		/* Insert symbol in sorted order */
insert(symbol_t sym, symbol_list_t list)
{
  if (!list)
    return mk_symbol_list(sym, list);
  int cmp = strcmp(symbol_name(sym), symbol_name(symbol_list_head(list)));
  if (cmp < 0)
    return mk_symbol_list(sym, list);
  else if (cmp > 0)
    return mk_symbol_list(symbol_list_head(list),
			  insert(sym, symbol_list_tail(list)));
  else				/* Should never happen because this */
    return list;		/* is only called from adjoin */
}

static symbol_list_t		/* Add a symbol to a set */
adjoin(const char *sym, symbol_list_t list)
{
  if (member(sym, list))
    return list;
  else
    return insert(make_symbol(sym), list);
}

static symbol_list_t		/* Create a singleton set */
one(const char *sym)
{
  return mk_symbol_list(make_symbol(sym), 0);
}

static int			/* Set of a hash table's keys */
collect_syms(hashtab_key_t k, hashtab_datum_t d, void *args)
{
  symbol_list_t *list = (symbol_list_t *)args;
  *list = adjoin(k, *list);
  return 0;
}

static symbol_list_t		/* Set of user names */
user_set(ebitmap_t *users)
{
  policydb_t *p = flowdb->policy;
  uint32_t i;
  symbol_list_t r = 0;
  for (i = ebitmap_startbit(users); i < ebitmap_length(users); i++)
    if (ebitmap_get_bit(users, i))
      r = adjoin(p->p_user_val_to_name[i], r);
  return r;
}

static symbol_list_t		/* Set of role names */
role_set(ebitmap_t *roles)
{
  policydb_t *p = flowdb->policy;
  uint32_t i;
  symbol_list_t r = 0;
  for (i = ebitmap_startbit(roles); i < ebitmap_length(roles); i++)
    if (ebitmap_get_bit(roles, i))
      r = adjoin(p->p_role_val_to_name[i], r);
  return r;
}

static symbol_list_t		/* Set of type names */
type_set(ebitmap_t *types)
{
  policydb_t *p = flowdb->policy;
  uint32_t i;
  symbol_list_t r = 0;
  for (i = ebitmap_startbit(types); i < ebitmap_length(types); i++)
    if (ebitmap_get_bit(types, i))
      r = adjoin(p->p_type_val_to_name[i], r);
  return r;
}

/* Labeled Transition System generation section */

static tran_t			/* Specify legal combinations of */
security_contexts(int next) 	/* users, roles, and types. */
{
  policydb_t *p = flowdb->policy;
  tran_t (*mk_users)(symbol_list_t) = next ? mk_next_users : mk_tran_users;
  tran_t (*mk_roles)(symbol_list_t) = next ? mk_next_roles : mk_tran_roles;
  tran_t (*mk_types)(symbol_list_t) = next ? mk_next_types : mk_tran_types;

  tran_t user_init = mk_roles(one(OBJECT_R));
  uint32_t nusr = p->p_users.nprim;
  uint32_t i;
  for (i = 0; i < nusr; i++) {
    user_datum_t *datum = p->user_val_to_struct[i];
    user_init =			/* Specify roles compatible */
      mk_tran_or(user_init,	/* with each user */
		 mk_tran_and(mk_users(one(p->p_user_val_to_name[i])),
			     mk_roles(role_set(&datum->roles))));
  }

  /* The object_r role should be compatible with only non-process
     types, but the binary policy does not tell us which types are
     non-process types.  Therefore, we assert that object_r is
     compatible with all types. */
  tran_t role_init = mk_roles(one(OBJECT_R));
  uint32_t nrls = p->p_roles.nprim;
  for (i = 0; i < nrls; i++)
    if (i != OBJECT_R_VAL - 1) { /* Skip object_r */
      role_datum_t *datum = p->role_val_to_struct[i];
      role_init =		/* Specify types compatible */
	mk_tran_or(role_init,	/* with each role */
		   mk_tran_and(mk_roles(one(p->p_role_val_to_name[i])),
			       mk_types(type_set(&datum->types))));
    }

  return mk_tran_and(user_init, role_init);
}

static tran_t 			/* Implement role allow semantics */
role_allow(void)
{
  policydb_t *p = flowdb->policy;
  tran_t t = mk_tran_or(mk_tran_not(mk_tran_classes(one("process"))),
			mk_tran_not(mk_tran_permissions(one("transition"))));
  t = mk_tran_or(t, mk_same_roles());

  role_allow_t *role_allow = p->role_allow;
  while (role_allow) {
    char *r = p->p_role_val_to_name[role_allow->role - 1];
    char *nr = p->p_role_val_to_name[role_allow->new_role - 1];
    t = mk_tran_or(t, mk_tran_and(mk_tran_roles(one(r)),
				  mk_next_roles(one(nr))));
    role_allow = role_allow->next;
  }
  return t;
}

static int 			/* Given a transition relation and a */
genstates(tran_t tran, tran_list_t spec) /* list of specs, construct */
{				/* the remaining parts of the lts. */
  policydb_t *p = flowdb->policy;
  symbol_list_t types = 0;
  symbol_list_t roles = 0;
  symbol_list_t users = 0;
  symbol_list_t classes = 0;
  symbol_list_t permissions = 0;

  if (hashtab_map(p->p_types.table, collect_syms, &types))
    return 1;
  if (hashtab_map(p->p_roles.table, collect_syms, &roles))
    return 1;
  if (hashtab_map(p->p_users.table, collect_syms, &users))
    return 1;
  if (hashtab_map(p->p_classes.table, collect_syms, &classes))
    return 1;

  /* Collect all permissions */
  uint32_t ncls = p->p_classes.nprim;
  uint32_t i;
  for (i = 0; i < ncls; i++) {
    class_datum_t *datum = p->class_val_to_struct[i];
    if (hashtab_map(datum->permissions.table, collect_syms, &permissions))
      return 1;
    if (datum->comdatum) {
      if (hashtab_map(datum->comdatum->permissions.table,
		      collect_syms, &permissions))
      return 1;
    }
  }

  lts_t lts = mk_lts(types, roles, users, classes, permissions,
		     security_contexts(0), tran, spec);

  print_lts(stdout, lts);
  /* One could flush the obstack here,
     but there's no point as we're done. */
  return 0;
}

/* Handle type allow assertions */

typedef struct {
  symbol_list_t list;
  uint32_t allowed;
  int write_dir;
} perm_flow_args_t;

/* Collect permissions for a given flow direction. */

static int
perm_flow_fun(hashtab_key_t k, hashtab_datum_t d, void *args)
{
  perm_flow_args_t *fpa = (perm_flow_args_t *)args;
  perm_flow_t *pf = (perm_flow_t *)d;

  if (fpa->write_dir ? !write_like(pf->dir) : !read_like(pf->dir))
    return 0;

  uint32_t value = pf->datum->value;
  if (fpa->allowed & (1 << (value - 1)))
    fpa->list = adjoin(k, fpa->list);

  return 0;
}

/*
   Augments the transition relation with permission flow results.
   Constructs a formula of the form:

   t=type1 & t'=type2 & c=class & p:{perm1, perm2,...}
*/

static tran_t
types_allowed(tran_t tran, perm_flow_args_t *args,
	      uint32_t source_type, uint32_t target_type,
	      uint32_t target_class)
{
  if (args->list) {
    policydb_t *p = flowdb->policy;
    tran_t (*mk_source)(symbol_list_t)
      = args->write_dir ? mk_tran_types : mk_next_types;
    tran_t (*mk_target)(symbol_list_t)
      = args->write_dir ? mk_next_types : mk_tran_types;
    tran_t src_type = mk_source(one(p->p_type_val_to_name[source_type]));
    tran_t tgt_type = mk_target(one(p->p_type_val_to_name[target_type]));
    tran_t cls = mk_tran_classes(one(p->p_class_val_to_name[target_class]));
    tran_t prms = mk_tran_permissions(args->list);
    tran_t t = mk_tran_and(src_type, tgt_type);
    t = mk_tran_and(t, cls);
    t = mk_tran_and(t, prms);
    tran = mk_tran_or(tran, t);
  }
  return tran;
}

/* Collect flow in a class and in its parent. */

static int
flow_in_class(class_flow_t *cf, perm_flow_args_t *args)
{
  if (hashtab_map(cf->perms.table, perm_flow_fun, args))
    return 1;
  if (cf->datum->comdatum) {
    common_flow_t *comf = flowdb->common + cf->datum->comdatum->value - 1;
    if (hashtab_map(comf->perms.table, perm_flow_fun, args))
      return 1;
  }
  return 0;
}

static int
avtab_fun(avtab_key_t *k, avtab_datum_t *d, void *p)
{
  perm_flow_args_t args[1];
  tran_t *tran = (tran_t *)p;
  tran_t t = *tran;
  uint32_t source_type = k->source_type - 1;
  uint32_t target_type = k->target_type - 1;
  uint32_t target_class = k->target_class - 1;
  uint32_t allowed = avtab_allowed(d);
  class_flow_t *cf = flowdb->class + target_class;

  args->list = 0;
  args->write_dir = 1;	/* Handle write-like case */
  args->allowed = allowed;
  if (flow_in_class(cf, args))
    return 1;
  t = types_allowed(t, args, source_type, target_type, target_class);

  args->list = 0;
  args->write_dir = 0;	/* Handle read-like case */
  args->allowed = allowed;
  if (flow_in_class(cf, args))
    return 1;
  t = types_allowed(t, args, source_type, target_type, target_class);
  *tran = t;
  return 0;
}

/* Augments a transition relation based on a list of avtab entries. */

static int
avtab(avtab_ptr_t avt, tran_t *tran)
{
  while (avt) {
    if (avtab_fun(&(avt->key), &(avt->datum), tran))
      return 1;
    avt = avt->next;
  }
  return 0;
}

/* Handle constrains */

/* The following functions translate constraint expressions into
   formulas.  The only operators supported are == and !=. */

/* For U1 op U2, R1 op R2, or T1 op T2. */
static int
attr_as_form(uint32_t attr, uint32_t op, tran_t *tran)
{
  tran_t t;
  switch (attr) {
  case CEXPR_USER:
    t = mk_same_users();
    break;
  case CEXPR_ROLE:
    t = mk_same_roles();
    break;
  case CEXPR_TYPE:
    t = mk_same_types();
    break;
  default:
    return 1;
  }
  *tran = op == CEXPR_EQ ? t : mk_tran_not(t);
  return 0;
}

/* For U1 op user_set, U2 op user_set, R1 op role_set, R2 op role_set,
   T1 op type_set, op T2 type_set. */
static int
names_as_form(int write_dir, uint32_t attr, uint32_t op,
	      ebitmap_t *names, tran_t *tran)
{
  tran_t t;
  symbol_list_t list;
  int target = attr & CEXPR_TARGET ? write_dir: !write_dir;
  switch(attr & ~CEXPR_TARGET) {
  case CEXPR_USER:
    list = user_set(names);
    if (target)
      t = mk_next_users(list);
    else
      t = mk_tran_users(list);
    break;
  case CEXPR_ROLE:
    list = role_set(names);
    if (target)
      t = mk_next_roles(list);
    else
      t = mk_tran_roles(list);
    break;
  case CEXPR_TYPE:
    list = type_set(names);
    if (target)
      t = mk_next_types(list);
    else
      t = mk_tran_types(list);
    break;
  default:
    return 1;
  }
  *tran = op == CEXPR_EQ ? t : mk_tran_not(t);
  return 0;
}

/* Checks the operator, then dispatches to appropriate formula
   generator. */
static int
atom_as_form(int write_dir, constraint_expr_t *expr, tran_t *tran)
{
  if (!expr)
    return 1;

  uint32_t attr = expr->attr;
  uint32_t op = expr->op;
  if (op != CEXPR_EQ && op != CEXPR_NEQ)
    return 1;

  switch (expr->expr_type) {
  case CEXPR_ATTR:
    return attr_as_form(attr, op, tran);
  case CEXPR_NAMES:
    return names_as_form(write_dir, attr, op, &(expr->names), tran);
  default:
    return 1;
  }
}

/* Constraint expression interpreter.  Constraint expressions are
   stored in reverse Polish notation.  The C control stack is used as
   an evaluation stack. */
static int
rec_expr_as_form(int write_dir, constraint_expr_t **expr, tran_t *out)
{
  tran_t arg1, arg2;
  if (atom_as_form(write_dir, *expr, &arg1))
    return 1;
  *expr = (*expr)->next;

  for (;;) {
    if (!*expr) {		/* End of constraint expression */
      *out = arg1;
      return 0;
    }
    switch ((*expr)->expr_type) {
    case CEXPR_NOT:		/* Handle constraint negation */
      arg1 = mk_tran_not(arg1);
      break;
    case CEXPR_AND:
    case CEXPR_OR:
      *out = arg1;		/* End of this expression */
      return 0;
      break;
    default:			/* Binary expression expected */
      if (rec_expr_as_form(write_dir, expr, &arg2))
	return 1;
      if (!*expr) 		/* Malformed constraint expression */
	return 1;
      switch ((*expr)->expr_type) {
      case CEXPR_AND:		/* Handle constraint conjunction */
	arg1 = mk_tran_and(arg1, arg2);
	break;
      case CEXPR_OR:		/* Handle constraint disjunction */
	arg1 = mk_tran_or(arg1, arg2);
	break;
      default:
	return 1;		/* Malformed constraint expression */
      }
    }
    *expr = (*expr)->next;
  }
}

/* Driver for the above function. */
static int
expr_as_form(int write_dir, constraint_expr_t *expr, tran_t *out)
{
  if (rec_expr_as_form(write_dir, &expr, out))
    return 1;
  else
    return expr != 0;  /* Check for malformed constraint expression */
}

/* Generate formulas for one constraint node.  Uses the same
   techniques as is used in avtab_fun. */
static int
constrain(class_datum_t *datum, constraint_node_t *node, tran_t *tran)
{
  perm_flow_args_t args[1];
  policydb_t *p = flowdb->policy;
  uint32_t value = datum->value;
  class_flow_t *cf = flowdb->class + value - 1;
  char *name = p->p_class_val_to_name[value - 1];
  sepol_access_vector_t allowed = node->permissions;
  constraint_expr_t *expr = node->expr;
  tran_t t;

  args->list = 0;
  args->write_dir = 1;	/* Handle write-like case */
  args->allowed = allowed;
  if (flow_in_class(cf, args))
    return 1;
  if (args->list && !expr_as_form(1, expr, &t)) {
    t = mk_tran_or(mk_tran_not(mk_tran_permissions(args->list)), t);
    t = mk_tran_or(mk_tran_not(mk_tran_classes(one(name))), t);
    *tran = mk_tran_and(*tran, t);
  }

  args->list = 0;
  args->write_dir = 0;	/* Handle read-like case */
  args->allowed = allowed;
  if (flow_in_class(cf, args))
    return 1;
  if (args->list && !expr_as_form(0, expr, &t)) {
    t = mk_tran_or(mk_tran_not(mk_tran_permissions(args->list)), t);
    t = mk_tran_or(mk_tran_not(mk_tran_classes(one(name))), t);
    *tran = mk_tran_and(*tran, t);
  }

  return 0;
}

static int			/* Applies the above function */
constraints(tran_t *tran) {	/* to every constraint node */
  policydb_t *p = flowdb->policy;
  uint32_t ncls = p->p_classes.nprim;
  uint32_t i;
  for (i = 0; i < ncls; i++) {
    class_datum_t *datum = p->class_val_to_struct[i];
    constraint_node_t *constraints = datum->constraints;
    while (constraints) {
      if (constrain(datum, constraints, tran))
	return 1;
      constraints = constraints->next;
    }
  }
  return 0;
}

static int
genlts(void)
{
  policydb_t *p = flowdb->policy;
  tran_t tran = mk_tran_false();

  /* Handle unconditional avtab entries. */
  if (avtab_map(&(p->te_avtab), avtab_fun, &tran))
    return 1;

  /* Handle conditional avtab entries. */
  cond_list_t *cond_list = p->cond_list;
  while (cond_list) {
    cond_av_list_t *av_list
      = cond_list->cur_state ? cond_list->true_list : cond_list->false_list;
    while (av_list) {
      if (avtab(av_list->node, &tran))
	return 1;
      av_list = av_list->next;
    }
    cond_list = cond_list->next;
  }

  /* Build remaining parts of the transition relation. */
  tran_t all = mk_tran_and(security_contexts(0), security_contexts(1));
  all = mk_tran_and(all, role_allow());
  all = mk_tran_and(all, tran);
  if (constraints(&all))
    return 1;

  return genstates(all, 0);
}

/* Permission-to-flow mapping printer */

/* The permissions are printed in alphabetical order, and the policy
   determines the order in which commons and classes are printed, so
   there is one canonical form for the output, and it is easily
   compared. */

static int			/* Collect defined permissions */
perm_flow_defined_fun(hashtab_key_t k, hashtab_datum_t d, void *args)
{
  perm_flow_t *pf = (perm_flow_t *)d;
  symbol_list_t *list = (symbol_list_t *)args;
  if (pf->dir >= FLOW_NONE && pf->dir <= FLOW_BOTH)
    *list = adjoin(k, *list);
  return 0;
}

static int			/* Print flow of list of permissions */
print_perm_flow(hashtab_t table, symbol_list_t list)
{
  while (list) {
    const char *name = symbol_name(symbol_list_head(list));
    perm_flow_t *pf = (perm_flow_t *)hashtab_search(table, (char *)name);
    if (!pf) {
      fprintf(stderr, "Cannot find permission %s\n", name);
      return 1;
    }

    char *dir;			/* String describing the flow */
    switch (pf->dir) {
    case FLOW_NONE:
      dir = "none";
      break;
    case FLOW_READ:
      dir = "read";
      break;
    case FLOW_WRITE:
      dir = "write";
      break;
    case FLOW_BOTH:
      dir = "{ read write }";
      break;
    default:
      fprintf(stderr, "Undefined flow");
      return 1;
    }
    printf("    %-28s : %s\n", name, dir);
    list = symbol_list_tail(list);
  }
  return 0;
}

static int			/* Print a common if it defines flow */
gencommap(common_flow_t *comf)
{
  hashtab_t table = comf->perms.table;
  symbol_list_t list = 0;
  if (hashtab_map(table, perm_flow_defined_fun, &list))
    return 1;
  if (list) {
    policydb_t *p = flowdb->policy;
    char *name = p->p_common_val_to_name[comf->datum->value - 1];
    printf("\ncommon %s\n{\n", name);
    print_perm_flow(table, list);
    printf("}\n");
  }
  return 0;
}

/* Print a class, even if none of its permissions define a flow
   direction.  For a class that inherits from a common, a comment is
   generated naming the common. */

static int
genclsmap(class_flow_t *cf)
{
  hashtab_t table = cf->perms.table;
  symbol_list_t list = 0;
  policydb_t *p = flowdb->policy;
  char *name = p->p_class_val_to_name[cf->datum->value - 1];
  printf("\nclass %s\n", name);

  common_datum_t *datum = cf->datum->comdatum;
  if (datum) {
    char *comname = p->p_common_val_to_name[datum->value - 1];
    printf("# inherits from %s\n", comname);
  }

  if (hashtab_map(table, perm_flow_defined_fun, &list))
    return 1;
  if (list) {
    printf("{\n");
    print_perm_flow(table, list);
    printf("}\n");
  }
  return 0;
}

static int			/* Prints the permission mappings of the */
genmap(void)			/* commons followed by the classes */
{
  policydb_t *p = flowdb->policy;
  uint32_t ncom = p->p_commons.nprim;
  uint32_t i;
  for (i = 0; i < ncom; i++)
    if (gencommap(flowdb->common + i))
      return 1;

  uint32_t ncls = p->p_classes.nprim;
  for (i = 0; i < ncls; i++) {
    if (genclsmap(flowdb->class + i))
      return 1;
  }

  return 0;
}

/* Data input section */

static int			/* Read binary policy */
getpol(const char *program, const char *policy)
{
  int fd = open(policy, O_RDONLY);
  if (fd < 0) {
    fprintf(stderr, "Can't open '%s':  %s\n",
	    policy, strerror(errno));
    return 1;
  }

  struct stat sb[1];
  if (fstat(fd, sb) < 0) {
    fprintf(stderr, "Can't stat '%s':  %s\n",
	    policy, strerror(errno));
    return 1;
  }

  void *map = mmap(NULL, sb->st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "Can't map '%s':  %s\n",
	    policy, strerror(errno));
    return 1;
  }

  struct policy_file pf = {
    .type = PF_USE_MEMORY,
    .data = map,
    .len = sb->st_size};

  if (policydb_read(flowdb->policy, &pf, 1)) {
    fprintf(stderr, "%s:  error(s) encountered while parsing configuration\n",
	    program);
    return 1;
  }

  return 0;
}

/* Read flow information. */

/* Add initial permissions with an undefinied flow. */

static int
initperms(hashtab_key_t k, hashtab_datum_t d, void *args)
{
  perm_datum_t *datum = (perm_datum_t *)d;
  hashtab_t table = (hashtab_t)args;
  perm_flow_t *flow = (perm_flow_t *)obmalloc(sizeof(perm_flow_t));
  flow->datum = datum;
  flow->dir = FLOW_UNDEF;
  return HASHTAB_SUCCESS != hashtab_insert(table, k, flow);
}

/* Add initial commons. */

static int
initcom(hashtab_key_t k, hashtab_datum_t d, void *args)
{
  common_datum_t *datum = (common_datum_t *)d;
  common_flow_t *com = (common_flow_t *)args;
  com += datum->value - 1;
  com->datum = datum;
  if (symtab_init(&com->perms, datum->permissions.table->size))
    return 1;
  com->perms.nprim = datum->permissions.nprim;
  return hashtab_map(datum->permissions.table, initperms, com->perms.table);
}

/* Allocate space for all flow information, and initialize flow
   permissions to have an undefined flow. */

static int
initflow(void)
{
  policydb_t *p = flowdb->policy;
  uint32_t i;
  uint32_t ncls = p->p_classes.nprim;
  class_flow_t *cls = (class_flow_t *)obmalloc(ncls * sizeof(class_flow_t));
  flowdb->class = cls;

  for (i = 0; i < ncls; i++) {
    class_datum_t *datum = p->class_val_to_struct[i];
    cls[i].datum = datum;
    if (symtab_init(&cls[i].perms, datum->permissions.table->size))
      return 1;
    cls[i].perms.nprim = datum->permissions.nprim;
    if (hashtab_map(datum->permissions.table, initperms, cls[i].perms.table))
      return 1;
  }

  uint32_t ncom = p->p_commons.nprim;
  if (ncom <= 0)
    ncom = 1;
  common_flow_t *com = (common_flow_t *)obmalloc(ncom * sizeof(common_flow_t));
  flowdb->common = com;

  return hashtab_map(p->p_commons.table, initcom, com);
}

/* Add a flow and check to make sure it is consistent with the
   previous flow value. */

static int
addflow(perm_flow_t *flow, int common, char *name, char *perm, flow_dir_t dir)
{
  if (flow->dir == dir)
    return 0;
  if (flow->dir == FLOW_UNDEF) {
    flow->dir = dir;
    return 0;
  }
  fprintf(stderr, "Incompatible flows found for permission %s in %s %s\n",
	  perm, common ? "common" : "class", name);
  return 1;
}

/* Add flow information from the permission flow mapping input file.
   This function mostly looks up names and lets addflow do the real
   work. */
int
action(int common, char *name, char *perm, flow_dir_t dir)
{
  policydb_t *p = flowdb->policy;
  if (common) {
    hashtab_t table = p->p_commons.table;
    common_datum_t *datum = (common_datum_t *)hashtab_search(table, name);
    if (!datum) {
      fprintf(stderr, "Cannot find common %s\n", name);
      return 1;
    }
    table = flowdb->common[datum->value - 1].perms.table;
    perm_flow_t *flow = (perm_flow_t *)hashtab_search(table, perm);
    if (flow)
      return addflow(flow, common, name, perm, dir);
    fprintf(stderr, "Cannot find permission %s in common %s\n", perm, name);
    return 1;
  }
  else {
    hashtab_t table = p->p_classes.table;
    class_datum_t *datum = (class_datum_t *)hashtab_search(table, name);
    if (!datum) {
      fprintf(stderr, "Cannot find class %s\n", name);
      return 1;
    }
    table = flowdb->class[datum->value - 1].perms.table;
    perm_flow_t *flow = (perm_flow_t *)hashtab_search(table, perm);
    if (flow)
      return addflow(flow, common, name, perm, dir);
    if (datum->comdatum) {
      table = flowdb->common[datum->comdatum->value - 1].perms.table;
      flow = (perm_flow_t *)hashtab_search(table, perm);
      char *parent = p->p_common_val_to_name[datum->comdatum->value - 1];
      if (flow)
	return addflow(flow, 1, parent, perm, dir);
    }
    fprintf(stderr, "Cannot find permission %s in class %s\n", perm, name);
    return 1;
  }
}

static int
getflow(const char *flow)	/* Read flow direction information */
{
  if (initflow())
    return 1;
  if (flow && !freopen(flow, "r", stdin)) {
    perror(flow);
    return 1;
  }
  setfile(flow);
  int rc = yyparse();
  scanner_free();
  return rc;
}

static int
go(const char *program, int mapping, const char *output,
   const char *policy, const char *flow)
{
  obstack_init(stack);
  pprint_init();

  int rc = getpol(program, policy);
  if (rc)
    return rc;

  rc = getflow(flow);
  if (rc)
    return rc;

  if (output && !freopen(output, "w", stdout)) {
    perror(output);
    return 1;
  }

  formula_stack(stack);
  if (mapping)
    return genmap();
  else
    return genlts();
}

static void
print_version(const char *program)
{
  if (package)
    program = package;
  fprintf(stderr, "Package: %s %s\n", program, version);
}

static void
usage(const char *prog)
{
  fprintf(stderr,
	  "Usage: %s [options] policy [mapping]\n"
	  "Options:\n"
	  "  -o file -- output to file (default is standard output)\n"
	  "  -m      -- print permission-to-flow mapping\n"
	  "  -v      -- print version information\n"
	  "  -h      -- print this message\n"
	  "Use - as a file name to specify standard input\n",
	  prog);
  print_version(prog);
}

int
main(int argc, char **argv)	/* Process command line options */
{
  extern char *optarg;
  extern int optind;

  int mapping = 0;
  char *output = 0;

  for (;;) {
    int c = getopt(argc, argv, "o:mvh");
    if (c == -1)
      break;
    switch (c) {
    case 'o':
      output = optarg;
      break;
    case 'm':
      mapping = 1;
      break;
    case 'v':
      print_version(argv[0]);
      return 0;
    case 'h':
      usage(argv[0]);
      return 0;
    default:
      usage(argv[0]);
      return 1;
    }
  }

  switch (argc - optind) {
  case 2:			/* Policy file name required */
  case 1:			/* Flow file name optional */
    break;
  default:
    fprintf(stderr, "Bad arg count\n");
    usage(argv[0]);
    return 1;
  }

  return go(argv[0], mapping, output, argv[optind], argv[optind + 1]);
}
