/*
 * ewfacquire
 * Reads data from a file and writes it in EWF format
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/utsname.h>
#include <errno.h>
#include <inttypes.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#ifdef LINUX
#include <linux/fs.h>
#endif

#ifdef CYGWIN
#include <cygwin/fs.h>
#endif

#ifdef BSD
#include <sys/ioctl.h>
#include <sys/disk.h>
#include <sys/disklabel.h>
#endif

#ifdef DARWIN
#include <sys/ioctl.h>
#include <sys/disk.h>
#endif

#include "libewf.h"

#include "libewf_common.h"
#include "libewf_notify.h"

#include "ewfcommon.h"

/* Prints the executable usage information
 */
void usage( void )
{
	fprintf( stderr, "Usage: ewfacquire [ -hqsvV ] source\n" );

	fprintf( stderr, "\tsource: the source file or device\n" );

	fprintf( stderr, "\t-h:     shows this help\n" );
	fprintf( stderr, "\t-s:     swap byte pairs of the media data (from AB to BA)\n" );
	fprintf( stderr, "\t        (use this for big to little endian conversion and vice versa)\n" );
	fprintf( stderr, "\t-q:     quiet shows no status information\n" );
	fprintf( stderr, "\t-v:     verbose output to stderr\n" );
	fprintf( stderr, "\t-V:     print version\n" );
}

/* Prints the executable version information
 */
void version( void )
{
	fprintf( stderr, "Copyright 2006 Joachim Metz, Hoffmann Investigations <%s> and contributors.\n", PACKAGE_BUGREPORT );
	fprintf( stderr, "This is free software; see the source for copying conditions. There is NO\n" );
	fprintf( stderr, "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" );
}

/* Prints an overview of the user provided input
 * and asks the user for confirmation
 * Return 1 if confirmed by user, 0 otherwise
 */
int confirm_input( char *filename, char *case_number, char *description, char *evidence_number, char *examiner_name, char *notes, int8_t compression_level, uint8_t compress_empty_block, uint8_t ewf_format, uint64_t acquiry_offset, uint64_t acquiry_size, uint64_t ewf_file_size, uint64_t sectors_per_block, uint64_t error_granularity, uint8_t read_error_retry, uint8_t wipe_block_on_read_error )
{
	char *user_input    = NULL;
	char *yes_no[ 2 ]   = { "yes", "no" };
	int input_confirmed = -1;

	fprintf( stdout, "The following information was provided:\n" );

	fprintf( stdout, "Image path and filename:\t%s.", filename );

	if( ewf_format == LIBEWF_FORMAT_SMART )
	{
		fprintf( stdout, "s01\n" );
	}
	else
	{
		fprintf( stdout, "E01\n" );
	}
	fprintf( stdout, "Case number:\t\t\t" );

	if( case_number != NULL )
	{
		fprintf( stdout, "%s", case_number );
	}
	fprintf( stdout, "\n" );

	fprintf( stdout, "Description:\t\t\t" );

	if( description != NULL )
	{
		fprintf( stdout, "%s", description );
	}
	fprintf( stdout, "\n" );

	fprintf( stdout, "Evidence number:\t\t" );

	if( evidence_number != NULL )
	{
		fprintf( stdout, "%s", evidence_number );
	}
	fprintf( stdout, "\n" );

	fprintf( stdout, "Examiner name:\t\t\t" );

	if( examiner_name != NULL )
	{
		fprintf( stdout, "%s", examiner_name );
	}
	fprintf( stdout, "\n" );

	fprintf( stdout, "Notes:\t\t\t\t" );

	if( notes != NULL )
	{
		fprintf( stdout, "%s", notes );
	}
	fprintf( stdout, "\n" );

	fprintf( stdout, "Compression used:\t\t" );

	if( compression_level == EWF_COMPRESSION_FAST )
	{
		fprintf( stdout, "fast\n" );
	}
	else if( compression_level == EWF_COMPRESSION_BEST )
	{
		fprintf( stdout, "best\n" );
	}
	else if( compression_level == EWF_COMPRESSION_NONE )
	{
		fprintf( stdout, "none\n" );

		fprintf( stdout, "Compress empty blocks:\t\t" );

		if( compress_empty_block == 0 )
		{
			fprintf( stdout, "no\n" );
		}
		else
		{
			fprintf( stdout, "yes\n" );
		}
	}
	fprintf( stdout, "EWF file format:\t\t" );

	if( ewf_format == LIBEWF_FORMAT_SMART )
	{
		fprintf( stdout, "SMART\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_FTK )
	{
		fprintf( stdout, "FTK Imager\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_ENCASE1 )
	{
		fprintf( stdout, "Encase 1\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_ENCASE2 )
	{
		fprintf( stdout, "Encase 2\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_ENCASE3 )
	{
		fprintf( stdout, "Encase 3\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_ENCASE4 )
	{
		fprintf( stdout, "Encase 4\n" );
	}
	else if( ewf_format == LIBEWF_FORMAT_ENCASE5 )
	{
		fprintf( stdout, "Encase 5\n" );
	}
	else
	{
		fprintf( stdout, "\n" );
	}
	fprintf( stdout, "Acquiry start offet:\t\t%" PRIu64 "\n", acquiry_offset );
	fprintf( stdout, "Amount of bytes to acquire:\t%" PRIu64 "\n", acquiry_size );
	fprintf( stdout, "Evidence file size:\t\t%" PRIu64 " kbytes\n", ( ewf_file_size / 1024 ) );
	fprintf( stdout, "Block size:\t\t\t%" PRIu64 " sectors\n", sectors_per_block );
	fprintf( stdout, "Error granularity:\t\t%" PRIu64 " sectors\n", error_granularity );
	fprintf( stdout, "Retries on read error:\t\t%" PRIu8 "\n", read_error_retry );

	fprintf( stdout, "Wipe sectors on read error:\t" );

	if( wipe_block_on_read_error == 0 )
	{
		fprintf( stdout, "no\n" );
	}
	else
	{
		fprintf( stdout, "yes\n" );
	}
	fprintf( stdout, "\n" );

	/* Ask for confirmation
	 */
	while( input_confirmed == -1 )
	{
		user_input = get_user_input_fixed_value( stdout, "Continue acquiry with these values", yes_no, 2, 0 );
	
		if( libewf_common_strncmp( user_input, "yes", 3 ) == 0 )
		{
			input_confirmed = 1;
		}
		else if( libewf_common_strncmp( user_input, "no", 2 ) == 0 )
		{
			input_confirmed = 0;
		}
		else
		{
			fprintf( stdout, "Selected option not supported, please try again or terminate using Ctrl^C.\n" );
		}
		libewf_common_free( user_input );
	}
	fprintf( stdout, "\n" );

	return( input_confirmed );
}

/* Prints information about the acquiry process
 */
void print_acquiry_information( int64_t byte_count, time_t timestamp_start, time_t timestamp_end )
{
	time_t timestamp_acquiry;

	struct tm *acquiry_time       = NULL;
	char *bytes_string            = NULL;
	char *bytes_per_second_string = NULL;
	uint64_t seconds_acquiry      = 0;
	uint64_t bytes_per_second     = 0;

	timestamp_acquiry = timestamp_end - timestamp_start;
	acquiry_time      = localtime( &timestamp_acquiry );
	seconds_acquiry   = (uint64_t) difftime( timestamp_end, timestamp_start );
	bytes_string      = determine_human_readable_size_string( byte_count );

	if( bytes_string != NULL )
	{
		fprintf( stderr, "Written: %s (%" PRIi64 " bytes)", bytes_string, byte_count );
	}
	else
	{
		fprintf( stderr, "Written: %" PRIi64 " bytes", byte_count );
	}
	if( acquiry_time != NULL )
	{
		fprintf( stderr, " in" );

		if( acquiry_time->tm_yday > 0 )
		{
			fprintf( stderr, " %i day(s), %i hour(s), %i minute(s) and", acquiry_time->tm_yday, ( acquiry_time->tm_hour - 1 ), acquiry_time->tm_min );
		}
		else if( acquiry_time->tm_hour > 1 )
		{
			fprintf( stderr, " %i hour(s), %i minute(s) and", ( acquiry_time->tm_hour - 1 ), acquiry_time->tm_min );
		}
		else if( acquiry_time->tm_min > 0 )
		{
			fprintf( stderr, " %i minute(s) and", acquiry_time->tm_min );
		}
		fprintf( stderr, " %i second(s)", acquiry_time->tm_sec );
	}
	if( seconds_acquiry > 0 )
	{
		bytes_per_second        = byte_count / seconds_acquiry;
		bytes_per_second_string = determine_human_readable_size_string( bytes_per_second );

		fprintf( stderr, " with" );

		if( bytes_per_second_string != NULL )
		{
			fprintf( stderr, " %s/s", bytes_per_second_string );

			libewf_common_free( bytes_per_second_string );
		}
		fprintf( stderr, " (%" PRIu64 " bytes/second)", bytes_per_second );
	}
	fprintf( stderr, "\n" );
}

/* Print the status of the acquire process
 */
int8_t last_percentage = -1;

void print_percentage_callback( uint64_t bytes_read, uint64_t bytes_total )
{
	char *bytes_read_string  = NULL;
	char *bytes_total_string = NULL;
	int8_t new_percentage    = ( bytes_total > 0 ) ? ( ( bytes_read * 100 ) / bytes_total ) : 1;

	if( new_percentage > last_percentage )
	{
		last_percentage    = new_percentage;
		bytes_read_string  = determine_human_readable_size_string( bytes_read );
		bytes_total_string = determine_human_readable_size_string( bytes_total );

		if( ( bytes_read_string != NULL ) && ( bytes_total_string != NULL ) )
		{
			fprintf( stderr, "Status: bytes read: %s (%" PRIu64 " bytes) of total: %s (%" PRIu64 " bytes) (%" PRIi8 "%%).\n", bytes_read_string, bytes_read, bytes_total_string, bytes_total, last_percentage );
		}
		else
		{
			fprintf( stderr, "Status: bytes read: %" PRIu64 " of total: %" PRIu64 " (%" PRIi8 "%%).\n", bytes_read, bytes_total, last_percentage );
		}

		if( bytes_read_string != NULL )
		{
			libewf_common_free( bytes_read_string );
		}
		if( bytes_total_string != NULL )
		{
			libewf_common_free( bytes_total_string );
		}
	}
}

/* The main program
 */
int main( int argc, const char **argv )
{
#ifndef DIOCGMEDIASIZE
#ifdef DIOCGDINFO
	struct disklabel disk_label;
#endif
#endif
	struct stat input_file_stat;
	struct utsname utsname_buffer;
	time_t timestamp_start;
	time_t timestamp_end;
	char *filenames[ 1 ];

	LIBEWF_HANDLE *handle                    = NULL;
	LIBEWF_HANDLE *data_set                  = NULL;
	LIBEWF_HEADER_VALUES *header_values      = NULL;
	LIBEWF_STRING *calculated_md5hash_string = NULL;
	char *user_input                         = NULL;
	char *filename                           = NULL;
	char *case_number                        = NULL;
	char *description                        = NULL;
	char *evidence_number                    = NULL;
	char *examiner_name                      = NULL;
	char *notes                              = NULL;
	void *callback                           = &print_percentage_callback;

#ifdef DKIOCGETBLOCKCOUNT
	uint64_t block_count                     = 0;
	uint32_t block_size                      = 0;
#endif
	int64_t count                            = 0;
	uint64_t size_input_file                 = 0;
	uint64_t acquiry_offset                  = 0;
	uint64_t acquiry_size                    = 0;
	uint64_t ewf_file_size                   = 0;
	uint64_t sectors_per_block               = 0;
	uint64_t error_granularity               = 0;
	int8_t compression_level                 = EWF_COMPRESSION_NONE;
	uint8_t compress_empty_block             = 0;
	uint8_t wipe_block_on_read_error         = 0;
	uint8_t ewf_format                       = LIBEWF_FORMAT_UNKNOWN;
	uint8_t read_error_retry                 = 3;
	uint8_t swap_byte_pairs                  = 0;
	int file_descriptor                      = 0;
	int option                               = 0;
	char *compression_types[ 3 ]             = { "none", "fast", "best" };
	char *format_types[ 7 ]                  = { "smart", "ftk", "encase1", "encase2", "encase3", "encase4", "encase5" };
	char *sector_per_block_sizes[ 7 ]        = { "64", "128", "256", "512", "1024", "2048", "4096" };
	char *yes_no[ 2 ]                        = { "yes", "no" };

	fprintf( stderr, "ewfacquire version: %s\n\n", VERSION );

	while( ( option = getopt( argc, (char **) argv, "hsqvV" ) ) > 0 )
	{
		switch( option )
		{
			case '?':
			default:
				fprintf( stderr, "Invalid argument: %s\n", argv[ optind ] );

				usage();

				return( EXIT_FAILURE );

			case 'h':
				usage();

				return( EXIT_SUCCESS );

			case 's':
				swap_byte_pairs = 1;

				break;

			case 'q':
				callback = NULL;

				break;

			case 'v':
				libewf_verbose = 1;

				break;

			case 'V':
				version();

				return( EXIT_SUCCESS );
		}
	}
	if( optind == argc )
	{
		fprintf( stderr, "Missing source file or device.\n" );

		usage();

		return( EXIT_FAILURE );
	}
	/* Check if to read from stdin
	 */
	if( libewf_common_strncmp( argv[ optind ], "-", 1 ) == 0 )
	{
		fprintf( stderr, "Reading from stdin not supported.\n" );

		return( EXIT_FAILURE );
	}
	/* Check the input file or device size
	 */
	file_descriptor = libewf_common_open( argv[ optind ], O_RDONLY, 0644 );

	if( file_descriptor == -1 )
	{
		fprintf( stderr, "Error opening file or device: %s", argv[ optind ] );
#if HAVE_STRERROR
		fprintf( stderr, " with failure: %s", strerror( errno ) );
#endif
		fprintf( stderr, ".\n" );

		return( EXIT_FAILURE );
	}
	size_input_file = 0;

	if( fstat( file_descriptor, &input_file_stat ) == 0 )
	{
		size_input_file = input_file_stat.st_size;
	}
	if( size_input_file <= 0 )
	{
#ifdef BLKGETSIZE64
		ioctl( file_descriptor, BLKGETSIZE64, &size_input_file );
#else
#ifdef DIOCGMEDIASIZE
		ioctl( file_descriptor, DIOCGMEDIASIZE, &size_input_file );
#else
#ifdef DIOCGDINFO
		if( ioctl( file_descriptor, DIOCGDINFO, &disk_label ) != -1 )
		{
			size_input_file = disk_label.d_secperunit * disk_label.d_secsize;
		}
#else
#ifdef DKIOCGETBLOCKCOUNT
		ioctl( file_descriptor, DKIOCGETBLOCKSIZE, &block_size );
		ioctl( file_descriptor, DKIOCGETBLOCKCOUNT, &block_count );

		size_input_file = block_count * block_size;

#ifdef HAVE_DEBUG_OUTPUT
		fprintf( stderr, "block size: %" PRIu32 " block count: %" PRIu64 " ", block_size, block_count );
#endif

#else
		size_input_file = 0;
#endif
#endif
#endif
#endif

#ifdef HAVE_DEBUG_OUTPUT
		fprintf( stderr, "device size: %" PRIu64 "\n", size_input_file );
#endif
	}
	if( size_input_file <= 0 )
	{
		fprintf( stderr, "Unable to determine file or device size.\n" );

		return( EXIT_FAILURE );
	}
	header_values = libewf_header_values_alloc();

	if( header_values == NULL )
	{
		fprintf( stderr, "Unable to create header values.\n" );

		return( EXIT_FAILURE );
	}
	/* Determine acquiry system type
	 */
	if( uname( &utsname_buffer ) == -1 )
	{
		header_values->acquiry_operating_system = libewf_header_values_set_value( header_values->acquiry_operating_system, "Undetermined" );
	}
	else
	{
		header_values->acquiry_operating_system = libewf_header_values_set_value( header_values->acquiry_operating_system, utsname_buffer.sysname );
	}
	header_values->acquiry_software_version = libewf_header_values_set_value( header_values->acquiry_software_version, VERSION );

	/* Both time values will be generated automatically when set to NULL
	 */
	header_values->system_date      = NULL;
	header_values->acquiry_date     = NULL;
	header_values->password         = NULL;
	header_values->compression_type = NULL;

	do
	{
		if( case_number != NULL )
		{
			libewf_common_free( case_number );
		}
		if( description != NULL )
		{
			libewf_common_free( description );
		}
		if( evidence_number != NULL )
		{
			libewf_common_free( evidence_number );
		}
		if( examiner_name != NULL )
		{
			libewf_common_free( examiner_name );
		}
		if( notes != NULL )
		{
			libewf_common_free( notes );
		}

		/* Request the necessary case data
		 */
		fprintf( stdout, "Information about acquiry required, please provide the necessary input\n" );

		/* Output filename
		 */
		while( filename == NULL )
		{
			filename = get_user_input_variable( stdout, "Image path and filename without extension" );

			if( filename == NULL )
			{
				fprintf( stdout, "Filename is required, please try again or terminate using Ctrl^C.\n" );
			}
		}
		/* Case number
		 */
		case_number = get_user_input_variable( stdout, "Case number" );

		/* Description
		 */
		description = get_user_input_variable( stdout, "Description" );

		/* Evidence number
		 */
		evidence_number = get_user_input_variable( stdout, "Evidence number" );

		/* Examiner name
		 */
		examiner_name = get_user_input_variable( stdout, "Examiner name" );

		/* Notes
		 */
		notes = get_user_input_variable( stdout, "Notes" );

		/* Compression
		 */
		user_input = get_user_input_fixed_value( stdout, "Use compression", compression_types, 3, 0 );

		if( libewf_common_strncmp( user_input, "none", 4 ) == 0 )
		{
			compression_level = EWF_COMPRESSION_NONE;
		}
		else if( libewf_common_strncmp( user_input, "fast", 4 ) == 0 )
		{
			compression_level = EWF_COMPRESSION_FAST;
		}
		else if( libewf_common_strncmp( user_input, "best", 4 ) == 0 )
		{
			compression_level = EWF_COMPRESSION_BEST;
		}
		else
		{
			fprintf( stderr, "ewfacquire: unsuported compression type.\n" );

			return( EXIT_FAILURE );
		}
		libewf_common_free( user_input );

		/* Empty block compression
		 */
		if( compression_level == EWF_COMPRESSION_NONE )
		{
			user_input = get_user_input_fixed_value( stdout, "Compress empty blocks", yes_no, 2, 1 );

			if( libewf_common_strncmp( user_input, "yes", 3 ) == 0 )
			{
				compress_empty_block = 1;
			}
			else if( libewf_common_strncmp( user_input, "no", 2 ) == 0 )
			{
				compress_empty_block = 0;
			}
			else
			{
				fprintf( stderr, "ewfacquire: unsuported answer.\n" );

				return( EXIT_FAILURE );
			}
			libewf_common_free( user_input );
		}

		/* File format
		 */
		user_input = get_user_input_fixed_value( stdout, "Use EWF file format", format_types, 7, 6 );

		if( libewf_common_strncmp( user_input, "smart", 5 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_SMART;
		}
		else if( libewf_common_strncmp( user_input, "ftk", 3 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_FTK;
		}
		else if( libewf_common_strncmp( user_input, "encase1", 7 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_ENCASE1;
		}
		else if( libewf_common_strncmp( user_input, "encase2", 7 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_ENCASE2;
		}
		else if( libewf_common_strncmp( user_input, "encase3", 7 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_ENCASE3;
		}
		else if( libewf_common_strncmp( user_input, "encase4", 7 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_ENCASE4;
		}
		else if( libewf_common_strncmp( user_input, "encase5", 7 ) == 0 )
		{
			ewf_format = LIBEWF_FORMAT_ENCASE5;
		}
		else
		{
			fprintf( stderr, "ewfacquire: unsuported EWF file format type.\n" );

			return( EXIT_FAILURE );
		}
		libewf_common_free( user_input );

		/* Size and offset of data to acquire
		 */
		acquiry_offset = get_user_input_size_variable( stderr, "Start to acquire at offset", 0, size_input_file, 0 );
		acquiry_size   = get_user_input_size_variable( stderr, "Amount of bytes to acquire", 0, size_input_file, size_input_file );

		/* File size
		 */
		ewf_file_size  = get_user_input_size_variable( stdout, "Evidence file size in kbytes (2^10)", 1440, ( 2 * 1024 * 1024 ), ( 650 * 1024 ) );
		ewf_file_size *= 1024;

		/* Chunk size (sectors per block)
		 */
		user_input = get_user_input_fixed_value( stdout, "The amount of sectors to read at once", sector_per_block_sizes, 7, 0 );

		sectors_per_block = atoll( user_input );

		libewf_common_free( user_input );

		/* Error granularity
		 */
		error_granularity = get_user_input_size_variable( stdout, "The amount of sectors to be used as error granularity", 1, sectors_per_block, 64 );

		/* The amount of read error retry
		 */
		read_error_retry = get_user_input_size_variable( stdout, "The amount of retries when a reading error occurs", 0, 255, 3 );

		/* Wipe the sector on error
		 */
		user_input = get_user_input_fixed_value( stdout, "Wipe sectors on read error (mimic EnCase like behavior)", yes_no, 2, 0 );

		if( libewf_common_strncmp( user_input, "yes", 3 ) == 0 )
		{
			wipe_block_on_read_error = 1;
		}
		else if( libewf_common_strncmp( user_input, "no", 2 ) == 0 )
		{
			wipe_block_on_read_error = 0;
		}
		else
		{
			fprintf( stderr, "ewfacquire: unsuported answer.\n" );

			return( EXIT_FAILURE );
		}
		libewf_common_free( user_input );

		fprintf( stdout, "\n" );
	}
	/* Check if user is content with values
	 */
	while( confirm_input( filename, case_number, description, evidence_number, examiner_name, notes, compression_level, compress_empty_block, ewf_format, acquiry_offset, acquiry_size, ewf_file_size, sectors_per_block, error_granularity, read_error_retry, wipe_block_on_read_error ) == 0 );

	if( case_number != NULL )
	{
		header_values->case_number = libewf_header_values_set_value( header_values->case_number, case_number );

		libewf_common_free( case_number );
	}
	if( description != NULL )
	{
		header_values->description = libewf_header_values_set_value( header_values->description, description );

		libewf_common_free( description );
	}
	if( evidence_number != NULL )
	{
		header_values->evidence_number = libewf_header_values_set_value( header_values->evidence_number, evidence_number );

		libewf_common_free( evidence_number );
	}
	if( examiner_name != NULL )
	{
		header_values->examiner_name = libewf_header_values_set_value( header_values->examiner_name, examiner_name );

		libewf_common_free( examiner_name );
	}
	if( notes != NULL )
	{
		header_values->notes = libewf_header_values_set_value( header_values->notes, notes );

		libewf_common_free( notes );
	}
	/* Done asking user input
	 */
	timestamp_start = time( NULL );

	fprintf( stderr, "\nAcquiry started at: %s\n", ctime( &timestamp_start ) );
	fprintf( stderr, "This could take a while.\n" );

	filenames[ 0 ] = filename;

	handle = libewf_open( (const char **) filenames, 1, LIBEWF_OPEN_WRITE );

	if( handle == NULL )
	{
		fprintf( stderr, "Unable to create EWF file handle.\n" );

		return( EXIT_FAILURE );
	}
	data_set = libewf_write_set_parameters( handle, size_input_file, sectors_per_block, 512, error_granularity, ewf_file_size, compression_level, ewf_format, read_error_retry, wipe_block_on_read_error, compress_empty_block );

	if( data_set == NULL )
	{
		fprintf( stderr, "Unable to set acquiry parameters.\n" );

		libewf_close( handle );

		return( EXIT_FAILURE );
	}
	data_set = libewf_write_set_header_values( handle, header_values );

	if( data_set == NULL )
	{
		fprintf( stderr, "Unable to set header values.\n" );

		libewf_close( handle );

		return( EXIT_FAILURE );
	}
	handle->swap_byte_pairs = swap_byte_pairs;

	count = libewf_write_from_file_descriptor( handle, file_descriptor, acquiry_size, acquiry_offset, callback );

	if( count > -1 )
	{
		calculated_md5hash_string = libewf_data_md5hash( handle );

		if( calculated_md5hash_string == NULL )
		{
			fprintf( stderr, "MD5 hash calculated over data: N/A\n" );
		}
		else
		{
			fprintf( stderr, "MD5 hash calculated over data: %s\n", calculated_md5hash_string );

			libewf_string_free( calculated_md5hash_string );
		}
	}
	libewf_close( handle );

	libewf_common_close( file_descriptor );

	libewf_common_free( filename );
	libewf_header_values_free( header_values );

	timestamp_end = time( NULL );

	if( count <= -1 )
	{
		fprintf( stderr, "Acquiry failed at: %s\n", ctime( &timestamp_end ) );

		return( EXIT_FAILURE );
	}
	fprintf( stderr, "Acquiry completed at: %s\n", ctime( &timestamp_end ) );

	print_acquiry_information( count, timestamp_start, timestamp_end );

	return( EXIT_SUCCESS );
}

