/***************************************************************************
    smb4kglobal  -  This is the global namespace for Smb4K.
                             -------------------
    begin                : Sa Apr 2 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qobject.h>
#include <qlayout.h>
#include <qlabel.h>

// KDE includes
#include <kconfig.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <klocale.h>

// other includes
#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <errno.h>

// application specific includes
#include "smb4kglobal.h"
#include "smb4kdefs.h"
#include "smb4kerror.h"

class Smb4KGlobalPrivate
{
  public:
    /**
     * Constructor
     */
    Smb4KGlobalPrivate();

    /**
     * Destructor
     */
    ~Smb4KGlobalPrivate();

    /**
     * Returns a pointer to the global timer
     */
    QTimer *timer();

    /**
     * Returns a pointer to the global config object
     */
    KConfig *config();

    /**
     * Returns the handler for homes shares
     */
    Smb4KHomesSharesHandler *homesHandler();

    /**
     * Returns a pointer to the global password handler object
     */
    Smb4KPasswordHandler *passwordHandler();

    /**
     * Returns a pointer to the global Samba options handler
     */
    Smb4KSambaOptionsHandler *optionsHandler();

    /**
     * Returns the name of the temporary directory.
     */
    const QString &tempDir();

    /**
     * Return the user names defined for a certain homes share
     */
    const QStringList homesUsers( const QString &host );

  private:
    QTimer *m_timer;
    KConfig *m_config;
    Smb4KPasswordHandler *m_passwd_handler;
    Smb4KSambaOptionsHandler *m_options_handler;
    Smb4KHomesSharesHandler *m_homes_handler;
    QString m_temp_dir;
};


Smb4KGlobalPrivate::Smb4KGlobalPrivate()
{
  m_timer = new QTimer();
  m_timer->start( TIMER_INTERVAL, false );

  // Do NOT initialize these classes here; you'll
  // get crashes.
  m_config = NULL;
  m_passwd_handler = NULL;
  m_options_handler = NULL;
  m_homes_handler = NULL;

  m_temp_dir = QString::null;
}


Smb4KGlobalPrivate::~Smb4KGlobalPrivate()
{
  rmdir( m_temp_dir.local8Bit() );

  delete m_timer;
  delete m_config;
  delete m_passwd_handler;
  delete m_options_handler;
  delete m_homes_handler;
}


QTimer *Smb4KGlobalPrivate::timer()
{
  return m_timer;
}


KConfig *Smb4KGlobalPrivate::config()
{
  return m_config ? m_config : (m_config = new KConfig( "smb4krc", false, false, "config" ));
}


Smb4KHomesSharesHandler *Smb4KGlobalPrivate::homesHandler()
{
  if ( !m_homes_handler )
  {
    m_homes_handler = new Smb4KHomesSharesHandler();

    // Convert old entries:
    m_homes_handler->convert( m_config );
  }

  return m_homes_handler;
}


Smb4KPasswordHandler *Smb4KGlobalPrivate::passwordHandler()
{
#ifndef __FreeBSD__
  return m_passwd_handler ? m_passwd_handler :
                            (m_passwd_handler = new Smb4KPasswordHandler( config(), homesHandler() ));
#else
  return m_passwd_handler ? m_passwd_handler :
                            (m_passwd_handler = new Smb4KPasswordHandler( config(), homesHandler(), optionsHandler() ));
#endif
}


Smb4KSambaOptionsHandler *Smb4KGlobalPrivate::optionsHandler()
{
  return m_options_handler ? m_options_handler :
                             (m_options_handler = new Smb4KSambaOptionsHandler( config() ));
}


const QString &Smb4KGlobalPrivate::tempDir()
{
  if ( m_temp_dir.isEmpty() )
  {
    char tmpd_name[] = "/tmp/smb4k.XXXXXX";

    if ( mkdtemp( tmpd_name ) == NULL )
    {
      Smb4KError::error( ERROR_CREATING_TEMP_DIR, tmpd_name, strerror( errno ) );

      return QString::null;
    }

    m_temp_dir = QString( tmpd_name );
  }

  return m_temp_dir;
}


const QStringList Smb4KGlobalPrivate::homesUsers( const QString &host )
{
  if ( !m_homes_handler )
  {
    m_homes_handler = new Smb4KHomesSharesHandler();
  }

  return m_homes_handler->homesUsers( host );
}


static Smb4KGlobalPrivate p;



/****************************************************************************
   Returns the kernel version
****************************************************************************/

const QString Smb4KGlobal::kernelVersion()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.release ).section( "-", 0, 0 );
}


/****************************************************************************
   Returns the system name
****************************************************************************/

const QString Smb4KGlobal::systemName()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.sysname );
}


/****************************************************************************
   Returns the UMASK
****************************************************************************/

int Smb4KGlobal::getUMASK()
{
  return umask( 0 );
}


/***************************************************************************
   Returns a pointer to the global KConfig object.
***************************************************************************/

KConfig *Smb4KGlobal::config()
{
  return p.config();
}


/***************************************************************************
   Returns a pointer to the global timer object
***************************************************************************/

QTimer *Smb4KGlobal::timer()
{
  return p.timer();
}


/***************************************************************************
   Returns the timer interval
***************************************************************************/

const int Smb4KGlobal::timerInterval()
{
  return TIMER_INTERVAL;
}


/***************************************************************************
   Will return a specified user for a homes share
***************************************************************************/

const QString Smb4KGlobal::specifyUser( const QString &host, QWidget *parent, const char *name )
{
  return p.homesHandler()->specifyUser( host, parent, name );
}


Smb4KPasswordHandler *Smb4KGlobal::passwordHandler()
{
  return p.passwordHandler();
}


Smb4KSambaOptionsHandler *Smb4KGlobal::optionsHandler()
{
  return p.optionsHandler();
}


const QString &Smb4KGlobal::tempDir()
{
  return p.tempDir();
}


const QStringList Smb4KGlobal::homesUsers( const QString &host )
{
  return p.homesUsers( host );
}

