/*
 * ---------------------------------------------------------------------
 * diskimage.c
 *	Library for diskimages for use in nonvolatile memory device
 *	emulators like amdflash or serial eeproms 
 *
 * (C) 2005 Jochen Karrer
 *    Author: Jochen Karrer
 *
 * Status: Working 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/mman.h>
#include <sys/types.h>
#include "diskimage.h"

DiskImage *
DiskImage_Open(const char *name,size_t size,int flags) {
	DiskImage *di;
	int i;
	uint8_t buf[256];
	uint8_t emptyval;
	off_t pos;
	ssize_t count;
	di = malloc(sizeof(DiskImage));
	if(!di) {
		fprintf(stderr,"out of memory\n");
		exit(14);
	}
	memset(di,0,sizeof(DiskImage));
	di->size = size;
	if(flags & DI_RDWR) {
		if(flags & (DI_CREAT_FF | DI_CREAT_00)) {
			di->fd=open(name,O_RDWR|O_CREAT,0644);
		} else {
			di->fd=open(name,O_RDWR,0644);
		}
	} else {
		di->fd=open(name,O_RDONLY);
	}
	if(di->fd<0) {
		fprintf(stderr,"Can't open image \"%s\"",name);
		perror("");
		free(di);
		return NULL;
	}
	if(flock(di->fd,LOCK_EX|LOCK_NB)<0) {
		fprintf(stderr,"Can't get lock for diskimage \"%s\"\n",name);
		close(di->fd);
		free(di);
		return NULL;
	}
	if((pos=lseek(di->fd,0,SEEK_END))==(off_t)-1) {
		perror("lseek on flashfile failed");
		close(di->fd);
		free(di);
		return NULL;
	}
	count=di->size-pos;
	if(flags & DI_CREAT_FF) {
		emptyval = 0xff;
	} else {
		emptyval = 0x00;
	}
	for(i=0;i<256;i++) {
		buf[i]=emptyval;
	}
	while(count>256) {
		int result;
		result = write(di->fd,buf,256);
		if(result<=0) {
			perror("write failed");
			break;
		} else {
			count -= result;
		}
	}
	while(count>0) {
		if(write(di->fd,buf,1)<=0) {
			perror("write failed");
			break;
		}
		count--;
	}
	return di;
}

int
DiskImage_Read(DiskImage *di,off_t ofs,uint8_t *buf,int count) 
{
	int result;
	int cnt;
	if(lseek(di->fd,ofs,SEEK_SET) != ofs) {
		return -EINVAL;
	}
	for(cnt=0;cnt < count;) {
		result = read(di->fd,buf+cnt,count);
		if(result<=0) {
			if(cnt) {
				return cnt;
			} else {
				return result;
			}
		} 	
		cnt += result;
	}
	return cnt;
	
}
int
DiskImage_Write(DiskImage *di,off_t ofs,const uint8_t *buf,int count) 
{
	int result;
	int cnt;
	if(lseek(di->fd,ofs,SEEK_SET) != ofs) {
		return -EINVAL;
	}
	for(cnt=0;cnt < count;) {
		result = write(di->fd,buf+cnt,count);
		if(result<=0) {
			if(cnt) {
				return cnt;
			} else {
				return result;
			}
		} 	
		cnt += result;
	}
	return cnt;
}

void *
DiskImage_Mmap(DiskImage *di) 
{
	di->map=mmap(0,di->size,PROT_READ|PROT_WRITE,MAP_SHARED,di->fd,0);
	if(di->map == (void*)-1) {
		perror("mmap of diskimage failed");
		close(di->fd);
		free(di);
		return NULL;
	}
	return di->map;
}

void
DiskImage_Close(DiskImage *di) 
{
	if(di->map) {
		munmap(di->map,di->size);
		di->map=NULL;
	}
	flock(di->fd,LOCK_UN);
	close(di->fd);
	free(di);
}
