#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <bus.h>
#include <fio.h>
#include <cycletimer.h>
#include <unistd.h>
#include <errno.h>
#include <signode.h>
#include <configfile.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <m16c_uart.h>

typedef struct M16C_Uart {
	uint32_t base;
	uint8_t uxmr;
	uint8_t uxbrg;
	uint16_t uxtb;
	uint16_t uxrb;
	uint8_t uxc0;
	uint8_t uxc1;
	uint8_t txreg; /* shift register to the outside */

	int fd;
	FIO_FileHandler input_fh;
        FIO_FileHandler output_fh;
        int ifh_is_active;
        int ofh_is_active;
	CycleTimer tx_timer;
	CycleTimer rx_timer;
	SigNode *irqNode;
} M16C_Uart;

#define UxMR(uart)	(0x0+(uart)->base)
#define		UxMR_SMD_MASK	(7)
#define		UxMR_SMD_SHIFT	(0)
#define		UxMR_CKDIR	(1<<3)
#define		UxMR_STPS	(1<<4)
#define		UxMR_PRY	(1<<5)
#define 	UxMR_PRYE	(1<<6)
#define		UxMR_IOPOL	(1<<7)

#define UxBRG(uart)	(0x1+(uart)->base)
#define UxTB(uart)	(0x2+(uart)->base)
#define UxTBL(uart)	(0x2+(uart)->base)
#define UxTBH(uart)	(0x3+(uart)->base)
#define	 	UxTB_ABT	(1<<11)
#define	 	UxTB_OER	(1<<12)
#define	 	UxTB_FER	(1<<13)
#define	 	UxTB_PER	(1<<14)
#define	 	UxTB_SUM	(1<<15)
#define UxC0(uart)	(0x4+(uart)->base)
#define		UxC0_CLK0	(1<<0)
#define		UxC0_CLK1	(1<<1)
#define		UxC0_CRS	(1<<2)
#define		UxC0_TXEPT	(1<<3)
#define		UxC0_CRD	(1<<4)
#define		UxC0_NCH	(1<<5)
#define		UxC0_CKPOL	(1<<6)
#define		UxC0_UFORM	(1<<7)
#define UxC1(uart)	(0x5+(uart)->base)
#define		UxC1_TE		(1<<0)
#define		UxC1_TI		(1<<1)
#define		UxC1_RE		(1<<2)
#define		UxC1_RI		(1<<3)
#define		UxC1_U2IRS	(1<<4)
#define		UxC1_U2RRM	(1<<5)
#define		UxC1_UxLCH	(1<<6)
#define		UxC1_UxERE	(1<<7)
#define UxRB(uart)	(0x6+(uart)->base)
#define UxRBL(uart)	(0x6+(uart)->base)
#define UxRBH(uart)	(0x7+(uart)->base)
#define 	UxRB_ABT	(1<<11)
#define		UxRB_OER	(1<<12)
#define		UxRB_FER	(1<<13)
#define		UxRB_PER	(1<<14)
#define		UxRB_SUM	(1<<15)

#define UCON	(0x3b0)
#define 	UCON_U0IRS	(1<<0)
#define		UCON_U1IRS	(1<<1)
#define		UCON_U0RRM	(1<<2)
#define		UCON_U1RRM	(1<<3)
#define		UCON_CLKMD0	(1<<4)
#define		UCON_CLKMD1	(1<<5)
#define		UCON_RCSP	(1<<6)


static inline void enable_rx(M16C_Uart *);
static inline void disable_rx(M16C_Uart *);
static inline void enable_tx(M16C_Uart *);
static inline void disable_tx(M16C_Uart *);


static void
update_interrupts(M16C_Uart  *uart)
{
	
}

static void
update_serconfig(M16C_Uart *uart)
{
	if(!isatty(uart->fd)) {
                return;
        }
}


/*
 * ----------------------------------------------------
 *  Close the output/input source/destination
 * ----------------------------------------------------
 */
static void
uart_close(M16C_Uart *uart) {
        if(uart->fd<0) {
                return;
        }
        disable_rx(uart);
        disable_tx(uart);
        close(uart->fd);
        uart->fd = -1;
        return;
}

static int
serial_output(void *cd,int mask) 
{
	M16C_Uart *uart = (M16C_Uart *)cd;
	uint8_t c = uart->txreg;
	int result = write(uart->fd,&c,1);
	if(result == 1) {
		uart->uxc0 = uart->uxc0 | UxC0_TXEPT;
		disable_tx(uart);
		// trigger some interrupt
		update_interrupts(uart);
	} else if(result == 0) {
		fprintf(stderr,"M16C_Uart fd: EOF on output\n");
		uart_close(uart);
	} else if((result<0) && (errno == EAGAIN)) {
		return 0;
	} else {
		fprintf(stderr,"M16C_Uart fd: Error %d on output\n",errno);
		uart_close(uart);
	}
	return 0;
}

static void
enable_rx_timer(void *clientData) 
{
	M16C_Uart *uart = (M16C_Uart *)clientData;
	if(uart->uxc1 & UxC1_RE) {
		enable_rx(uart);
	}
}

static int
serial_input(void *cd,int mask) 
{
	uint8_t c;
	M16C_Uart *uart = (M16C_Uart *)cd;
	int result = read(uart->fd,&c,1);
	if(result == 1) {
		uart->uxrb = c; 
		if(uart->uxc1 & UxC1_RI) {
			/* Overrun  */
			uart->uxrb |= UxRB_OER | UxRB_SUM; 
		} 
		if(uart->uxc1 & UxC1_RE) {
			CycleTimer_Add(&uart->rx_timer,MicrosecondsToCycles(100),enable_rx_timer,uart);
		}
		uart->uxc1 |= UxC1_RI;
		disable_rx(uart);
		update_interrupts(uart);
	} else if(result == 0) {
		fprintf(stderr,"M16C_Uart fd: EOF on input\n");
		uart_close(uart);
	} else if((result < 0 ) && (errno == EAGAIN)) {
		return 0;
	} else {
		fprintf(stderr,"M16C_Uart fd: Error %d on input\n",errno);
		uart_close(uart);
	}
	return 0;
}
/*
 * -----------------------------------------------------------------
 * enable_tx
 *      Add the file event handler for transmission
 * -----------------------------------------------------------------
 */

static inline void
enable_tx(M16C_Uart *uart) 
{
        if((uart->fd>=0) && !(uart->ofh_is_active)) {
                FIO_AddFileHandler(&uart->output_fh,uart->fd,FIO_WRITABLE,serial_output,uart);
                uart->ofh_is_active=1;
        }
}

static inline void
disable_tx(M16C_Uart *uart) 
{
        if(uart->ofh_is_active) {
                FIO_RemoveFileHandler(&uart->output_fh);
        	uart->ofh_is_active=0;
        }
}
/*
 * ----------------------------------------------------------------
 * enable_rx
 *	Add the file event handler for reception
 * ----------------------------------------------------------------
 */
static inline void
enable_rx(M16C_Uart *uart) 
{
        if((uart->fd>=0) && !(uart->ifh_is_active)) {
                FIO_AddFileHandler(&uart->input_fh,uart->fd,FIO_WRITABLE,serial_input,uart);
                uart->ifh_is_active=1;
        }
}

static inline void
disable_rx(M16C_Uart *uart) 
{
        if(uart->ifh_is_active) {
                FIO_RemoveFileHandler(&uart->input_fh);
        	uart->ifh_is_active=0;
        }
}

/*
 * --------------------------------------------------------------
 * Timerhandler triggers transmission some time after
 * writing to the transmission register depending on
 * Baudrate 
 * --------------------------------------------------------------
 */
static void
trigger_tx(void *clientData) 
{
	M16C_Uart *uart = (M16C_Uart *)clientData;
	if(uart->uxc0 & UxC0_TXEPT) {
		/* Only unstarted tranmissions are avoided if transmitter is disabled */
		if(!(uart->uxc1 & UxC1_TE)) {
			return;
		}
		if(!(uart->uxc1 & UxC1_TI)) {
			uart->uxc0 &= ~UxC0_TXEPT;
			uart->txreg = uart->uxtb;
			uart->uxc1 |= UxC1_TI;
		} 
	} else {
		enable_tx(uart);
	}
	if(!(uart->uxc1 & UxC1_TI)) {
		CycleTimer_Add(&uart->tx_timer,MicrosecondsToCycles(100),trigger_tx,uart);
	} 
}
/*
 * -------------------------------------------------------------
 * Mode register
 * -------------------------------------------------------------
 */
static uint32_t
uxmr_read(void *clientData,uint32_t address,int rqlen)  
{
	M16C_Uart *uart = (M16C_Uart*) clientData;
	return uart->uxmr;
}

static void
uxmr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	M16C_Uart *uart = (M16C_Uart*) clientData;
	uart->uxmr = value;
	update_serconfig(uart);
}

/*
 * --------------------------------------------------------
 * Baud rate generator divides clcksrc by (brg+1)
 * The result is divided by 16 for async and by 2 for sync.
 * The clock src is f1SIO, f2SIO, f8SIO of f32SIO
 * --------------------------------------------------------
 */
static uint32_t
uxbrg_read(void *clientData,uint32_t address,int rqlen)  
{
	M16C_Uart *uart = (M16C_Uart*) clientData;
	return uart->uxbrg;
}

static void
uxbrg_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	M16C_Uart *uart = (M16C_Uart*) clientData;
	uart->uxbrg = value;
}

/* Transmit register is not readable */
static uint32_t
uxtb_read(void *clientData,uint32_t address,int rqlen)  
{
	return 0;
}

static void
uxtb_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	M16C_Uart *uart = (M16C_Uart*) clientData;
	uart->uxtb = value;
	if(!(uart->uxc1 & UxC1_TI)) {
		fprintf(stderr,"M16C_Uart warning: writing Tx register while TX buffer not empty\n");
	}
	uart->uxc1 = uart->uxc1 & ~UxC1_TI;
	if(!CycleTimer_IsActive(&uart->tx_timer)) {
		CycleTimer_Add(&uart->tx_timer,MicrosecondsToCycles(1),trigger_tx,uart);
	}
}

static uint32_t
uxc0_read(void *clientData,uint32_t address,int rqlen)  
{
	return 0;
}

static void
uxc0_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{

}
static uint32_t
uxc1_read(void *clientData,uint32_t address,int rqlen)  
{
	M16C_Uart *uart = (M16C_Uart *) clientData;
	return uart->uxc1;
}

static void
uxc1_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	M16C_Uart *uart = (M16C_Uart *) clientData;
	uart->uxc1 = value;
	if(value & UxC1_TE) {
		if(!CycleTimer_IsActive(&uart->tx_timer)) {
			CycleTimer_Add(&uart->tx_timer,MicrosecondsToCycles(1),trigger_tx,uart);
		}
	}
	if(value & UxC1_RE) {
		if(!CycleTimer_IsActive(&uart->rx_timer)) {
			CycleTimer_Add(&uart->rx_timer,MicrosecondsToCycles(100),enable_rx_timer,uart);
		}
	} else {
		disable_rx(uart);
	}
}

static uint32_t
uxrb_read(void *clientData,uint32_t address,int rqlen)  
{
	M16C_Uart *uart = (M16C_Uart *)clientData;
	if(uart->uxc1 & UxC1_RI) {
		uart->uxc1 &= ~UxC1_RI;
		update_interrupts(uart); 
	}
	return uart->uxrb;
}

static void
uxrb_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{

}

static void
M16C_UartMap(void *owner,uint32_t base,uint32_t mask,uint32_t flags) {
	M16C_Uart *uart = (M16C_Uart *) owner;

	IOH_New8(UxMR(uart),uxmr_read,uxmr_write,uart);
	IOH_New8(UxBRG(uart),uxbrg_read,uxbrg_write,uart);
	IOH_New16(UxTB(uart),uxtb_read,uxtb_write,uart);
	IOH_New8(UxC0(uart),uxc0_read,uxc0_write,uart);
	IOH_New8(UxC1(uart),uxc1_read,uxc1_write,uart);
	IOH_New16(UxRB(uart),uxrb_read,uxrb_write,uart);
	//IOH_New32(UCON);
}

typedef struct UartInfo {
	char *name;
	uint16_t baseaddr;
} UartInfo;
#define NUM_UARTS (3)
static UartInfo uartinfo[NUM_UARTS] = {
	{
		name: "uart0",
		baseaddr: 0x3a0,
	},
	{
		name: "uart1",
		baseaddr: 0x3a8,
	},
	{
		name: "uart2",
		baseaddr: 0x1fb,
	},
};

static void
M16C_UartNew(char *devname,uint32_t base)
{
	char *filename;
	M16C_Uart *uart = (M16C_Uart*) malloc(sizeof(M16C_Uart));
	if(!uart) {
		fprintf(stderr,"Out of memory allocating M16C Uart\n");
		exit(7);
	}
	memset(uart,0,sizeof(M16C_Uart));
	uart->fd = -1;
	uart->base = base;
	uart->uxmr = 0;
	uart->uxc0 = UxC0_TXEPT;
	uart->uxc1 = UxC1_TI;
	uart->irqNode = SigNode_New("%s.irq",devname);
	filename=Config_ReadVar("M16C",devname);
        if(filename) {
                /* does cygwin have /dev/stdin ? */
                if(!strcmp(filename,"stdin")) {
                        uart->fd = 0;
                } else {
                        uart->fd = open(filename,O_RDWR);
                }
                if(uart->fd<0) {
                        fprintf(stderr,"%s: Cannot open %s\n",devname,filename);
                        sleep(1);
                } else {
                        fcntl(uart->fd,F_SETFL,O_NONBLOCK);
                        fprintf(stderr,"M16C Serial %s Connected to %s\n",devname,filename);
                }
        } else {
                fprintf(stderr,"M16C Serial %s connected to nowhere\n",devname);
        }

}

void
M16C_UartsNew() 
{
	int i;
	for(i=0;i<NUM_UARTS;i++) {
		uint16_t iobase;
		char *name;
		UartInfo *inf = &uartinfo[i];
		iobase = inf->baseaddr;	
		name = inf->name;
		M16C_UartNew(name,iobase);
	}
}
