/*
 * -------------------------------------------------------------------------
 * Read Motorola S-Record files
 * 
 * (C) 2004  2005 Lightmaze Solutions AG
 * Author: Jochen Karrer 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "srec.h"

/*
 * ----------------------------------------------------
 * Special functions for reading hex numbers 
 * because sscanf is incredible slow
 * ----------------------------------------------------
 */
static int
hexparse_n(const char *str,unsigned int *retval,int bytes) {
	int i;
        unsigned int val=0;
	for(i=0;i<(bytes<<1);i++) {
		char c = *str;
                if(c>='0' && c<='9') {
                        val=val*16+(c - '0');
                } else if(c>='A' && c<='F') {
                        val=val*16+(c - 'A'+10);
                } else if(c>='a' && c<='f') {
                        val=val*16+(c - 'a'+10);
                } else {
			return -1;
		}
                str++;
        }
	*retval = val;
	return 1;
}

static inline int
hexparse(const char *str,uint8_t *retval) {
	int i;
	unsigned int val=0;
	for(i=0;i<2;i++) {
                if(*str>='0' && *str<='9') {
                        val=val*16+(*str - '0');
                } else if(*str>='A' && *str<='F') {
                        val=val*16+(*str - 'A'+10);
                } else if(*str>='a' && *str<='f') {
                        val=val*16+(*str - 'a'+10);
                } else {
			return -1;
		}
                str++;
        }
	*retval = val;
	return 1;
}

static int
parse_srec_data(uint8_t *buf,const char *line,int count,uint32_t sum) 
{
	int i;
	uint8_t chksum;
	uint8_t val;
	for(i=0;i<count-1;i++) {
		if(hexparse(line+2*i,&val)!=1) {
			fprintf(stderr,"can not parse data\n");
			return -9;
		}
		buf[i]=val;
		sum+=val;
	}	
	sum = ~sum & 0xff;
	if(hexparse(line+2*(count-1),&chksum)!=1) {
		fprintf(stderr,"Can not parse checksum\n");
		return -99;
	}
	if(sum!=chksum) {
		fprintf(stderr,"checksum error in srecord \"%s\" sum %02x chksum %08x\n",line,sum,chksum);
//warning no checksum
		return -1;
	}
	return count-1;
}
/*
 * ----------------------------------
 * Read one SRecord (one line) 
 * ----------------------------------
 */
static int 
read_record(FILE *file,uint32_t *addr,uint8_t *buf) {
	char line[100];
	int len;
	unsigned int sum;
	uint8_t count;
	if(fgets(line,80,file)!=line) {
		if(feof(file)) {
			return 0;
		}
		fprintf(stderr,"Error reading SRecords\n");
		return -1;
	}	
	len = strlen(line);
	if(len<4) {
		fprintf(stderr,"Not an SRecord: \"%s\"\n",line);
		return -1;
	}
	if(line[0]!='S') {
		fprintf(stderr,"Not an SRecord file\n");
		return -1;
	}
	if(hexparse(line+2,&count)!=1) {
		fprintf(stderr,"Can not read count from record\n");
		return -2;
	}
	if(len<(4+2*count)) {
		fprintf(stderr,"Truncated SRecord \"%s\"\n",line);
		return -45;
	}
	switch(line[1]) {
		case '0':
			{
				int i;
				count = parse_srec_data(buf,line+8,count-2,count);
				if(count>=10) {
					fprintf(stderr,"S0 record: \"");			
					for(i=0;i<count;i++) {
						fprintf(stderr,"%c",buf[i]);
					}
					fprintf(stderr,"\"\n");			
				} else {
					fprintf(stderr,"Ignore S0 record\n");			
				}
			}
			return 0;
			break;

		case '1':
			if(hexparse_n(line+4,addr,2)!=1) {
				fprintf(stderr,"Can not parse S2 address\n");
				return -34;
			}
			sum=(*addr&0xff)+((*addr>>8)&0xff) +((*addr>>16)&0xff) + ((*addr>>24)&0xff)+count;
			return parse_srec_data(buf,line+8,count-2,sum);
				
		case '2':
			if(hexparse_n(line+4,addr,3)!=1) {
				fprintf(stderr,"Can not parse S2 address\n");
				return -34;
			}
			sum=(*addr&0xff)+((*addr>>8)&0xff) +((*addr>>16)&0xff) + ((*addr>>24)&0xff)+count;
			return parse_srec_data(buf,line+10,count-3,sum);

		case '3':
			if(hexparse_n(line+4,addr,4)!=1) {
				fprintf(stderr,"Can not parse S3 address\n");
				return -34;
			}
			sum=(*addr&0xff)+((*addr>>8)&0xff) +((*addr>>16)&0xff) + ((*addr>>24)&0xff)+count;
			return parse_srec_data(buf,line+12,count-4,sum);

		case '5':
			/* fprintf(stderr,"Ignore S5 record\n"); */
			return 0;

		case '7':
			if(hexparse_n(line+4,addr,4)!=1) {
				fprintf(stderr,"bad S7 record\n");
				return -1;
			}
			fprintf(stderr,"S7 record (Start address 0x%08x) Ignored !\n",*addr);
			return 0;
			
		case '8':
			if(hexparse_n(line+4,addr,3)!=1) {
				fprintf(stderr,"bad S8 record\n");
				return -1;
			}
			fprintf(stderr,"Ignore S8 record (Start address 0x%06x)\n",*addr);
			return 0;
		case '9':
			if(hexparse_n(line+4,addr,2)!=1) {
				fprintf(stderr,"bad S9 record\n");
				return -1;
			}
			fprintf(stderr,"Ignore S8 record (Start address 0x%04x)\n",*addr);
			return 0;
		default: 
			fprintf(stderr,"Unknown SRecord type %c\n",line[1]);
			break;
	}
	return 0;	
}

int
XY_LoadSRecordFile(char *filename,XY_SRecCallback *callback,void *clientData) 
{
	uint8_t buf[100];
	uint32_t addr=0;
	int result;
	FILE *file = fopen(filename,"r");
	if(!file) {
		return -1;
	}
	while(!feof(file)) { 
		int count;
		if((count=read_record(file,&addr,buf))<0) {
			break;
		}
		if(callback) { 
			result=callback(addr,buf,count,clientData);
			if(result<0) {
				return result;
			}
		}
	}
	fclose(file);
	return 0;
}


#ifdef TEST
int
main(int argc,char *argv[]) {
	if(argc<2) {
		fprintf(stderr,"Argument missing\n");
		exit(4235);
	}
	Load_SRecords(argv[1]); 
	exit(0);	
}
#endif 
