/*
 * --------------------------------------------------------------------------
 *
 * Interface to USB devices 
 * (C) 2006 Jochen Karrer 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * -------------------------------------------------------------------------
 */

#include <usbproto.h>

#define EPNT_TYPE_CONTROL	(0)
#define EPNT_TYPE_ISO		(1)
#define EPNT_TYPE_BULK		(2)
#define EPNT_TYPE_INT		(3)

#define EPNT_DIR_IN		(1)
#define EPNT_DIR_OUT		(2)

struct UsbDevice;

typedef struct UsbDev_Endpoint {
	int type;
	int direction;
	unsigned int toggle; 
	struct UsbDevice *owner;

	/* Out Endpoint is input of device */
	int (*toDevice)(struct UsbDev_Endpoint *,const UsbRequest *);

	/* In Endpoint is output of device */
	int  (*fromDevice)(struct UsbDev_Endpoint *,UsbRequest *);
	void (*doNext)(struct UsbDev_Endpoint *);
} UsbDev_Endpoint;

enum usb_device_speed {
        USB_SPEED_UNKNOWN = 0,                  /* enumerating */
        USB_SPEED_LOW, USB_SPEED_FULL,          /* */
        USB_SPEED_HIGH,                         /* */
        USB_SPEED_VARIABLE,                     /* wireless (usb 2.5) */
};

enum usb_device_state {
        /* NOTATTACHED isn't in the USB spec, and this state acts
         * the same as ATTACHED ... but it's clearer this way.
         */
        USB_STATE_NOTATTACHED = 0,
        /* the chapter 9 device states */
        USB_STATE_ATTACHED,
        USB_STATE_POWERED,
        USB_STATE_DEFAULT,                      /* limited function */
        USB_STATE_ADDRESS,
        USB_STATE_CONFIGURED,                   /* most functions */

        USB_STATE_SUSPENDED

        /* NOTE:  there are actually four different SUSPENDED
         * states, returning to POWERED, DEFAULT, ADDRESS, or
         * CONFIGURED respectively when SOF tokens flow again.
         */
};

typedef struct UsbToken {
	uint8_t pid;
	uint8_t addr;
	uint8_t epnum;
	/* uint16_t frmnumb ? */
} UsbToken;
/*
 * ---------------------------------------------------------
 * UsbDevice structure
 * 	Many fields stolen from linux kernel usb_device
 * ---------------------------------------------------------
 */
typedef struct UsbDevice {
	int             devnum;         /* Address on USB bus (0-127) */
	enum usb_device_state   state;  /* configured, not attached, etc */
	enum usb_device_speed   speed;	/* high/full/low (or error) */
	int	ta_state;		/* Transaction state defined in figure 8.21 */
	UsbToken token;			/* Transaction state machine requires this */
	UsbDev_Endpoint *ep_in[16];
	UsbDev_Endpoint *ep_out[16];
} UsbDevice;


/*
 * Standard requests, for the bRequest field of a SETUP packet.
 *
 * These are qualified by the bRequestType field, so that for example
 * TYPE_CLASS or TYPE_VENDOR specific feature flags could be retrieved
 * by a GET_STATUS request.
 */
#define USB_REQ_GET_STATUS              0x00
#define USB_REQ_CLEAR_FEATURE           0x01
#define USB_REQ_SET_FEATURE             0x03
#define USB_REQ_SET_ADDRESS             0x05
#define USB_REQ_GET_DESCRIPTOR          0x06
#define USB_REQ_SET_DESCRIPTOR          0x07
#define USB_REQ_GET_CONFIGURATION       0x08
#define USB_REQ_SET_CONFIGURATION       0x09
#define USB_REQ_GET_INTERFACE           0x0A
#define USB_REQ_SET_INTERFACE           0x0B
#define USB_REQ_SYNCH_FRAME             0x0C

#define USB_REQ_SET_ENCRYPTION          0x0D    /* Wireless USB */
#define USB_REQ_GET_ENCRYPTION          0x0E
#define USB_REQ_SET_HANDSHAKE           0x0F
#define USB_REQ_GET_HANDSHAKE           0x10
#define USB_REQ_SET_CONNECTION          0x11
#define USB_REQ_SET_SECURITY_DATA       0x12
#define USB_REQ_GET_SECURITY_DATA       0x13
#define USB_REQ_SET_WUSB_DATA           0x14
#define USB_REQ_LOOPBACK_DATA_WRITE     0x15
#define USB_REQ_LOOPBACK_DATA_READ      0x16
#define USB_REQ_SET_INTERFACE_DS        0x17

/*
 * Descriptor types ... USB 2.0 spec table 9.5
 */
#define USB_DT_DEVICE                   0x01
#define USB_DT_CONFIG                   0x02
#define USB_DT_STRING                   0x03
#define USB_DT_INTERFACE                0x04
#define USB_DT_ENDPOINT                 0x05
#define USB_DT_DEVICE_QUALIFIER         0x06
#define USB_DT_OTHER_SPEED_CONFIG       0x07
#define USB_DT_INTERFACE_POWER          0x08


/* USB_DT_DEVICE: Device descriptor */
struct usb_device_descriptor {
        uint8_t  bLength;
        uint8_t  bDescriptorType;

        le16 bcdUSB;
        uint8_t  bDeviceClass;
        uint8_t  bDeviceSubClass;
        uint8_t  bDeviceProtocol;
        uint8_t  bMaxPacketSize0;
        le16 idVendor;
        le16 idProduct;
        le16 bcdDevice;
        uint8_t  iManufacturer;
        uint8_t  iProduct;
        uint8_t  iSerialNumber;
        uint8_t  bNumConfigurations;
} __attribute__ ((packed));


#define USB_MAX_PKTLEN (8192)


/*
 * The Interface  
 */
UsbDevice * UsbDevice_New(void);
void UsbDevice_RegisterEndpoint(UsbDevice *udev,UsbDev_Endpoint *ep);
