/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "audio.h"
#include "other.h"
#ifdef ENABLE_STREAMING
#include "output/streaming/streaming.h"
#endif
#ifdef ENABLE_LAME
#include "output/lame/lame.h"
#endif
#ifdef ENABLE_OGG
#include "output/vorbis/vorbis.h"
#endif
#ifdef ENABLE_DAEMON
#include "output/daemon/daemon.h"
#endif
#ifdef ENABLE_SNDFILE
#include "output/sndfile/fsndfile.h"
#endif
#ifdef ENABLE_AO
#include "output/ao/ao.h"
#endif
#include "file.h"
#include "output.h"
#include "volume.h"

/* Insert the correct function in the struct of outputs and
 * check the flags */
void
audio_init (void)
{
  audio *output = play->output;

  if (!output)
    fatal (_("Internal error."));


  while (output)
    {

      switch (output->audio_type)
	{
#ifdef ENABLE_STREAMING

#ifdef ENABLE_ICECAST
	case USE_ICECAST:
#endif

#ifdef ENABLE_ICECAST2
	case USE_ICECAST2:
#endif

#ifdef ENABLE_SHOUTCAST
	case USE_SHOUTCAST:
#endif
	  {
	    audio_data_streaming *streaming;

	    if (!(streaming = streaming_audio_device (output)))
	      fatal (_("Init output error."));

	    msg
	      (_
	       ("Encode Config:\n\tType %s\n\tRate %d\n\tQuality %d\n\tBitrate %d\n\tChannels %s"),
	       streaming->type == OUT_LAME ? "mp3lame" : "ogg/vorbis",
	       streaming->rate, streaming->quality, streaming->bitrate,
	       streaming->channels == OUT_MONO ? "mono" : "stereo");

	    msg
	      (_
	       ("Streaming Config:\n\tProtocol %s\n\tServer %s\n\tPort %d\n\tMount %s\n\tName %s\n\tGenre %s\n\tDescription %s\n\tUrl %s\n\t"),
	       streaming->connect == HTTP ? "http" : "https",
	       streaming->server, streaming->port, streaming->mount,
	       streaming->name ? streaming->name : "",
	       streaming->genre ? streaming->genre : "",
	       streaming->description ? streaming->description : "",
	       streaming->url ? streaming->url : "");

#ifdef ENABLE_ICECAST
	    if (output->audio_type == USE_ICECAST)
	      output->open = icecast_start;
	    else
#endif

#ifdef ENABLE_ICECAST2
	    if (output->audio_type == USE_ICECAST2)
	      output->open = icecast2_start;
	    else
#endif

#ifdef ENABLE_SHOUTCAST
	    if (output->audio_type == USE_SHOUTCAST)
	      output->open = shoutcast_start;

	    else
#endif

	      fatal
		(_
		 ("No support for any streaming protocol. Compile with Icecast, Icecast2 or Shoutcast."));

	    output->realtime = 1;

	    output->write = streaming_write;

	    output->close = streaming_quit;
	    output->check = NULL;
	    output->info = streaming_show;
	    output->data = (void *) streaming;
	    output->update = streaming_update;
	  }
	  break;
#endif

#ifdef ENABLE_LAME
	case USE_LAME:
	  {
	    audio_data_lame *lame;

	    if (!(lame = lame_audio_device (output)))
	      fatal (_("Init output error."));

	    output->realtime = 0;

	    output->open = lame_start;
	    output->write = lame_pre_write;
	    output->close = lame_pre_close;
	    output->check = NULL;
	    output->info = lame_info;
	    output->data = (void *) lame;
	    output->update = NULL;
	  }
	  break;
#endif

#ifdef ENABLE_OGG
	case USE_OGG:
	  {
	    audio_data_vorbis *vorbis;

	    if (!(vorbis = vorbis_audio_device (output)))
	      fatal (_("Init output error."));

	    output->realtime = 0;

	    output->open = vorbis_start;
	    output->write = vorbis_pre_write;
	    output->close = vorbis_pre_close;
	    output->check = NULL;
	    output->info = vorbis_show;
	    output->data = (void *) vorbis;
	    output->update = NULL;
	  }
	  break;
#endif

#ifdef ENABLE_DAEMON
	case USE_DAEMON:
	  {
	    output->realtime = 0;

	    output->open = daemon_output_init;
	    output->write = daemon_output_write;
	    output->close = daemon_output_close;
	    output->check = NULL;
	    output->info = daemon_output_show;
	    output->data = NULL;
	    output->update = NULL;
	  }
	  break;
#endif

#ifdef ENABLE_SNDFILE
	case USE_WAV:
	case USE_AIFF:
	case USE_AU:
	case USE_RAW:
	case USE_PAF:
	case USE_SVX:
	case USE_NIST:
	case USE_VOC:
	case USE_IRCAM:
	case USE_W64:
	case USE_MAT4:
	case USE_MAT5:
	case USE_PVF:
	case USE_XI:
	case USE_HTK:
#ifdef USE_MIDI
	case USE_MIDI:
#endif
#ifdef USE_AVR
	case USE_AVR:
#endif
#ifdef USE_WAVEX
	case USE_WAVEX:
#endif
	  {
	    audio_data_sndfile *sndfile;

	    if (!
		(sndfile =
		 (audio_data_sndfile *) malloc (sizeof (audio_data_sndfile))))
	      fatal (_("Init output error."));

	    output->realtime = 0;

	    output->open = sndfile_init;
	    output->write = sndfile_write;
	    output->close = sndfile_quit;
	    output->check = NULL;
	    output->info = sndfile_show;
	    output->data = (void *) sndfile;
	    output->update = NULL;
	  }
	  break;
#endif
	default:

#ifdef ENABLE_AO
	  output->realtime = 1;

	  output->open = ao_init;
	  output->write = ao_write;
	  output->close = ao_quit;
	  output->check = ao_check;
	  output->info = ao_show;
	  output->data = NULL;
          output->update = NULL;
#else
	  fatal (_("No support for libao. Compile with ao."));
#endif
	}

      /* Here: i run a separated thread for this ouput: */
      if (pthread_cond_init (&output->cond, NULL))
	fatal (_("Error: Cond init."));
      if (pthread_mutex_init (&output->mutex, NULL))
	fatal (_("Error: Mutex init."));
      if (pthread_create (&output->th, NULL, output_thread, output))
	fatal (_("Thread error."));

      output = output->next;
    }

}

/* This function check the cli output parameter and return the correct
 * type of output */
int
audio_type_select (char *str)
{
#ifdef ENABLE_AO
  if (!strcmp (str, "audio"))
    return USE_AUDIO;
  if (!strcmp (str, "null"))
    return USE_NULL;
  if (!strcmp (str, "stdout"))
    return USE_STDOUT;
  if (!strcmp (str, "alsa"))
    return USE_ALSA;
  if (!strcmp (str, "alsa09"))
    return USE_ALSA09;
  if (!strcmp (str, "oss"))
    return USE_OSS;
  if (!strcmp (str, "sun"))
    return USE_SUN;
  if (!strcmp (str, "esd"))
    return USE_ESD;
  if (!strcmp (str, "arts"))
    return USE_ARTS;
#endif

#ifdef ENABLE_SNDFILE
  if (!strcmp (str, "wav"))
    return USE_WAV;
  if (!strcmp (str, "aiff"))
    return USE_AIFF;
  if (!strcmp (str, "au"))
    return USE_AU;
  if (!strcmp (str, "raw"))
    return USE_RAW;
  if (!strcmp (str, "paf"))
    return USE_PAF;
  if (!strcmp (str, "svx"))
    return USE_SVX;
  if (!strcmp (str, "nist"))
    return USE_NIST;
  if (!strcmp (str, "voc"))
    return USE_VOC;
  if (!strcmp (str, "ircam"))
    return USE_IRCAM;
  if (!strcmp (str, "w64"))
    return USE_W64;
  if (!strcmp (str, "mat4"))
    return USE_MAT4;
  if (!strcmp (str, "mat5"))
    return USE_MAT5;
  if (!strcmp (str, "pvf"))
    return USE_PVF;
  if (!strcmp (str, "xi"))
    return USE_XI;
  if (!strcmp (str, "htk"))
    return USE_HTK;

#ifdef USE_MIDI
  if (!strcmp (str, "midi"))
    return USE_MIDI;
#endif

#ifdef USE_AVR
  if (!strcmp (str, "avr"))
    return USE_AVR;
#endif

#ifdef USE_WAVEX
  if (!strcmp (str, "wavex"))
    return USE_WAVEX;
#endif

#endif

#ifdef ENABLE_STREAMING

#ifdef ENABLE_ICECAST
  if (!strcmp (str, "icecast"))
    return USE_ICECAST;
#endif

#ifdef ENABLE_ICECAST2
  if (!strcmp (str, "icecast2"))
    return USE_ICECAST2;
#endif

#ifdef ENABLE_SHOUTCAST
  if (!strcmp (str, "shoutcast"))
    return USE_SHOUTCAST;
#endif

#ifdef ENABLE_LAME
  if (!strcmp (str, "lame"))
    return USE_LAME;
#endif

#ifdef ENABLE_OGG
  if (!strcmp (str, "vorbis") || !strcmp (str, "ogg"))
    return USE_OGG;
#endif

#ifdef ENABLE_DAEMON
  if (!strcmp (str, "daemon") || !strcmp (str, "sds"))
    return USE_DAEMON;
#endif

#endif

  msg (_("No output recognize: %s."), str);
  return 0;
}

/* Parse the parameter string and return the element.
 * If the string is: test=bye, this functions returns 
 * 'bye' */
char *
audio_parse_str (char *str, int *ch, int len)
{
  char tmp[256];
  char *output;
  int i = 0;

  while (*(str + *ch) != ',' && *ch < len && i < 256)
    {
      *(tmp + i) = *(str + *ch);
      (*ch)++;
      i++;
    }

  *(tmp + i) = 0;

  if (!(output = strdup (tmp)))
    fatal (_("Error: memory."));

  return output;
}

/* Like up, but of integer */
int
audio_parse_int (char *str, int *ch, int len)
{
  int output;

  output = atoi (str + *ch);

  while (*(str + *ch) != ',' && *(str + *ch) != ' ' && *ch < len)
    (*ch)++;

  return output;
}

/* Update the volumes */
void
audio_write (int channels, void *data, size_t length)
{
  signed short *ptr = (signed short *) data;
  int len;

  if (channels == 2)
    {
      len = length / 4;

      while (len--)
	{
	  *ptr++ = *ptr * (volume_l_get () / 100);
	  *ptr++ = *ptr * (volume_r_get () / 100);
	}
    }
  else if (channels == 1)
    {
      len = length / 2;

      while (len--)
	{
	  *ptr++ = *ptr * (volume_get () / 100);
	}
    }
}

/* EOF */
