// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// mathops.h
//
// Two classes - Vector3 and SolidAngle - to make 3-d math operations easier

#ifndef _VECTOR3_H
#define _VECTOR3_H

#include "../../lib/compat.h"
#include "mathdefs.h"
#include <cmath>
#include <cstdlib>
#include <climits>

// Euler angles for the conversions between celestial and galactic
//  coordinates:
//  A = 3.36473 +/- .00026 rad = RA of North Galactic Pole
//  D = 0.47370 +/- .00017 rad = Dec of North Galactic Pole
//  T = 0.57399 +/- .00017 rad = Galactic Longitude at the first point where
//                               Celestial and Galactic Equators cross
//
// Constants are given in sine and cosine form to decrease the number
//  of floating-point calculations.

const double COS_A = -0.97520799,
             SIN_A = -0.22129027,
             COS_D =  0.88988651,
             SIN_D =  0.45618198,
             COS_T =  0.83974115,
             SIN_T =  0.54298693;


class SolidAngle;  // forward declaration for below

// The Vector3 class defines the standard 3-D vector operations, plus
//  functions to convert into spherical coordinates and between
//  celestial and galactic systems.

class Vector3 { 
 private:
  double x, y, z; 
 public:
  Vector3() : x(0), y(0), z(0) { }
  Vector3(double X, double Y, double Z) : x(X), y(Y), z(Z) { }

  Vector3 &operator += (Vector3 v);
  Vector3 &operator -= (Vector3 v);
  Vector3 operator + (Vector3 v) const;
  Vector3 operator - (Vector3 v) const;
  Vector3 operator * (double c) const;
  friend Vector3 operator * (double c, Vector3 v);
  Vector3 operator / (double c) const;

  inline bool operator == (Vector3 v) const
    { return x == v.x && y == v.y && z == v.z; }
  inline bool operator != (Vector3 v) const
    { return ! (*this == v); }

  double magnitude() const;
  Vector3 unitvector() const;
  double dot(Vector3 v) const;
  Vector3 cross(Vector3 v) const;
  SolidAngle toSpherical() const;
  Vector3 toGalactic() const;
  Vector3 toCelestial() const;

  inline double getX() const { return x; }
  inline double getY() const { return y; }
  inline double getZ() const { return z; }
};

// The SolidAngle class is used to hold latitude and longitude in a convenient
//  structure, and allow conversion to Cartesian coordinates and between
//  celestial and galactic systems.

class SolidAngle {
 private:
  double theta /* latitude */,   // both in radians, of course
         phi   /* longitude */;
 public:
  SolidAngle() : theta(0), phi(0) { }
  SolidAngle(double Phi, double Theta) : theta(Theta), phi(Phi)
  {
    while (phi >= 2 * M_PI) phi -= (2 * M_PI);
    while (phi < 0.0)       phi += (2 * M_PI);
    if (theta > M_PI_2) theta = M_PI_2;
    if (theta < -M_PI_2) theta = -M_PI_2;
  }

  inline bool operator == (SolidAngle a) const
    { return theta == a.theta && phi == a.phi; }
  inline bool operator != (SolidAngle a) const
    { return ! (*this == a); }

  Vector3 toCartesian(double radius) const;
  SolidAngle toGalactic() const;
  SolidAngle toCelestial() const;

  inline double getTheta() const { return theta; }
  inline double getPhi()   const { return phi;   }
};


#endif // #define _VECTOR3_H
