// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  parse.cc
  Contains parsing functions allowing the starconvert utility to
  read in a specification file.
*/

#include "parse.h"
#include <iostream>

using std::endl;
using std::cerr;
using std::string;

void parse_config_line(string, parsedata *);
void parse_config_none(const StringList &, comments *, unsigned int);
void parse_config_coords(const StringList &, coordinates *, unsigned int);
void parse_config_charact(const StringList &, characteristics *, unsigned int);
void parse_config_systems(const StringList &, systems *, unsigned int);
void parse_config_names(const StringList &, namedata *, unsigned int);
void parse_config_subst(const StringList &, namedata *, unsigned int);

enum errortype { TOOMANYARGS, BADKEYWORD, BADOPTION };

void printerror(errortype type, const string & keyword, 
		const string & option, unsigned int lineno)
{
  switch (type) {
    case TOOMANYARGS:
      cerr << starstrings::ssprintf(_(
"Warning!  Too many arguments for keyword `%s' on line %d;\n\
ignoring extras."),
	keyword.c_str(), lineno) << endl;
      break;
    case BADKEYWORD:  
      {
        cerr << starstrings::ssprintf(_(
"Warning!  Unrecognized keyword `%s' on line %d; ignoring."),
	  keyword.c_str(), lineno) << endl;
	if (keyword.find(' ') < keyword.size())
	  cerr << _("Did you forget a semicolon?") << endl;
        break;
      }
    case BADOPTION:   
      {
        cerr << starstrings::ssprintf(_(
"Warning!  Unrecognized option `%s' to keyword `%s'\n\
on line %d; ignoring."),
	  option.c_str(), keyword.c_str(), lineno) << endl;
	if (keyword.find(' ') < keyword.size())
	  cerr << _("Did you forget a semicolon?") << endl;
	break;
      }
  }
  return;
}


// parse_config_file(): This is the only function called from main() in
//  the convert.cc file.  The rest in this file are only used internally.

void parse_config_file(std::istream &configfile, parsedata *data)
{
  // initialize everything with a default value.
  data->Comments.s.start = 0; data->Comments.s.len = 0;
  data->Coord.isCelestial = true;
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++) {
    data->Coord.s[i].start = 0; data->Coord.s[i].len = 0;
  }
  data->Charact.s = std::vector<substring>();
  data->Charact.distarray = std::vector<stardistance>();
  data->Charact.magarray = std::vector<magnitude>();
  data->Systems.comp.start = data->Systems.sep.start = 0;
  data->Systems.comp.len = data->Systems.sep.len = 0;
  data->Systems.isSeparationCommented = false;
  data->Systems.sep_prefixes = StringList();
  data->Names.names = std::vector<name>();
  data->Names.isSubstCaseSensitive = false;

  // now, go through each line of the config file.
  char temp[256];
  while (configfile.getline(temp, 256, '\n')) {
    temp[255] = 0;
    parse_config_line(string(temp), data);
  }

  // finally, check that the most important fields have been filled in.
  if (!(data->Coord.s[0].len		   /* ra hrs */
	&& data->Coord.s[3].len            /* declination sign */
	&& data->Coord.s[4].len            /* declination deg */
	&& data->Charact.distarray.size()) /* # of distance options */) {
    
    /* TRANSLATORS: each line of this message should begin with "***" */
    cerr << _("*** Not enough information in config file about how to obtain\n\
*** star coordinates.")
	 << endl << "*** " << _("Exiting.") << endl;
    exit(EXIT_FAILURE);
  }
  if (! data->Names.names.size()) {
    /* TRANSLATORS: each line of this message should begin with "***" */
    cerr << _("*** Not enough information in config file about how to obtain\n\
*** star names.")
	 << endl << "*** " << _("Exiting.") << endl;
    exit(EXIT_FAILURE);
  }
  if (! data->Charact.distarray.size()) {
    /* TRANSLATORS: each line of this message should begin with "***" */
    cerr << _("*** No information given in config file about how to obtain\n\
*** star distances.")
	 << endl << "*** " << _("Exiting.") << endl;
    exit(EXIT_FAILURE);
  }
  if (data->Charact.magarray.size() != data->Charact.distarray.size()) {
    /* TRANSLATORS: each line of this message should begin with "***" */
    cerr << _("*** Distance and magnitude measurement options do not occur\n\
*** in pairs in config file.")
	 << endl << "*** " << _("Exiting.") << endl;
    exit(EXIT_FAILURE);
  }
  for (unsigned int i = 0; i < data->Charact.distarray.size(); i++) {
    if (data->Charact.distarray[i].type == SPECCLASS
	&& data->Charact.magarray[i].type == ABSOLUTE) {
      cerr << _(
      /* TRANSLATORS: each line of this message should begin with "***".
         The quoted phrases "magnitude; absolute" and "distance; specclass"
	 are keywords and should not be translated. */
"*** \"magnitude; absolute\" and \"distance; specclass\" is not a\n\
*** valid pair of measurements.")
	   << endl << "*** " << _("Exiting.") << endl;
      exit(EXIT_FAILURE);
    }
  }
  if (! data->Charact.s.size()) {
    for (unsigned int i = 0; i < data->Charact.distarray.size(); i++) {
      if (data->Charact.distarray[i].type == SPECCLASS) {
	cerr << _(
	/* TRANSLATORS: each line of this message should begin with "***" */
"*** No information given in config file about how to obtain\n\
*** spectral classes, but a rule for obtaining distances depends on it.")
	     << endl << "*** " << _("Exiting.") << endl;
	exit(EXIT_FAILURE);
      }
    }
    cerr << _(
"Warning!  No information given in config file about how to obtain\n\
spectral classes.") << endl;
  }
  return;
}


// parse_config_line(): Strips out comments and blank lines, then passes
//  whatever's left to helper functions.

void parse_config_line(string line, parsedata *data)
{
  static parsetype currenttype = NONE;
  static unsigned int lineno = 0;

  // increment line number
  lineno++;
  
  // first, destroy #'ed-out comments in the config file line.
  if (line.find('#') < line.size())
    line = line.substr(0, line.find('#'));
  if (starstrings::isempty(line)) return;

  StringList strlist = StringList(line, ';');
  strlist.stripspace();
  strlist.utf8ize();

  // check for type separators
  if (starstrings::lowercase(strlist[0]) == "[coordinates]")
    { currenttype = COORDS; return; }
  else if (starstrings::lowercase(strlist[0]) == "[characteristics]")
    { currenttype = CHARACT; return; }
  else if (starstrings::lowercase(strlist[0]) == "[systems]")
    { currenttype = SYSTEMS; return; }
  else if (starstrings::lowercase(strlist[0]) == "[names]")
    { currenttype = NAMES; return; }
  else if (starstrings::lowercase(strlist[0]) == "[substitutions]")
    { currenttype = SUBST; return; }

  // check for keywords and parse with helper functions.
  switch (currenttype) {
  case NONE:    parse_config_none(strlist, &(data->Comments), lineno); break;
  case COORDS:  parse_config_coords(strlist, &(data->Coord), lineno); break;
  case CHARACT: parse_config_charact(strlist, &(data->Charact), lineno); break;
  case SYSTEMS: parse_config_systems(strlist, &(data->Systems), lineno); break;
  case NAMES:   parse_config_names(strlist, &(data->Names), lineno); break;
  case SUBST:   parse_config_subst(strlist, &(data->Names), lineno); break;
  }
  return;
}


// Following are all the helper functions, each of which parses a set
//  of keywords.

void parse_config_none(const StringList &strlist,
		       comments *com, unsigned int lineno)
{
  if (starstrings::case_compare(strlist[0], "comments")) {
    if (strlist.size() > 1) {
      com->s.start = starmath::atoi(strlist[1]) - 1;
      com->s.len = (strlist.size() > 2)
	? starmath::atoi(strlist[2]) - com->s.start : 1;
    }
    if (strlist.size() > 3)
      printerror(TOOMANYARGS, "comments", 0, lineno);
  }
  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_coords(const StringList &strlist, coordinates *coor,
			 unsigned int lineno)
{
  if (starstrings::case_compare(strlist[0], "type")) {
    if (strlist.size() > 1) {
      if (starstrings::case_compare(strlist[1], "galactic"))
	coor->isCelestial = false;
      else if (! starstrings::case_compare(strlist[1], "celestial"))
	printerror(BADOPTION, "type", strlist[1], lineno);
      if (strlist.size() > 2)
	printerror(TOOMANYARGS, "type", 0, lineno);
    }
  }

  else {
    if (strlist.size() > 1) {
      unsigned int i, j;
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, strlist[0], 0, lineno);
      for (i = 0; i < NUM_COORD_OPTIONS; i++)
	for (j = 0; j < 2; j++)
	  if (starstrings::case_compare(strlist[0], coordoptions[i][j])) {
	    coor->s[i].start = starmath::atoi(strlist[1]) - 1;
	    coor->s[i].len = (strlist.size() > 2)
	      ? starmath::atoi(strlist[2]) - coor->s[i].start : 1;
	    return;
	  }
      printerror(BADKEYWORD, strlist[0], 0, lineno);
    }
  }
  return;
}


void parse_config_charact(const StringList &strlist, characteristics * ch,
			  unsigned int lineno)
{
  if (starstrings::case_compare(strlist[0], "specclass")) {
    if (strlist.size() > 1) {
      substring s;
      s.start = starmath::atoi(strlist[1]) - 1;
      s.len = (strlist.size() > 2) ?
	starmath::atoi(strlist[2]) - starmath::atoi(strlist[1]) + 1 : 1;
      ch->s.push_back(s);
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, "specclass", 0, lineno);
    }
  }
   
  else if (starstrings::case_compare(strlist[0], "distance")) {
    if (strlist.size() > 1) {
      unsigned int i;
      stardistance d;
      for (i = 0; i < NUM_DISTANCE_UNITS; i++)
	if (starstrings::case_compare(strlist[1], distanceunits[i])) break;
      if (i == NUM_DISTANCE_UNITS) {
	printerror(BADOPTION, "distance", strlist[1], lineno);
	return;
      }
      else if (starstrings::case_compare(strlist[1], "specclass")) {
	d.type = SPECCLASS;
	d.s.start = d.s.len = 0;
	ch->distarray.push_back(d);
	if (strlist.size() > 2)
	  printerror(TOOMANYARGS, "distance", 0, lineno);
      }
      else if (strlist.size() > 2) {
	d.type = eunittype(i);
	d.s.start = starmath::atoi(strlist[2]) - 1;
	d.s.len = (strlist.size() > 3) ?
	  starmath::atoi(strlist[3]) - d.s.start : 1;
	if (d.type == ARCSEC || d.type == MILLIARCSEC) {
	  if (strlist.size() >= 7) { // we're using max_error specifier
	    d.err.start = starmath::atoi(strlist[4]) - 1;
	    d.err.len = starmath::atoi(strlist[5]) - d.err.start;
	    d.maxerror = starmath::atof(strlist[6]);
	    d.minparallax = 0.0;
	  }
	  else {
	    d.err.start = d.err.len = 0;

	    if (strlist.size() >= 5) { // using min_parallax specifier
	      d.minparallax = starmath::atof(strlist[4]);
	    }
	    else { // plain vanilla parallax
	      d.minparallax = 0.0;
	    }
	  }
	  if (strlist.size() > 7 || strlist.size() == 6)
	    printerror(TOOMANYARGS, "distance", 0, lineno);
	}
	else {
	  d.minparallax = 0.0;
	  if (strlist.size() > 4)
	    printerror(TOOMANYARGS, "distance", 0, lineno);
	}
	ch->distarray.push_back(d);
      }
    }
  }

  else if (starstrings::case_compare(strlist[0], "magnitude")) {
    if (strlist.size() > 1) {
      magnitude m;
      if (starstrings::case_compare(strlist[1], "absolute"))
	m.type = ABSOLUTE;
      else if (starstrings::case_compare(strlist[1], "visual"))
	m.type = VISUAL;
      else {
	printerror(BADOPTION, "magnitude", strlist[1], lineno);
	return;
      }
      if (strlist.size() > 2) {
	m.s.start = starmath::atoi(strlist[2]) - 1;
	m.s.len = (strlist.size() > 3) ?
	  starmath::atoi(strlist[3]) - m.s.start : 1;
	ch->magarray.push_back(m);
	if (strlist.size() > 4)
	  printerror(TOOMANYARGS, "magnitude", 0, lineno);
      }
    }
  }

  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_systems(const StringList & strlist, systems *sys, 
			  unsigned int lineno)
{
  if (starstrings::case_compare(strlist[0], "component")) {
    if (strlist.size() > 1) {
      sys->comp.start = starmath::atoi(strlist[1]) - 1;
      sys->comp.len = (strlist.size() > 2)
	? starmath::atoi(strlist[2]) - sys->comp.len : 1;
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, "component", 0, lineno);
    }
  }

  else if (starstrings::case_compare(strlist[0], "separation")) {
    if (strlist.size() > 1) {
      if (starstrings::case_compare(strlist[1], "comments")) {
	sys->isSeparationCommented = true;
	for (unsigned int i = 2; i < strlist.size(); i++)
	  sys->sep_prefixes.push_back(strlist[i]);
      }
      else {
        sys->sep.start = starmath::atoi(strlist[1]) - 1;
	sys->sep.len = (strlist.size() > 2)
	  ? starmath::atoi(strlist[2]) - sys->sep.len : 1;
	if (strlist.size() > 3)
	  printerror(TOOMANYARGS, "separation", 0, lineno);
      }
    }
  }

  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_names(const StringList & strlist,
			namedata *nmd, unsigned int lineno)
{
  unsigned int i = 0;
  for (i = 0; i < NUM_NAME_TYPES; i++)
    if (starstrings::case_compare(strlist[0], nametypes[i]))
      break;
  if (i == NUM_NAME_TYPES) {
    printerror(BADKEYWORD, strlist[0], 0, lineno);
    return;
  }
  
  if (strlist.size() > 1) {
    name n;
    n.name_prefixes = StringList();
    if (strlist[0] == string("other")) {
      n.type = OTHER;
      if (strlist[1] == string("comments")) {
	n.s.start = n.s.len = 0;
	n.isNameCommented = true;
	for (unsigned int j = 2; j < strlist.size(); j++)
	  n.name_prefixes.push_back(strlist[j]);
      }
      else {
	n.s.start = starmath::atoi(strlist[1]) - 1;
	n.s.len = (strlist.size() > 2) 
	  ? starmath::atoi(strlist[2]) - n.s.start : 1;
	n.isNameCommented = false;
	for (unsigned int j = 3; j < strlist.size(); j++)
	  n.name_prefixes.push_back(strlist[j]);
      }
    }

    else {
      n.type = enametype(i);
      if (strlist[1] == string("comments")) {
	n.s.start = n.s.len = 0;
	n.isNameCommented = true;
	if (strlist.size() > 2)
	  printerror(TOOMANYARGS, strlist[0], 0, lineno);
      }
      else {
	n.s.start = starmath::atoi(strlist[1]) - 1;
	n.s.len = (strlist.size() > 2) 
	  ? starmath::atoi(strlist[2]) - n.s.start : 1;
	n.isNameCommented = false;
	if (strlist.size() > 3)
	  printerror(TOOMANYARGS, strlist[0], 0, lineno);
      }
    }
    nmd->names.push_back(n);
  }
  return;
}


void parse_config_subst(const StringList &strlist,
			namedata *nmd, unsigned int lineno)
{
  static int count = 0;

  count++;
  if (count == 1 && starstrings::case_compare(strlist[0], "case-sensitive")) {
    nmd->isSubstCaseSensitive = true;
    if (strlist.size() > 1)
      printerror(TOOMANYARGS, "case-sensitive", 0, lineno);
  }
  else if (count == 1 && starstrings::case_compare(strlist[0], "case-insensitive")) {
    if (strlist.size() > 1)
      printerror(TOOMANYARGS, "case-insensitive", 0, lineno);
  }
  else if (count == 1 && strlist.size() == 1)
    printerror(BADKEYWORD, strlist[0], 0, lineno);
  else if (strlist.size() == 2)
    nmd->substs.push_back(substitution(strlist[0], strlist[1], 0));
  else if (strlist.size() == 3) {
    int posn = starmath::atoi(strlist[2]) - 1;
    if (posn < 0)
      cerr << starstrings::ssprintf(_(
"Warning!  Third item in a substitution rule must be greater than 0,\n\
but is `%s' on line %d; ignoring."), strlist[2].c_str(), lineno) << endl;
    else
      nmd->substs.push_back(substitution(strlist[0], strlist[1], posn));
  }
  else
    cerr << starstrings::ssprintf(_(
"Warning!  Line %d is not a valid substitution rule; ignoring."), lineno)
	 << endl;

  return;
}

