// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  filedialogs.cc
  This file contains code for the dialog boxes in the StarPlot File menu,
  and also code for reading to / writing from the .starplotrc settings file.
*/

#include <gtk/gtk.h>
#include "starplot.h"

using std::endl;
using std::string;

bool read_params(const string & filename);
bool write_params(const string & filename, const string & header);
void my_gtk_file_select (void (*)(GtkWidget *, GtkFileSelection *),
			 const string &);

void starplot_quit()
{
  delete globals::program_viewer;
  delete globals::chart_stararray;
  write_rc_file();
  gtk_main_quit();
  return;
}


/* File selection dialogues */
/* These cover selection of data / parameter files to load / save, and also
 *  the selection of the help browser program. */

/* This opens a new file */
void file_ok_sel( GtkWidget        *w,
		  GtkFileSelection *fs )
{
  globals::chart_rules.ChartFileNames = StringList(1,
		  gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs)));
  redraw_all(FILE_CHANGE);
}

/* This merges the selected file with the other(s) already open */
void merge_ok_sel( GtkWidget        *w,
		   GtkFileSelection *fs )
{
  string filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
  				
  // make sure the file we want to merge isn't already open
  //  (not foolproof due to the possibility of symlinks, etc., but better
  //  than nothing)
  iterate (StringList, globals::chart_rules.ChartFileNames, name_ptr)
    if (*name_ptr == filename) return;

  globals::chart_rules.ChartFileNames.push_back(filename);
  redraw_all(FILE_CHANGE);
}

/* This opens a new set of chart parameters */
void param_open_sel( GtkWidget        *w,
		     GtkFileSelection *fs )
{
  string filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
  if (read_params(filename))
    redraw_all(FILE_CHANGE);
}

/* This saves the current chart parameters to a file */
void param_save_sel( GtkWidget        *w,
		     GtkFileSelection *fs )
{
  string filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
  
  if (!write_params(filename, string("# ") + _("StarPlot Parameter File\n")))
    my_gtk_error (0, starstrings::ssprintf(
	  _("Sorry, couldn't create file \"%s\"."), filename.c_str()).c_str());
}

/* This saves the current chart as a PNG image file */
void file_save_sel( GtkWidget *w, GtkFileSelection *fs )
{
  char * filename = new char[std::strlen(gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs))) + 1];
  strcpy(filename, gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
  
  GError *error = NULL;
  GdkPixbuf *buffer =
    gdk_pixbuf_get_from_drawable (NULL, globals::program_pixmap,
      gdk_colormap_get_system(), 0, 0, 0, 0, -1, -1);
  
  if (!buffer)
    my_gtk_error (0,
	_("Sorry, an error occurred\ncreating the buffer."));
  
  else if (!gdk_pixbuf_save (buffer, filename, "png", &error, 0))
    my_gtk_error (0, starstrings::ssprintf(
	  _("Sorry, couldn't create file \"%s\"."), filename).c_str());

  delete [] filename;
}

/* This selects a help file browser */
void help_ok_sel( GtkWidget        *w,
		  GtkFileSelection *fs )
{
  globals::program_help_browser =
    gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs));
}

/* This selects a help file browser and also opens the help documentation */
void help_open_sel( GtkWidget        *w,
		    GtkFileSelection *fs )
{
  help_ok_sel(w, fs);
  helpdocs();
}


/* We need several functions for the menu generator, plus one for the case
 *  where a help browser must be selected before opening help docs,
 *  but they all call the same interface, just choose the correct file
 *  selection function */
void file_open()
{
  my_gtk_file_select(file_ok_sel,
      _("StarPlot: Select data file to open"));
}
void file_merge()  
{
  my_gtk_file_select(merge_ok_sel,
      _("StarPlot: Select data file to merge"));
}
void params_open()
{
  my_gtk_file_select(param_open_sel,
      _("StarPlot: Select parameter file to use"));
}
void params_save()
{
  my_gtk_file_select(param_save_sel,
      _("StarPlot: Save chart parameters as:"));
}
void file_save()
{
  my_gtk_file_select(file_save_sel,
      _("StarPlot: Save chart as PNG image"));
}
void help_select() 
{
  my_gtk_file_select(help_ok_sel,
      _("StarPlot: Select help browser program"));
}
void help_select_and_open()
{
  my_gtk_file_select(help_open_sel,
      _("StarPlot: Select help browser program"));
}


/* The interface which creates the dialog */
void my_gtk_file_select 
(void (*select_function)(GtkWidget *, GtkFileSelection *), const string &title)
{ 
  GtkWidget *filew;

  filew = gtk_file_selection_new (title.c_str());
  gtk_window_set_modal (GTK_WINDOW (filew), true);
  g_signal_connect (G_OBJECT (filew), "destroy",
		      G_CALLBACK (gtk_widget_destroy), 
		      GTK_WIDGET (filew));

  /* Connect the ok_button to the appropriate file selection function */
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (filew)->ok_button),
		      "clicked", G_CALLBACK (select_function), filew);
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION
					 (filew)->ok_button),
			     "clicked", G_CALLBACK (gtk_widget_destroy),
			     G_OBJECT (filew));

  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION
					 (filew)->cancel_button),
			     "clicked", G_CALLBACK (gtk_widget_destroy),
			     G_OBJECT (filew));

  /* Set the default filename */
  if (select_function == help_ok_sel || select_function == help_open_sel) {
    if (starstrings::isempty(globals::program_help_browser))
      gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew),
				       "/usr/bin/");
    else
      gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew),
				       globals::program_help_browser.c_str());
  }
  else if (select_function == param_open_sel ||
	   select_function == param_save_sel ||
	   select_function == file_save_sel)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew), "~/");
  else
    gtk_file_selection_set_filename (GTK_FILE_SELECTION(filew), DATADIR "/");

  gtk_widget_show(filew);
}



// read_params() : Read a StarPlot parameters file.  Note that this function
//  will have problems if any of the filenames stored in ~/.starplotrc contain
//  trailing spaces or any of these characters:  # = ( ) [ ] { }
//  I think we can live with this limitation.

bool read_params(const string &filename)
{
  std::ifstream paramstream;
  paramstream.open(filename.c_str());
  if (!paramstream.good()) return false;

  int files = 0;
  char line[1000];
  while (paramstream.getline(line, 1000)) {
    // delete comments and munch all kinds of parentheses (which are used
    //  only to make the RC file entries look nicer)
    for (unsigned int i = 0; i < strlen(line); i++) {
      if (line[i] == '#') line[i] = 0;
      if (line[i] == '(' || line[i] == ')' || line[i] == '[' || line[i] == ']'
	  || line[i] == '{' || line[i] == '}')
	line[i] = ' ';
    }
    StringList assign(line, '=');
    if (assign.size() < 2) continue;
    assign.stripspace();
    
    if (assign[0] == string("DataFile")) {
      if (files == 0)
	globals::chart_rules.ChartFileNames = StringList(1, assign[1]);
      else
	globals::chart_rules.ChartFileNames.push_back(assign[1]);
      files++;
    }
    else if (assign[0] == string("Location")) {
      StringList coords(assign[1], ',');
      if (coords.size() != 3) continue;
      starstrings::stripspace(coords[2]);
      globals::chart_rules.ChartLocation = Vector3(starmath::atof(coords[0]),
						   starmath::atof(coords[1]),
						   starmath::atof(coords[2]));
    }
    else if (assign[0] == string("Orientation")) {
      StringList orient(assign[1], ',');
      if (orient.size() != 2) continue;
      orient.stripspace();
      globals::chart_rules.ChartOrientation =
	SolidAngle(starmath::atof(orient[0]),
		   starmath::atof(orient[1]));
    }
    else if (assign[0] == string("Radius"))
      globals::chart_rules.ChartRadius = starmath::atof(assign[1]);
    else if (assign[0] == string("MagLimits")) {
      StringList maglim(assign[1], ',');
      if (maglim.size() != 2) continue;
      maglim.stripspace();
      globals::chart_rules.ChartBrightestMagnitude = starmath::atof(maglim[0]);
      globals::chart_rules.ChartDimmestMagnitude = starmath::atof(maglim[1]);
    }
    else if (assign[0] == string("HRMagLimits")) {
      StringList maglim(assign[1], ',');
      if (maglim.size() != 2) continue;
      maglim.stripspace();
      globals::hr_mag_bright = starmath::atof(maglim[0]);
      globals::hr_mag_dim = starmath::atof(maglim[1]);
    }
    else if (assign[0] == string("AutosetDimMag"))
      globals::chart_rules.ChartAutosetDimmestMagnitude = static_cast<bool>
	      (starmath::atoi(assign[1]));
    else if (assign[0] == string("HideCompanions"))
      globals::chart_rules.ChartHideCompanions = static_cast<bool>
	      (starmath::atoi(assign[1]));
    else if (assign[0] == string("SpecClassDisplay")) {
      StringList classes(assign[1], ',');
      if (classes.size() != 10) continue;
      classes.stripspace();
      for (unsigned int i = 0; i < 10; i++)
	globals::chart_rules.StarClasses[i] = static_cast<bool>
	  (starmath::atoi(classes[i]));
    }
    else if (assign[0] == string("CelestialCoords"))
      set_coord_item(static_cast<bool>(starmath::atoi(assign[1])), false);
    else if (assign[0] == string("StarDiameters"))
      set_diameter_item(static_cast<star_diameter_t>(starmath::atoi(assign[1])),
		        false);
    else if (assign[0] == string("StarLabels"))
      set_label_item(static_cast<star_label_t>(starmath::atoi(assign[1])),
			false);
    else if (assign[0] == string("StarBars"))
      set_toggle_item(&globals::chart_rules.StarBars,
		      static_cast<bool>(starmath::atoi(assign[1])),
		      TOGGLE_BARS, false);
    else if (assign[0] == string("ChartGrid"))
      set_toggle_item(&globals::chart_rules.ChartGrid,
		      static_cast<bool>(starmath::atoi(assign[1])),
		      TOGGLE_GRID, false);
    else if (assign[0] == string("ChartLegend"))
      set_toggle_item(&globals::chart_rules.ChartLegend,
		      static_cast<bool>(starmath::atoi(assign[1])),
		      TOGGLE_LEGEND, false);
#ifndef BROWSER
    else if (assign[0] == string("HelpBrowser"))
      globals::program_help_browser = assign[1];
#endif
    else if (assign[0] == string("FontDescription"))
      globals::display_fontname = assign[1];
  }

  return true;
}


/* write_params() : Save the current StarPlot parameters to a file. */

bool write_params(const string & filename, const string & header)
{
  std::ofstream paramstream;
  paramstream.open(filename.c_str());
  if (!paramstream.good()) return false;

  paramstream << header << endl;

  iterate (StringList, globals::chart_rules.ChartFileNames, name_ptr)
    paramstream << "DataFile = " << *name_ptr << endl << endl;

  paramstream << "# " << _("(X, Y, Z) of chart center, in light-years") << endl
	      << "Location = ( "
	      << globals::chart_rules.ChartLocation.getX() << ", "
	      << globals::chart_rules.ChartLocation.getY() << ", "
	      << globals::chart_rules.ChartLocation.getZ() << " )" << endl;
  paramstream << "Radius = "
	      << globals::chart_rules.ChartRadius << " # " << _("light-years")
	      << endl << endl;

  paramstream << "# " <<
    /* TRANSLATORS: RA stands for "Right Ascension"; Dec for "Declination" */
    _("(RA, Dec) of orientation vector, in radians")
	      << endl
	      << "Orientation = ( "
	      << globals::chart_rules.ChartOrientation.getPhi()
	      << ", " << globals::chart_rules.ChartOrientation.getTheta()
	      << " )" << endl << endl;
	      
  paramstream << "MagLimits = [ "
	      << globals::chart_rules.ChartBrightestMagnitude << ", "
	      << globals::chart_rules.ChartDimmestMagnitude << " ]" << endl;
  paramstream << "HRMagLimits = [ "
	      << globals::hr_mag_bright << ", " << globals::hr_mag_dim << " ]"
	      << endl;
  paramstream << "AutosetDimMag = "
	      << globals::chart_rules.ChartAutosetDimmestMagnitude
	      << endl;
  paramstream << "HideCompanions = "
	      << globals::chart_rules.ChartHideCompanions << endl << endl;

  paramstream << "# SpecClassDisplay: { O, B, A, F, G, K, M, "
	      /* TRANSLATORS: "dwarf" refers to a class of star. */
	      << _("dwarf") << ", " <<  _("unclassified")
	      << ", " <<  _("non-stellar") << " }"
	      << endl;
  paramstream << "SpecClassDisplay = { ";
  for (unsigned int i = 0; i < 9; i ++)
    paramstream << globals::chart_rules.StarClasses[i] << ", ";
  paramstream << globals::chart_rules.StarClasses[9] << " }" << endl << endl;

  paramstream << "CelestialCoords = "
	      << globals::chart_rules.CelestialCoords << endl;
  paramstream << "StarDiameters = "
	      << globals::chart_rules.StarDiameters << endl;
  paramstream << "StarLabels = "
	      << globals::chart_rules.StarLabels << endl;
  paramstream << "StarBars = " << globals::chart_rules.StarBars << endl;
  paramstream << "ChartGrid = " << globals::chart_rules.ChartGrid << endl;
  paramstream << "ChartLegend = " << globals::chart_rules.ChartLegend << endl;
#ifndef BROWSER
  if (! starstrings::isempty(globals::program_help_browser))
    paramstream << "HelpBrowser = " << globals::program_help_browser << endl;
#endif
  paramstream << "FontDescription = " << globals::display_fontname << endl;

  paramstream << endl;
  return true;
}



// read_rc_file() : Called when the program begins.  

bool read_rc_file()
{
  string homedir = std::getenv("HOME");
  return read_params(homedir + "/.starplotrc");
}


// write_rc_file() : Called as the program is exiting.

bool write_rc_file()
{
  string homedir = std::getenv("HOME");
  string header = string("# ") + _("StarPlot user settings file") + "\n\n" +
    /* TRANSLATORS: Prefix each line of this text with a '#' character. */ _(
"# Read by StarPlot when starting and rewritten automatically on\n\
# program exit.  If you edit this file, note that distances are in\n\
# light-years and angles are in radians.\n\n");
  return write_params(homedir + "/.starplotrc", header);
}

