
/* $Id: main.c,v 400.1 2002/07/25 08:43:16 sgifford Exp $ */

#include "config.h"

#include "startalk.h"
#include "phonebook.h"
#include "stdata.h"
#include "stsettings.h"
#include "filedata.h"
#include "filesettings.h"
#include "stdebug.h"
#include "alarm.h"
#include "getopt.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>

#define MODE_DEFAULT   0
#define MODE_READ      1
#define MODE_WRITE     2
#define MODE_CLEAR     3
#define MODE_TEST      4
#define MODE_PARSETEST 5

int testing;

#define DEFAULT_PHONEDEV "/dev/pcsphone"
char *dev = DEFAULT_PHONEDEV;
int serialfd = -1;

#ifdef STTY_SERIALSETUP
#define DEFAULT_STTY "raw crtscts -echo ispeed 19200 ospeed 19200"
char *stty = DEFAULT_STTY;
#endif

/*#define DEFAULT_MODEMINIT "ATE1V1Q0"*/
#define DEFAULT_MODEMINIT "AT S7=45 S0=0 L1 V1 X4 &c1 E1 Q0"
char *modeminit = DEFAULT_MODEMINIT;

#define DEFAULT_TIMEOUT 10
int startalk_timeout = DEFAULT_TIMEOUT;

#define INITIALIZE_TRIES 2

void
usage(char *progname, const char *format, ...)
{
  va_list a;
  
  if (format)
  {
    fprintf(stderr,"%s: ",progname);
    va_start(a, format);
    vfprintf(stderr, format, a);
    va_end(a);
  }
  fprintf(stderr,"Usage: %s [-c] [-r|-w|-t] [-n entrynum(s)] [-v] [-d debuglevel] [-p phonetty]\n",progname);
  fprintf(stderr,"\
\t-c: Clear out phonebook entries.  USE WITH CARE!!\n\
\t    When used with the -w option, will clear out all entries\n\
\t    which are not in the file read in.\n\
\t    When used with the -n option, will clear out the entries\n\
\t    in the specified range.\n\
\t-r: Read entire phonebook from phone and dump to stdout\n\
\t-w: Read phonebook from stdin and write to phone\n"
#ifdef STARTALK_SETTINGS          
"\t-s: Read/write settings instead of phonebook\n"
#endif
"\t-t: Test the phone and StarTalk\n\
");
  fprintf(stderr,"\
\t-T: Set default timeout (in seconds) for communication with the phone.\n\
\t    Default is %d; use -1 for no timeout.\n\
",DEFAULT_TIMEOUT);
  fprintf(stderr,"\
\t-n: Specify to perform the read, write, or clear on\n\
\t    entries specified by n.\n\
\t    You can use commas to specify multiple entry numbers, and can\n\
\t    use the dash character to specify a range of numbers.\n\
\t    For example, '-n 1-9,25-30,99'\n\
");
  fprintf(stderr,"\
\t-e: Display empty phonebook entries\n\
");
  fprintf(stderr,"\
\t-p: Specify the tty/device to be used to talk to the phone.\n\
\t    Default is '%s'.\n\
",DEFAULT_PHONEDEV);
#ifdef STTY_SERIALSETUP
  fprintf(stderr,"\
\t-S: Specify arguments to stty to set up the tty\n\
\t    Default is '%s'.\n\
",DEFAULT_STTY);
#endif
  fprintf(stderr,"\
\t-i: Specify init string sent to phone\n\
\t    Default is '%s'.\n\
",DEFAULT_MODEMINIT);
  fprintf(stderr,"\
\t-v: Verbose mode (same as -d 1)\n\
\t-d: Debug level\n\
\t    1: Verbose\n\
\t    2: Downright noisy\n\
\t    3: Full packet dumps\n\
\t    4: Tediously report on the inner workings of StarTalk.\n\
\t-h: Help (display this message)\n\
");
  fprintf(stderr,"\
This is startalk version %s\n\
",VERSION);
  exit(2);
}

struct pbset {
  int count;
  unsigned char num[MAX_ENTRIES];
};


int
add_pos(struct pbset *set, int num)
{
  if ( (num < 1) || (num > MAX_ENTRIES) )
  {
    return errorf("Phonebook entry number out of range");
  }
  if (set->count >= MAX_ENTRIES)
  {
    return errorf("Too many phonebook entry numbers given (how?)");
  }
  set->num[set->count++]=num;
  return 0;
}

/* This code is mostly derived from a patch I did
 * to strobe.
 */
int
process_set(struct pbset *set, const char *changes)
{
  char *s;
  char *pos, *pstart, *minus;
  int i;
  int firstnum,lastnum;
  int stopnow = 0;

  set->count=0;

  s=strdup(changes);
  if (!s)
    die("Malloc error (strdup)");

  pstart=pos=s;

  while(!stopnow)
  {
    switch(*pos)
    {
      case '\0':
        stopnow=1;
        /* FALL THROUGH */
      case ',':
      case ':':
        *pos=0;
        if ( (minus=strchr(pstart,'-')))
        {
          *minus=0;
          firstnum=strlen(pstart) == 0
            ? 1
            : atoi(pstart);
          lastnum=strlen(minus+1) == 0
            ? MAX_ENTRIES
            : atoi(minus+1);
          for(i=firstnum;i<=lastnum;i++)
            if (add_pos(set,i) != 0)
              stopnow=2;
        }
        else
        {
          if (add_pos(set,atoi(pstart)) != 0)
            stopnow=2;
        }
        pstart=pos+1;
        break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
      case '-':
        break;
      default:
        errorf("Invalid character specifying phonebook entry number(s)");
        stopnow=2;
    }
    pos++;
  }

  free(s);
  if (stopnow==2)
    return 1;
  else
    return 0;
}
    
void
signal_cleanup(int signum)
{
  /* Set the signal back to default, so if they hit CTRL-C,
   * and we hang while cleaning up, they can hit CTRL-C again
   * and actually kill the program.
   */
  
  signal(signum,SIG_DFL);
  psignal(signum,"Caught signal, trying to clean up.\nSend signal or CTRL-C again to cancel cleanup.\nSignal was");
  startalk_prepfor_exit();
  switch(signum)
  {
    case SIGSEGV:
      abort();
      break;
    default:
      _exit(1);
  }
}


int
main(int argc, char *argv[])
{
  int i, j;
  struct startalk_phonebook_entry *pbe;
  struct startalk_settings *se;
  int mode;
  extern char *optarg;
  int c;
  int use_phone=0;
  int clearout = 0;
  struct pbset *active = NULL;
  int default_active;
  int try;
  int showempty = 0;
  int settings = 0;
  
  mode = MODE_DEFAULT;
  while ( (c = getopt(argc, argv, "crwtT:n:ep:PsS:i:vV:d:h")) != -1 )
  {
    debugf(4,"Getopt returned '%c' (%d)\n",isprint(c)?c:'?',c);
    switch(c)
    {
      case 'c':
        debugf(4,"Processing -c argument...");
        clearout = 1;
        debugf(4,"done!\n");
        break;

      case 'r':
        debugf(4,"Processing -r argument...");
        mode=MODE_READ;
        use_phone=1;
        debugf(4,"done!\n");
        break;

      case 'w':
        debugf(4,"Processing -w argument...");
        mode=MODE_WRITE;
        use_phone=1;
        debugf(4,"done!\n");
        break;

      case 't':
        debugf(4,"Processing -t argument...");
        testing=1;
        debugf(4,"done!\n");
        break;

      case 'T':
        debugf(4,"Processing -T '%s' argument...",optarg);
        if (atoi(optarg) == 0)
          usage(argv[0],"Invalid setting '%s' for -T switch\n",optarg);
	if (startalk_timeout < 0)
	  startalk_timeout = 0;
        startalk_timeout = atoi(optarg);
        debugf(4,"done!\n");
        break;

      case 'n':
        debugf(4,"Processing -n '%s' argument...",optarg);
        if ( (active = malloc(sizeof(struct pbset))) == NULL)
          die("malloc error allocating memory for -n:");
        if (process_set(active,optarg) != 0)
        {
          errdump("processing phonebook entries");
          usage(argv[0],NULL);
        }
        default_active=0;
        debugf(4,"done!\n");
        break;

      case 'e':
        debugf(4,"Processing -e argument...");
        showempty = 1;
        debugf(4,"done!\n");
        break;

      case 'v':
        debugf(4,"Processing -v argument...");
        if (!debuglevel(1))
        {
          debugf(2,"debuglevel already set, ignoring -v");
          setdebuglevel(1);
        }
        debugf(4,"done!\n");
        break;

      case 'V':
        debugf(4,"Processing -V '%s' argument...",optarg);
        if (startalk_set_version(optarg) != 0)
          die("Error setting version to '%s'",optarg);
        debugf(4,"done!\n");
        break;
        
      case 'p':
        debugf(4,"Processing -p '%s' argument...",optarg);
        if ( (dev=strdup(optarg)) == NULL)
          die("malloc error allocating memory for -p:");
        debugf(4,"done!\n");
        break;

      case 'P':
        debugf(4,"Processing -P argument...");
        mode = MODE_PARSETEST;
        debugf(4,"done!\n");
        break;
        
      case 'S':
        debugf(4,"Processing -s '%s' argument...",optarg);
#ifdef STTY_SERIALSETUP
        if ( (stty=strdup(optarg)) == NULL)
          die("malloc error allocating memory for -s:");
#else
        die("Support for -s option not compiled in.");
#endif
        debugf(4,"done!\n");
        break;

#ifdef STARTALK_SETTINGS
      case 's':
        debugf(4,"Processing -s argument...");
        settings = 1;
        debugf(4,"done!\n");
        break;
#endif
        
      case 'i':
        debugf(4,"Processing -i '%s' argument...",optarg);
        if ( (modeminit=malloc(strlen(optarg)+3)) == NULL)
          die("malloc error allocating memory for -i:");
        if (strncasecmp(optarg,"at",2) != 0) 
          strcpy(modeminit,"AT");
        else
          modeminit[0]='\0';
        strcat(modeminit,optarg);
        debugf(4,"done!\n");
        break;
        
      case 'd':
        debugf(4,"Processing -d '%s' argument...",optarg);
        if (atoi(optarg) == 0)
          usage(argv[0],"Invalid setting '%s' for -d switch\n",optarg);
        setdebuglevel(atoi(optarg));
        debugf(4,"done processing -d '%s' argument!\n",optarg);
        break;

      case 'h':
      case '?':
      default:
        debugf(4,"Processing -h, -?, or unknown argument...");
        usage(argv[0],NULL);
        debugf(4,"done!\n");
    }
  }
  debugf(3,"Done processing arguments!\n");

  debugf(3,"Selecting mode...");
  /* If no mode was specified on the command line . . . */
  if ( (mode == MODE_DEFAULT) ) 
  {
    if (clearout)
    {
      use_phone = 1;
      if (!active)
        die("You must specify either a '-w' option with -c,\nor select a range to delete with '-n'.\nTo clear out the entire phonebook, use '-n 1-%d'.\nRun %s with no arguments to see more help.",MAX_ENTRIES,argv[0]);

      mode = MODE_CLEAR;
    }
    else if (testing)
    {
      mode = MODE_TEST;
      use_phone = 1;
    }
    else
      usage(argv[0],NULL);
  }
  debugf(3,"Selected mode %d!\n",mode);

  debugf(3,"Building entry list...");
  if (!active)
  {
    /* Build a pbset for all entries. */
    active = malloc(sizeof(struct pbset));
    if (!active) die("Malloc error");
    for(i=0;i<MAX_ENTRIES;i++)
      active->num[i]=i+1;
    active->count=MAX_ENTRIES;
    default_active=1;
  } 
  debugf(3,"built!\n");

  /* Now is when we start to care about the phone being reset if
   * we are interrupted.
   */
  debugf(3,"Setting up interrupt handlers...");
  signal(SIGHUP,signal_cleanup);
  signal(SIGINT,signal_cleanup);
  signal(SIGPIPE,signal_cleanup);
  signal(SIGTERM,signal_cleanup);
  signal(SIGQUIT,signal_cleanup);
  signal(SIGSEGV,signal_cleanup);
  init_alarm();
  debugf(3,"done!\n");
  
  if (use_phone)
  {
    debugf(4,"OK, I'm going to start communicating with the phone now.\nHang on tight!\n");
    serialfd = startalk_open(dev);
    if (serialfd == -1)
      die("opening phone on device '%s'",dev);
    if (serialfd < 3)
      die("opening phone on device '%s': file descriptor too low",dev);
    
    for(try=1;try <= INITIALIZE_TRIES;try++)
    {
      if (try > 1)
      {
        debugf(1,"Timeout setting up phone on try #%d; trying %d more time%s\n",try-1,INITIALIZE_TRIES-try+1,(INITIALIZE_TRIES-try+1) == 1 ? "" : "s");
        if (startalk_timeout < 10)
          sleep(15-startalk_timeout);
        else
          sleep(5);
        clearerrors();
      }

      if (startalk_initialize(serialfd) != 0)
      {
        if (check_alarm())
        {
          errorf("Timeout initializing phone.\n");
          debugf(1,"Timeout initializing phone.\n");
          continue;
        }
        else
          die("initializing phone");
      }
      
    
      if (startalk_datamode(serialfd) != 0)
      {
        if (check_alarm())
        {
          errorf("Timeout putting phone into data mode.\n");
          debugf(1,"Timeout putting phone into data mode.\n");
          continue;
        }
        else
          die("entering data mode");
      }

      debugf(1,"Looks good so far.  Phone is open and initialized.\n");
      break;
    }
    if (try > INITIALIZE_TRIES)
    {
      die("configuring phone");
    }
    
    if (debuglevel(1))
    {
      if (startalk_get_greeting(serialfd,NULL,0) != 0)
        die("getting phone greeting");
    }
  }

  if (settings)
  {
    if (startalk_initialize_stsettings(serialfd) != 0)
      die("initializing stsettings");
    if (mode == MODE_READ)
    {
      if (!(se = startalk_get_settings(serialfd)))
        die("Error reading phone settings");
      startalk_print_settings(se);
    }
    else if ( (mode == MODE_WRITE) || (mode == MODE_PARSETEST) )
    {
      if ( (se = startalk_get_settings_from_file(stdin)) == NULL)
        die("Error reading new settings from file");
      if (mode == MODE_WRITE)
      {
        if (startalk_put_settings(serialfd, se) != 0)
          die("Error writing settings to phone");
      }
      else
        startalk_print_settings(se);
    }
    else
    {
      die("Invalid startalk mode selected ( ?? )");
    }
  }
  else
  {
    if (mode != MODE_PARSETEST)
      if (startalk_initialize_stdata(serialfd) != 0)
        die("initializing stdata");

    startalk_dump_phonebook_offsets(); /* Does nothing without debugging */

    if (mode == MODE_READ)
    {
      i=0;
      for(i=0;i<active->count;i++)
      {
        if ( (pbe=startalk_get_phonebook_entry(serialfd,active->num[i])) == NULL)
        {
          warn("Error reading phonebook entry %d",active->num[i]);
        }
        else if (!startalk_phonebook_entry_is_empty(pbe) || showempty)
        {
          startalk_print_phonebook_entry(pbe);
        }
      }
    }
    else if ( (mode == MODE_WRITE) || (mode == MODE_PARSETEST) || (mode == MODE_CLEAR) )
    {
      int wrote[MAX_ENTRIES];

      if (clearout)
        for(i=0;i<active->count;i++)
        {
          wrote[active->num[i]]=0;
        }
      
      i=0;
      if (mode != MODE_CLEAR)
      {
        while ( (pbe = startalk_get_phonebook_entry_from_file(stdin)) != NULL)
        {
          if (pbe->pos == -1)
            pbe->pos=active->num[i];
          else
          {
            /* This is inefficient.  FIX XXX */
            for(j=0;j<active->count;j++)
              if (pbe->pos == active->num[j])
                break;
            if (j >= active->count)
            {
              debugf(1,"Entry %d not specified in range, skipping.",pbe->pos);
              continue;
            }
          }
          debugf(1,"Writing entry at %d\n",pbe->pos);
          if (debuglevel(1))
            startalk_fprint_phonebook_entry(stderr, pbe);
          
          if (mode == MODE_PARSETEST)
          {
            startalk_print_phonebook_entry(pbe);
          }
          if (mode != MODE_PARSETEST &&
              (startalk_put_phonebook_entry(serialfd, pbe->pos, pbe) != 0))
            warn("Error writing phonebook entry %d",pbe->pos);
          else
          {
            if (clearout)
              wrote[pbe->pos]=1;
          }
          /* free(pbe) ? */ /* FIX XXX */
          if (i++ >= active->count)
            break;
        }
      }
      if (clearout)
      {
        for (i=0;i<active->count;i++)
        {
          if (!wrote[active->num[i]])
          {
            debugf(1,"Deleting entry at %d\n",i);
            if (mode == MODE_PARSETEST)
            {
              printf("position: %d\n\n",active->num[i]);
            }
            else /* MODE_PARSETEST */
            {
              if (startalk_remove_phonebook_entry(serialfd, active->num[i]) != 0)
                warn("Error removing phonebook entry #%d",active->num[i]);
            }
          }
        }
      }
    }
    else if (mode == MODE_TEST)
    {
      char testcmds[][3][10] = {
        { "test 0", "2", "\x60\x01" },
        { "test 1", "2", "\xC7\x0D" },
        { "test 2", "3", "\xCF\x01\x00" },
        { "test 3", "3", "\xCF\x07\x00" },
        { "test 4", "3", "\xCF\x07\x01" },
        { "test 5", "3", "\xCF\x07\x03" },
        { "", "", ""},
      };
      int i;
      char resbuf[8192];
      int rbsize;
      
      for(i=0;testcmds[i][0][0];i++)
      {
        debugf(1,"Running test %d (%s)\n",i,testcmds[i][0]);
        rbsize = 8192;
        if (startalk_run_cmd(serialfd, testcmds[i][2], testcmds[i][1][0]-'0',resbuf,&rbsize) != 0)
          die("running test %d\n",i);
      }
    }
  }
  if (use_phone)
  {
    startalk_uninitialize();
    debugf(1,"Phone is reset.\n");
  }

  finished();
  /* Never reached, placate -Wall */
  exit(0);
}
