/* $Id: stdata.c,v 400.1 2002/07/25 08:43:16 sgifford Exp $ */

#include "stdata.h"
#include "phonebook.h"
#include "startalk.h"
#include "stdebug.h"

#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#define ST_PB_READ_OFFSET       (1)
#define ST_HEADERLEN		(4)

#define ST_PB_START		(0)
#define ST_PB_POSITION          (ST_PB_START)
#define ST_PB_NAME1_START	(ST_PB_START + 1)
#define ST_PB_NAME1_LEN         (12)
#define ST_PB_NAME2_START	(ST_PB_NAME1_START + ST_PB_NAME1_LEN)
#define ST_PB_NAME2_LEN         (12)
#define ST_PB_UNKNOWN_START     (ST_PB_NAME2_START + ST_PB_NAME2_LEN)
#define ST_PB_UNKNOWN_LEN       (2)
#define ST_PB_EMAIL_START       (ST_PB_UNKNOWN_START+ST_PB_UNKNOWN_LEN)
#define ST_PB_EMAIL_LEN         (st_pb_email_len)
#define ST_PB_NUMSTART		(ST_PB_EMAIL_START+ST_PB_EMAIL_LEN)
#define ST_PB_PHONENUM_LEN	(16)
#define ST_PB_PHONENUM_COUNT	(4)
#define ST_PB_CHECKSUM_SIZE	(2)
#define ST_PB_TOTAL_SIZE	(ST_PB_NUMSTART + ((ST_PB_PHONENUM_LEN) * (ST_PB_PHONENUM_COUNT)) + ST_PB_CHECKSUM_SIZE)

int st_pb_email_len = -1;

/* Private prototypes */
static void startalk_encode_name(char *wbuf, char *rbuf, int len, char pad);
static char startalk_decode_digit(char d);
static void startalk_encode_phone(char *wbuf, char *rbuf, int len);
static int startalk_put_phonebook_buffer(int serialfd, char *sendbuf, int size);
static int startalk_get_phonebook_buffer(int serialfd, char *buf, int *size, int num);
void startalk_getstr(char *dest, const char *src, int maxlen);

void
startalk_dump_phonebook_offsets(void)
{
  debugf(3,"Phonebook Offsets:\n");
  debugf(3,"\t%30s %-3d\n","Phonebook Starts",ST_PB_START);
  debugf(3,"\t%30s %-3d\n","Name1 Starts",ST_PB_NAME1_START);
  debugf(3,"\t%30s %-3d\n","Name2 Starts",ST_PB_NAME2_START);
  debugf(3,"\t%30s %-3d\n","Unknown area Starts",ST_PB_UNKNOWN_START);
  debugf(3,"\t%30s %-3d\n","Unknown area length",ST_PB_UNKNOWN_LEN);
  debugf(3,"\t%30s %-3d\n","Email address Starts",ST_PB_EMAIL_START);
  debugf(3,"\t%30s %-3d\n","Email address length",ST_PB_EMAIL_LEN);
  debugf(3,"\t%30s %-3d\n","Phone numbers start",ST_PB_NUMSTART);
  debugf(3,"\t%30s %-3d\n","Phone number entry size",ST_PB_PHONENUM_LEN);
  debugf(3,"\t%30s %-3d\n","Phone number entry count",ST_PB_PHONENUM_COUNT);
  debugf(3,"\t%30s %-3d\n","Total phone number size",ST_PB_PHONENUM_LEN*ST_PB_PHONENUM_COUNT);
  debugf(3,"\t%30s %-3d\n","Checksum size",ST_PB_CHECKSUM_SIZE);
  debugf(3,"\t%30s %-3d\n","Total entry size",ST_PB_TOTAL_SIZE);
}

int
startalk_initialize_stdata(int serialfd)
{
  char buf[8192];
  int msgsize;
  char *pbv;

  if (st_pb_email_len >= 0)
  {
    debugf(3, "startalk_initialize_stdata: phonebook entry format already determined!");
    return 0;
  }
  
  debugf(3,"Determining phonebook entry format...\n");

  msgsize = 8192;
  if (startalk_get_phonebook_buffer(serialfd, buf, &msgsize, 0) < 0)
    return errorf("Error determining phonebook entry format");
  
  if (msgsize == 100)
  {
    pbv = "1";
  }
  else if (msgsize == 150)
  {
    pbv = "2";
  }
  else
  {
    die("Unknown phonebook format!  Phonebook entry was %d bytes.",msgsize);
  }

  return startalk_set_version(pbv);
}

int
startalk_set_version(char *v)
{
  if (strcmp(v,"1") == 0)
  {
    st_pb_email_len = 0;
  }
  else if (strcmp(v,"2") == 0)
  {
    st_pb_email_len = 50;
  }
  else
  {
    return reterr(-1,"Unknown phonebook version '%s'",v);
  }
  return 0;
}
   

struct startalk_phonebook_entry *
startalk_get_phonebook_entry(int serialfd, int num)
{
  struct startalk_phonebook_entry *pbe;
  
  unsigned char *entry;
  unsigned char buf[8192];
  char phonebuf[33];
  char *phonepos;
  int msgsize;
  int whichbyte;
  int numtype;
  int i,j;
  int digit;
  
  /* Sanity check */
  if (st_pb_email_len < 0)
    die("st_pb_email_len < 0!!");

  pbe = startalk_new_phonebook_entry();
  if (!pbe)
    return (void *) reterr(0,"Error allocating new phonebook entry");


  msgsize = 8191;
  if (startalk_get_phonebook_buffer(serialfd, (char *)buf, &msgsize, num) < 0)
    return (void *) reterr(0,"Error reading phonebook entry %d",num);
  
  entry = (unsigned char *)buf + ST_HEADERLEN + ST_PB_READ_OFFSET;
  
  /* OK, now actually dump it. */
  if ( (entry[ST_PB_NAME1_START] == 0) && (entry[ST_PB_NUMSTART] == 0) && (entry[ST_PB_NUMSTART+1] == 0) && (entry[ST_PB_EMAIL_START] == 0) )
  {
    /* Empty. */
    startalk_empty_phonebook_entry(pbe);
    pbe->pos = num;
    return pbe;
  }

  pbe->pos = num;
  startalk_getstr(pbe->name,(char *)&entry[ST_PB_NAME1_START],MAX_NAMELEN);
  debugf(3,"Found name '%s' at entry offset %d\n",pbe->name,ST_PB_NAME1_START);

  startalk_getstr(pbe->company,(char *)&entry[ST_PB_NAME2_START],MAX_COMPANYLEN);
  debugf(3,"Found company '%s' at entry offset %d\n",pbe->company,ST_PB_NAME2_START);

  if (ST_PB_EMAIL_LEN > 0)
  {
    startalk_getstr(pbe->email,(char *)&entry[ST_PB_EMAIL_START],MAX_EMAILLEN);
    debugf(3,"Found email '%s' at entry offset %d\n",pbe->email,ST_PB_EMAIL_START);
  }
  else
  {
    debugf(3,"Email address not supported in this version, skipping.\n");
  }
  
  for (i=0;i<4;i++)
  {
    whichbyte = ST_PB_NUMSTART + i / 2;
    if (i%2)
      numtype = entry[whichbyte] % 16;
    else
      numtype = entry[whichbyte] / 16;
    if (numtype == 0)
      continue;
    debugf(3,"Found number type '%d' at entry offset %d.%d\n",numtype,whichbyte,(i%2)?5:0); 

    j=ST_PB_NUMSTART+2+16*i;

    phonepos = phonebuf;
    while(1)
    {
      digit = entry[j] / 16;
      if (digit)
      {
        *phonepos = startalk_decode_digit(digit);
      }
      else
        break;
      phonepos++;

      digit = entry[j] % 16;
      if (digit)
      {
        *phonepos = startalk_decode_digit(digit);
      }
      else
        break;
      phonepos++;

      j++;
      if (j >= (ST_PB_NUMSTART+2+16*(i+1)))
        break;
    }
    *phonepos = 0;
 
    debugf(3,"Found number '%s' at entry offset %d\n",phonebuf,ST_PB_NUMSTART+2+ST_PB_PHONENUM_LEN*i);
    if (!startalk_add_phone_to_entry(pbe, numtype, phonebuf))
      warn("adding phone number to entry");
  }

  return pbe;
}

static void
startalk_encode_name(char *wbuf, char *rbuf, int len, char pad)
{
  int i;
  
  for (i=0;i<len;i++)
  {
    if (*rbuf)
    {
      *wbuf = *rbuf;
      rbuf++;
    }
    else
    {
      *wbuf = pad;
    }
    wbuf++;
  }
}

static char
startalk_decode_digit(char d)
{
  switch(d)
  {
    case 10:
      return '0';
    case 1: case 2: case 3: case 4: case 5:
    case 6: case 7: case 8: case 9:
      return d+'0';
    case 11:
      return '*';
    case 12:
      return '#';
    case 13:
      return 'o';
    case 14:
      return 'n';
    default:
      return '\0';
  }
}

static void
startalk_encode_phone(char *wbuf, char *rbuf, int len)
{
  int i;
  unsigned char byte;
  unsigned char *uwbuf, *urbuf;

  uwbuf = (unsigned char *)wbuf;
  urbuf = (unsigned char *)rbuf;
  
  for (i=0;i<len*2;i++)
  {
    switch(urbuf[i])
    {
      case '0':
        byte = 10;
        break;
      case '1': case '2': case '3': case '4': case '5':
      case '6': case '7': case '8': case '9':
        byte = urbuf[i] - '0';
        break;
      case '*':
        byte = 11;
        break;
      case '#':
        byte = 12;
        break;
      case 'o':
      case 'O':
        byte=13;
        break;
      case 'n':
      case 'N':
        byte=14;
        break;
      default:
        byte = 0;
        break;
    }
    
    if ( (i % 2) == 0)
    {
      *uwbuf = byte << 4;
    }
    else
    {
      *uwbuf |= byte;
      uwbuf++;
    }
  }
}

int
startalk_put_phonebook_entry(int serialfd, int num, struct startalk_phonebook_entry *pbe)
{
  unsigned char sendbuf[256];
  char *entry;
  int byte;
  int i;
  
  /* Sanity check */
  if (st_pb_email_len < 0)
    die("st_pb_email_len < 0!!");

  /* Check if this is an empty phonebook entry.  If so, we'll
   * delete it.
   */
  if ( (pbe->name[0] == '\0') && (pbe->company[0] == '\0') &&
       (pbe->numphones == 0))
    return startalk_remove_phonebook_entry(serialfd, num);
  
  /* OK, let's build this request packet. */
  /* Header */
  /* Bytes 0-3 */
  sendbuf[0] = 0xcf;
  sendbuf[1] = 0x01;
  sendbuf[2] = 0x02;
  sendbuf[3] = 0x00;

  entry = (char *)sendbuf + 4;
  /* Position in phonebook */
  /* Byte 4 */
  entry[ST_PB_POSITION] = num;
  debugf(3,"Storing position %d at entry offset %d\n",num,ST_PB_POSITION);
  
  /* Next 12 characters are name */
  /* Bytes 5-16 */
  startalk_encode_name(&entry[ST_PB_NAME1_START],pbe->name,ST_PB_NAME1_LEN,(pbe->company[0]?' ':'\0'));
  debugf(3,"Storing name '%s' at entry offset %d\n",pbe->name,ST_PB_NAME1_START);
  
  /* Next 12 are company */
  startalk_encode_name(&entry[ST_PB_NAME2_START],pbe->company,ST_PB_NAME2_LEN,'\0');
  debugf(3,"Storing company '%s' at entry offset %d\n",pbe->company,ST_PB_NAME2_START);

  /* Bytes 29-30 */
  /* Now we have some unknown null bytes */
  memset(entry+ST_PB_UNKNOWN_START,0,ST_PB_UNKNOWN_LEN);

  /* Bytes 31-80, if we support email */
  if (ST_PB_EMAIL_LEN > 0)
  {
    startalk_encode_name(&entry[ST_PB_EMAIL_START],pbe->email,ST_PB_EMAIL_LEN,'\0');
    debugf(3,"Storing email address '%s' at entry offset %d\n",pbe->company,ST_PB_EMAIL_START);
  }
  else
  {
    if (pbe->email[0])
    {
      warn("WARNING: Entry %d - Email address not supported on this StarTAC phone.\n",num);
    }
    debugf(3,"Skipping email address.\n");
  }

  debugf(3,"Storing %d bytes of NUL at entry offset %d\n",ST_PB_UNKNOWN_LEN,ST_PB_UNKNOWN_START);
  
  /* Bytes 31-32 or 81-83 */
  /* And two bytes to indicate the types of the phone numbers */
  /* Maybe these need to be sorted, but it's much easier for now
   * not to.
   */
  for(i=0;i<4;i++)
  {
    if (i < pbe->numphones)
      byte = pbe->phone[i]->type;
    else
      byte = 0;

    if ( (i % 2) == 0)
      entry[ST_PB_NUMSTART+i/2] = byte << 4;
    else
      entry[ST_PB_NUMSTART+i/2] |= byte;
    debugf(3,"Storing phonebook entry type %d at entry offset %d.%d\n",byte,ST_PB_NUMSTART+i/2,(i%2)?5:0);
  }

  /* Bytes 33-97 or 83-147 */
  /* Then 16 bytes for each of the 4 phone numbers.  Each digit is
   * encoded in one nybble, so we can hold 32 digits this way.
   * The digit '0' is encoded as 0x0a, and the extra bytes are all
   * NULLs.
   */

  for (i=0;i<ST_PB_PHONENUM_COUNT;i++)
  {
    if (i < pbe->numphones)
    {
      startalk_encode_phone((char *)&entry[ST_PB_NUMSTART+2+i*ST_PB_PHONENUM_LEN],pbe->phone[i]->num,ST_PB_PHONENUM_LEN);
      debugf(3,"Storing telephone number #%d at entry offset %d\n",i,ST_PB_NUMSTART+2+i*ST_PB_PHONENUM_LEN);
    }
    else
    {
      memset(&entry[ST_PB_NUMSTART+2+i*ST_PB_PHONENUM_LEN], 0, ST_PB_PHONENUM_LEN);
      debugf(3,"Stored empty telephone number #%d at entry offset %d\n",i,ST_PB_NUMSTART+2+i*ST_PB_PHONENUM_LEN);
    }
  }

  debugf(3,"OK, sending completed packet to phone\n");

  return startalk_put_phonebook_buffer(serialfd, (char *)sendbuf, ST_PB_TOTAL_SIZE + 4);
}

int
startalk_remove_phonebook_entry(int serialfd, int num)
{
  unsigned char *sendbuf;
  int rv;
  
  /* Sanity check */
  if (st_pb_email_len < 0)
    die("st_pb_email_len < 0!!");

  if ( (sendbuf = malloc(ST_PB_TOTAL_SIZE+4)) == NULL)
    die("malloc error in startalk_remove_phonebook_entry:");
  /* OK, let's build this request packet. */
  /* Header */
  /* Bytes 0-3 */
  sendbuf[0] = 0xcf;
  sendbuf[1] = 0x01;
  sendbuf[2] = 0x02;
  sendbuf[3] = 0x00;

  /* Position in phonebook */
  /* Byte 4 */
  sendbuf[4] = num;
  /* ST_PB_TOTAL_SIZE includes the entry number, which we don't
   * want to zero out.
   */
  memset(&sendbuf[5],0,ST_PB_TOTAL_SIZE-1);

  rv = startalk_put_phonebook_buffer(serialfd, (char *)sendbuf, ST_PB_TOTAL_SIZE + 4);

  free(sendbuf);

  return rv;
}

static int
startalk_put_phonebook_buffer(int serialfd, char *sendbuf, int size)
{
  unsigned char resbuf[256];
  int rbsize = 256;
  
  if (startalk_run_cmd(serialfd, sendbuf, size, (char *)resbuf, &rbsize) < 0)
    return errorf("Error running command to write phonebook entry");

  if (rbsize<8)
  {
    return errorf("phone returned error while writing phone book entry: response too short (%d bytes)",rbsize);
  }
  else if (resbuf[6] != 0xac)
  {
    return errorf("phone returned error while writing phone book entry: response code 0x%02x",resbuf[6]);
  }
  
  debugf(3,"Response read succesfully!\n");
    
  return 0;
}

static int
startalk_get_phonebook_buffer(int serialfd, char *buf, int *size, int num)
{
  unsigned char cmdbuf[5] = { 0xcf, 0x01, 0x01, 0x00, 0x00 };
  unsigned char *ubuf = (unsigned char *)buf;
  
  debugf(3,"Reading phonebook entry %d\n",num);
  cmdbuf[4]=num;
  if (startalk_run_cmd(serialfd, (char *)cmdbuf, 5, (char *)buf, size) != 0)
    return errorf("Error running command to read phonebook entry");

  if (st_pb_email_len >= 0)
  {
    if (*size != (ST_PB_TOTAL_SIZE + ST_HEADERLEN + ST_PB_READ_OFFSET + ST_PB_CHECKSUM_SIZE))
      return errorf("Bad phonebook entry received, of unrecognized size %d (expected %d)",*size, ST_PB_TOTAL_SIZE+ST_HEADERLEN+ST_PB_READ_OFFSET+ST_PB_CHECKSUM_SIZE);
  }
  
  /* Make sure our results are OK. */
  if ( (ubuf[5] != 0x5C) && (ubuf[5] != 0x8E) )
    return errorf("Bad phonebook entry received, byte 5 should be 0x5C or 0x8E, but it's 0x%02x instead!",ubuf[5]);

  debugf(3,"Phonebook entry read succesfully!\n");

  return 0;
}

void
startalk_getstr(char *dest, const char *src, int maxlen)
{
  char *end;
  
  strncpy(dest,src,maxlen);
  dest[maxlen]='\0';

  /* Trim spaces from end */
  end = strchr(dest,'\0');
  while (end > dest)
  {
    end--;
    if (isspace(*end))
      *end='\0';
    else
      break;
  }
}
