/* This file is part of Strigi Desktop Search
 *
 * Copyright (C) 2007 Jos van den Oever <jos@vandenoever.info>
 * Copyright (C) 2007 Alexandr Goncearenco <neksa@neksa.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "fieldpropertiesdb.h"
#include "fieldproperties.h"
#include "fieldtypes.h"
#include <vector>
#include <sys/types.h>
#ifdef HAVE_DIRENT_H
#include <dirent.h>
#else
#include "stgdirent.h"
#endif
#include <sys/stat.h>
#include <config.h>
#include <cstdlib>
#include <cstring>
using namespace Strigi;
using namespace std;

class FieldPropertiesDb::Private {
public:
    map<string, FieldProperties> properties;
    static FieldProperties empty;

    Private();
    vector<string> getdirs(const string&) const;
    void loadProperties(const string& dir);
    void parseProperties(char*data);
    void handlePropertyLine(FieldProperties::Private& p, const char*name,
        const char* locale, const char*value);
    void storeProperties(FieldProperties::Private& props);
    void warnIfLocale(const char* name, const char* locale);
};
FieldProperties FieldPropertiesDb::Private::empty;

FieldPropertiesDb&
FieldPropertiesDb::db() {
    static FieldPropertiesDb db;
    return db;
}
FieldPropertiesDb::FieldPropertiesDb() :p(new Private()) {
}
FieldPropertiesDb::~FieldPropertiesDb() {
    delete p;
}
const FieldProperties&
FieldPropertiesDb::properties(const std::string& uri) const {
    map<std::string, FieldProperties>::const_iterator j = p->properties.find(uri);
    if (j == p->properties.end()) {
	return FieldPropertiesDb::Private::empty;
    } else {
	return j->second;
    }
}
const map<string, FieldProperties>&
FieldPropertiesDb::allProperties() const {
    return p->properties;
}

FieldPropertiesDb::Private::Private() {
    const char* dirpath = getenv("XDG_DATA_HOME");
    vector<string> dirs;
    if (dirpath) {
        dirs = getdirs(dirpath);
    } else {
        dirpath = getenv("HOME");
        if (dirpath) {
             dirs = getdirs(string(dirpath)+"/.local/share");
        }
    }
    dirpath = getenv("XDG_DATA_DIRS");
    vector<string> d;
    if (dirpath) {
        d = getdirs(dirpath);
    } else {
        d = getdirs(INSTALLDIR "/share:/usr/local/share:/usr/share");
    }
    copy(d.begin(), d.end(), back_insert_iterator<vector<string> >(dirs));
    vector<string>::const_iterator i;
    for (i=dirs.begin(); i!=dirs.end(); i++) {
        loadProperties(*i);
    }
}
vector<string>
FieldPropertiesDb::Private::getdirs(const string& direnv) const {
    vector<string> dirs;
    string::size_type lastp = 0;
    string::size_type p = direnv.find(':');
    while (p != string::npos) {
        dirs.push_back(direnv.substr(lastp, p-lastp));
        lastp = p+1;
        p = direnv.find(':', lastp);
     }
    dirs.push_back(direnv.substr(lastp));
    return dirs;
}
void
FieldPropertiesDb::Private::loadProperties(const string& dir) {
    string pdir = dir + "/strigi/fieldproperties/";
    DIR* d = opendir(pdir.c_str());
    if (!d) return;
    struct dirent* de = readdir(d);
    struct stat s;
    char* data = 0;
    while (de) {
        string path(pdir+de->d_name);
        if (!stat(path.c_str(), &s)) {
            if (S_ISREG(s.st_mode)) {
                FILE* f = fopen(path.c_str(), "r");
                if (f) {
                    data = (char*)realloc(data, s.st_size+1);
                    if (fread(data, 1, s.st_size, f) == (size_t)s.st_size) {
                        //fprintf(stderr, "parsing %s\n", path.c_str());
                        data[s.st_size] = '\0';
                        parseProperties(data);
                    }
                    fclose(f);
                }
            }
        }
        de = readdir(d);
    }
    closedir(d);
    free(data);
}
void
FieldPropertiesDb::Private::parseProperties(char* data) {
    FieldProperties::Private props;
    char* p = data;
    char* nl;
    do {
        nl = strchr(p, '\n');
        if (nl) *nl = '\0';
        if (*p == '#' || *p == '[') {
            storeProperties(props);
        }
        char* eq = strchr(p, '=');
        if (eq) {
            *eq = '\0';
            // find square brackets
            char* sbo = strchr(p, '[');
            if (sbo > eq) sbo = 0;
            char* sbc;
            if (sbo) {
                sbc = strchr(sbo, ']');
                if (sbc > eq || !sbc) {
                   sbo = 0;
                }
            }
            if (sbo) {
                *sbo = '\0';
                sbo++;
                *sbc = '\0';
            }
            handlePropertyLine(props, p, sbo, eq+1);
        }
        
        p = nl + 1;
    } while(nl);
    if (props.uri.size()) {
        properties[props.uri] = props;
    }
}
void
FieldPropertiesDb::Private::storeProperties(FieldProperties::Private& p) {
    if (p.uri.size()) {
        properties[p.uri] = p;
    }
    p.clear();
}
void
FieldPropertiesDb::Private::handlePropertyLine(FieldProperties::Private& p,
        const char*name, const char* locale, const char*value) {
    if (strcmp(name, "Uri") == 0) {
        warnIfLocale(name, locale);
        if (p.uri.size()) {
            fprintf(stderr, "Uri is already defined for %s.\n", p.uri.c_str());
        } else {
            p.uri.assign(value);
        }
    } else if (strcmp(name, "TypeUri") == 0) {
        if (p.typeuri.size()) {
            fprintf(stderr, "TypeUri is defined twice for %s.\n",
                p.uri.c_str());
        } else {
            p.typeuri.assign(value);
        }
    } else if (strcmp(name, "Name") == 0) {
        if (locale) {
            FieldProperties::Localized l(p.localized[locale]);
            if (l.name.size()) {
                fprintf(stderr, "Name[%s] is already defined for %s.\n",
                     locale, p.uri.c_str());
            } else {
                l.name.assign(value);
                p.localized[locale] = l;
            }
        } else if (p.name.size()) {
            fprintf(stderr, "Name is already defined for %s.\n", p.uri.c_str());
        } else {
            p.name.assign(value);
        }
    } else if (strcmp(name, "Description") == 0) {
        if (locale) {
            FieldProperties::Localized l(p.localized[locale]);
            if (l.description.size()) {
                fprintf(stderr, "Description[%s] is already defined for %s.\n",
                     locale, p.uri.c_str());
            } else {
                l.description.assign(value);
                p.localized[locale] = l;
            }
        } else if (p.description.size()) {
            fprintf(stderr, "Description is already defined for %s.\n",
                p.uri.c_str());
        } else {
            p.description.assign(value);
        }
    } else if (strcmp(name, "ParentUri") == 0) {
        p.parentUris.push_back(value);
    } else if (strcmp(name, "Binary") == 0) {
	if (strcasecmp(value,"false") == 0) {
    	    p.binary = false;
	} else if (strcasecmp(value,"true") == 0) {
	    p.binary = true;
	} else {
	    fprintf(stderr, "Binary flag value[%s] for %s is unrecognized. Should be in set {True,False}.\n",
	    value, p.uri.c_str());
	}	
    } else if (strcmp(name, "Compressed") == 0) {
	if (strcasecmp(value,"false") == 0) {
    	    p.compressed = false;
	} else if (strcasecmp(value,"true") == 0) {
	    p.compressed = true;
	} else {
	    fprintf(stderr, "compressed flag value[%s] for %s is unrecognized. Should be in set {True,False}.\n",
	    value, p.uri.c_str());
	}	
    } else if (strcmp(name, "Indexed") == 0) {
	if (strcasecmp(value,"false") == 0) {
    	    p.indexed = false;
	} else if (strcasecmp(value,"true") == 0) {
	    p.indexed = true;
	} else {
	    fprintf(stderr, "Indexed flag value[%s] for %s is unrecognized. Should be in set {True,False}.\n",
	    value, p.uri.c_str());
	}	
    } else if (strcmp(name, "Stored") == 0) {
	if (strcasecmp(value,"false") == 0) {
    	    p.stored = false;
	} else if (strcasecmp(value,"true") == 0) {
	    p.stored = true;
	} else {
	    fprintf(stderr, "Stored flag value[%s] for %s is unrecognized. Should be in set {True,False}.\n",
	    value, p.uri.c_str());
	}	
    } else if (strcmp(name, "Tokenized") == 0) {
	if (strcasecmp(value,"false") == 0) {
    	    p.tokenized = false;
	} else if (strcasecmp(value,"true") == 0) {
	    p.tokenized = true;
	} else {
	    fprintf(stderr, "Tokenized flag value[%s] for %s is unrecognized. Should be in set {True,False}.\n",
	    value, p.uri.c_str());
	}	
    } else if (strcmp(name, "MinCardinality") == 0) {
	p.min_cardinality = atoi(value);
    } else if (strcmp(name, "MaxCardinality") == 0) {
	p.max_cardinality = atoi(value);
    }
}
void
FieldPropertiesDb::Private::warnIfLocale(const char* name, const char* locale) {
    if (locale) {
        fprintf(stderr, "Warning: you cannot define a locale for the field %s.",
            name);
    }
}
void
FieldPropertiesDb::addField(const std::string& key, const std::string& type,
        const std::string& parent) {
    FieldProperties::Private props;
    props.uri = key;
    props.typeuri = type;
    if (parent.size()) {
        props.parentUris.push_back(parent);
    }
    p->properties[key] = props;
}
void
FieldPropertiesDb::addField(const std::string& key) {
    FieldProperties::Private props;
    props.uri = key;
    props.typeuri = FieldRegister::stringType;
    p->properties[key] = props;
}
void
FieldProperties::Private::clear() {
    uri.clear();
    name.clear();
    description.clear();
    localized.clear();
    typeuri.clear();
    parentUris.clear();

    indexed = true;
    stored = true;
    tokenized = true;
    compressed = false;
    binary = false;
    
    min_cardinality = -1; /** unlimited */
    max_cardinality = -1; /** unlimited */
}
