/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "CommitDialog.h"
#include "CommitItemLvi.h"
#include "WcModel.h"
#include "ScModel.h"
#include "TextEdit.h"
#include "PostCmdResult.h"
#include "NullCmdResult.h"
#include "Commit.h"
#include "settings/FontSettings.h"
#include "settings/CommitLogHistory.h"
#include "commands/DiffParam.h"
#include "commands/CommitParam.h"
#include "sublib/Gui.h"
#include "sublib/SplitLayout.h"
#include "svn/Path.h"
#include "svn/CommitItem.h"
#include "util/String.h"
#include "util/Condition.h"

// qt
#include <qlayout.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qtooltip.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qlistview.h>
#include <qvbox.h>
#include <qlistbox.h>



CommitDialog::CommitDialog( bool direct, ScModel* model, QWidget *parent,
  ID tid )
: super( parent, NULL, true, Qt::WStyle_Customize | Qt::WStyle_Dialog |
  Qt::WStyle_NormalBorder | Qt::WStyle_Title | Qt::WStyle_SysMenu ),
  _direct(direct), _model(model), _tid(tid), _commitTriggered(false),
  _cond(NULL), _result(NULL)
{
  setCaption( _q("subcommander:commit") );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    QGroupBox* gb = new QGroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("commit options: ") );
    //gb->setInsideSpacing(10);
    gb->setInsideMargin(1);
    gb->setFlat(true);
    vbl->addWidget(gb);

    QHBoxLayout* h1 = new QHBoxLayout;
    vbl->addLayout(h1);
    {
      _recurse = new QCheckBox( _q("&recursive"), this );
      _recurse->setChecked(_model->getOptionCommandRecurse());
      _recurse->setEnabled(_direct);
      h1->addWidget(_recurse);

      _keepLocks = new QCheckBox( _q("&keep locks"), this );
      _keepLocks->setChecked(false);
      _keepLocks->setEnabled(_direct);
      h1->addWidget(_keepLocks);

      h1->addStretch(1); 

      _log = new QComboBox(false,this);
      _log->setMaximumWidth(300);
      h1->addWidget(_log);

      connect( _log, SIGNAL(activated(int)), this, SLOT(logHistory(int)) );
      {
        CommitLogHistory* logHistory = model->getCommitLogHistory();
        CommitLogHistory::Logs logs;
        logHistory->getLogs(logs);

        for( CommitLogHistory::Logs::iterator it = logs.begin(); it != logs.end(); it++ )
        {
          QString qlog = QString::fromUtf8( *it );
          _log->insertItem(qlog,0);
        }
      }
    }

    QFont font = _model->getFontSettings()->getEditorFont();
    QFontMetrics m(font);

    QWidget*      vw     = new QWidget(this);
    VSplitLayout* vsplit = new VSplitLayout(vw,SplitLayout::pTwo);
    vbl->addWidget(vw);
    {
      QVBox* vb = new QVBox(vw);
      vb->setSpacing(4);
      {
        _edit = new TextEdit(vb);
        _edit->setFont( font );
        _edit->setUndoRedoEnabled(true);
        _edit->setTabChangesFocus(true);
        _edit->setSizePolicy( QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding) );
        //_edit->setWrapPolicy( QTextEdit::AtWordBoundary );
        //_edit->setWordWrap( QTextEdit::FixedColumnWidth );
        //_edit->setWrapColumnOrWidth(72);
        _edit->setMinimumWidth( 500 );
        _edit->setWordWrap( QTextEdit::NoWrap );

        _pos = new QLabel(vb);

        connect( _edit, SIGNAL(cursorChanged(int,int)), SLOT(cursorChanged(int,int)) );
        connect( _edit, SIGNAL(textChanged()), SLOT(logChanged()) );
      }
      vsplit->addWidgetOne(vb,false,5);

      QVBox* vb2 = new QVBox(vw);
      vb2->setSpacing(5);
      {
        QWidget* vb2w = new QWidget(vb2);
        QGridLayout* gl = new QGridLayout(vb2w,2,3);
        gl->setMargin(0);
        gl->setSpacing(2);
        {
          QLabel* lUrl  = new QLabel( _q("destination url: "), vb2w );
          QLabel* lPath = new QLabel( _q("source path: "), vb2w );

          gl->addWidget( lUrl, 0, 0 );
          gl->addWidget( lPath, 1, 0 );

          _url  = new QLabel(vb2w);
          _path = new QLabel(vb2w);

          gl->addWidget( _url, 0, 2 );
          gl->addWidget( _path, 1, 2 );

          gl->setColStretch( 0, 1 );
          gl->setColStretch( 1, 0 );
          gl->setColSpacing( 1, 10 );
          gl->setColStretch( 2, 10 );
        }

        _list = new QListView(vb2);
        _list->setShowToolTips( true );
        _list->setAllColumnsShowFocus(true);
        _list->setItemMargin( 2 );
        _list->addColumn( _q("kind") );
        _list->addColumn( _q("flags") );
        _list->addColumn( _q("item") );
        _list->setColumnWidthMode( 0, QListView::Maximum );
        _list->setResizeMode( QListView::LastColumn );

        connect( _list, SIGNAL(doubleClicked( QListViewItem*, const QPoint&, int )),
          SLOT(doubleClicked( QListViewItem*, const QPoint&, int )) );
      }
      vsplit->addWidgetTwo(vb2,false,3);
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _ls = new QPushButton(this);
      _ls->setEnabled(_direct);
      _ls->setText( _q("List &Items") );
      _ls->setDefault(_direct);
      hu->addWidget(_ls);
      
      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("C&ommit") );
      _ok->setDefault(!_direct);
      hu->addWidget(_ok);
      
      _ca = new QPushButton(this);
      _ca->setText( _q("&Cancel") );
      _ca->setEnabled(true);
      hu->addWidget(_ca);
      
      hu->addSpacing(getSizeGripSpacing());

      connect( _ls, SIGNAL(clicked()), SLOT(list()) );
      connect( _ok, SIGNAL(clicked()), SLOT(commit()) );
      connect( _ca, SIGNAL(clicked()), SLOT(cancel()) );
    }
  }
}

CommitDialog::~CommitDialog()
{
}

void CommitDialog::setPaths( const svn::Paths& items )
{
  _items = items;
}

void CommitDialog::setCommitItems( const svn::CommitItems& items )
{
  svn::Paths urls;
  svn::Paths paths;
  for( svn::CommitItems::const_iterator it = items.begin(); it != items.end(); it++ )
  {
    const svn::CommitItemPtr item = *it;

    if( ! item->getUrl().isEmpty() )
      urls.push_back( item->getUrl() );  

    if( ! item->getPath().isEmpty() )
      paths.push_back( item->getPath() );        
  }
  
  QString urlPrefix = QString::fromUtf8(svn::Path::findPrefix(urls));
  QString pathPrefix = QString::fromUtf8(svn::Path::findPrefix(paths));
  
  _url->setText(urlPrefix);
  _path->setText(pathPrefix);

  for( svn::CommitItems::const_iterator it = items.begin(); it != items.end(); it++ )
  {
    const svn::CommitItemPtr item = *it;    
    QString itemStr = QString::fromUtf8(item->getUrl()).remove(0,urlPrefix.length()+1);

    CommitItemLvi* lvi = new CommitItemLvi( _list, item, itemStr );
  }
  
  logChanged();
}

void CommitDialog::setCondition( sc::Condition* cond )
{
  _cond = cond;
}

void CommitDialog::setResult( CommitResult* res )
{
  _result = res;
}

void CommitDialog::doubleClicked( QListViewItem* item, const QPoint& p, int i )
{
  CommitItemLvi* lvi  = dynamic_cast<CommitItemLvi*>(item);
  sc::String     path = lvi->getItem()->getPath();

  if( path.isEmpty() )
    return;

  DiffParam* param = new DiffParam( path, new svn::Revision(svn::Revision_Base), 
    path, new svn::Revision(svn::Revision_Working), 0 /*no peg*/, false, true,
    false, false, false );
  NullCmdResult* pcres = new NullCmdResult();

  WcModel model(_model);
  model.diff( param, pcres );
}

void CommitDialog::cursorChanged( int x, int y )
{
  _pos->setText( QString(_q("Line: %1  Column: %2")).arg(y+1,3).arg(x+1,3) );
}

void CommitDialog::logChanged()
{
  _ok->setEnabled( (!_edit->text().isEmpty()) ); 
}

void CommitDialog::logHistory( int index )
{
  // insertParagraph adds an unwanted newline at the end of our log text,
  // so we use insert.
  _edit->insert(_log->text(index));
}

void CommitDialog::list()
{
  _commitTriggered = true;
  
  _recurse->setEnabled(false);
  _keepLocks->setEnabled(false);
  _ls->setEnabled(false);
  
  CommitParam*   param = new CommitParam( _items, _recurse->isChecked(),
    _keepLocks->isChecked(), new Commit(this) );
  PostCmdResult* pcres = new PostCmdResult(_tid);

  WcModel model(_model);
  model.commit( param, pcres );
}

void CommitDialog::commit()
{
  _recurse->setEnabled(false);
  _keepLocks->setEnabled(false);
  _ls->setEnabled(false);
  _ok->setEnabled(false);

  // commit not yet triggered by list?
  if( _direct && ! _commitTriggered )
  {
    // then skip list and pass the log message
    sc::String log(_edit->text().utf8());

    CommitParam* param = new CommitParam( _items, _recurse->isChecked(),
      _keepLocks->isChecked(), new Commit(log) );
    PostCmdResult* pcres = new PostCmdResult(_tid);

    WcModel model(_model);
    model.commit( param, pcres );
  }

  finish(true);
  accept();
}

void CommitDialog::cancel()
{
  finish(false);
  reject();
}

void CommitDialog::finish( bool commit )
{
  sc::String log(_edit->text().utf8());

  if( ! log.isEmpty() )
  {
    CommitLogHistory* logHistory = _model->getCommitLogHistory();
    logHistory->addLog(log);
  }

  if( _result )
  {
    _result->setCommit(commit);
    _result->setLog(log);
  }

  if( _cond )
  {
    _cond->wakeAll();
  }
}
