/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "DiffDialog.h"
#include "RevisionWidget.h"
#include "ExternProvider.h"
#include "BaseModel.h"
#include "PostCmdResult.h"
#include "NullCmdResult.h"
#include "DiffSummarizeLvi.h"
#include "TextWindow.h"
#include "ScModel.h"
#include "commands/DiffParam.h"
#include "events/DiffSummarizeEvent.h"
#include "events/EventSupport.h"
#include "events/ScParamEvent.h"
#include "sublib/Gui.h"
#include "sublib/ExternButton.h"
#include "svn/DiffSummarize.h"

// qt
#include <qlayout.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qtooltip.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qhbox.h>
#include <qlistview.h>

// sys
#include <assert.h>



///////////////////////////////////////////////////////////////////////////////

class DiffDialogResultVisitor :
  public ParamVisitor<DiffParam>
{
public:
  DiffDialogResultVisitor(DiffDialog* w) : _w(w)
  {
  }

  void visit( DiffParam* p )
  {
    if( p->getPatch() )
    {
      QString title = _q("subcommander:patch ");
      title += p->getPathOrUrl1();

      TextWindow* tw = new TextWindow( title, _w->getFontSettings(), 0 );
      tw->loadText(QString::fromUtf8(p->getPatchFile()));
      tw->show();
    }
  }

private:
  DiffDialog* _w;
};

///////////////////////////////////////////////////////////////////////////////

class SummarizeBaton : public svn::DiffSummarizeBaton
{
public:
  SummarizeBaton( DiffDialog* dlg ) : _dlg(dlg)
  {
  }

  void summarize( svn::DiffSummarizePtr summarize )
  {
    postEvent( _dlg, new DiffSummarizeEvent(summarize) );
  }

private:
  DiffDialog* _dlg;
};

///////////////////////////////////////////////////////////////////////////////

DiffDialog::DiffDialog( BaseModel* model, bool folder, ExternProvider* p,
  QWidget *parent )
: super( parent, 0 ), TargetId(this), _model(model), _p(p), _folder(folder)
{
  setCaption( _q("subcommander:diff") );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    QGroupBox* gb = new QGroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("diff options: ") );
    gb->setInsideMargin(0);
    gb->setFlat(true);
    vbl->addWidget(gb);

    QGridLayout* gl = new QGridLayout(vbl,4,4);
    gl->setMargin(0);
    gl->setColStretch( 0, 0 );
    gl->setColStretch( 1, 1 );
    gl->setColStretch( 2, 3 );
    gl->setColStretch( 3, 0 );
    {
      {
        _rep1L = new QLabel(this);
        _rep1  = new QComboBox(this);
        _rep1B = new ExternButton(this);

        _rep1L->setBuddy(_rep1);
        _rep1L->setText( _q("&target:") );
        _rep1->setEditable(true);
        _rep1->setAutoCompletion(true);

        gl->addWidget( _rep1L, 1, 1 );
        gl->addWidget( _rep1,  1, 2 ); 
        gl->addWidget( _rep1B, 1, 3 ); 

        connect( _rep1, SIGNAL(activated(const QString&)),
          SLOT(checkButtons(const QString&)) );
        connect( _rep1, SIGNAL(highlighted(const QString&)),
          SLOT(checkButtons(const QString&)) );
        connect( _rep1, SIGNAL(textChanged(const QString&)),
          SLOT(checkButtons(const QString&)) );

        connect( _rep1B, SIGNAL(clicked()), SLOT(selectRep1Url()) );

        QToolTip::add( _rep1, _q("target for a diff between two revisions") );
      }
      {
        _type  = new QCheckBox(this);
        _rep2L = new QLabel(this);
        _rep2  = new QComboBox(this);
        _rep2B = new ExternButton(this);

        _rep2L->setBuddy(_rep2);
        _rep2L->setText( _q("&new target:") );
        _rep2->setEditable(true);
        _rep2->setAutoCompletion(true);

        gl->addWidget( _type,  2, 0 );
        gl->addWidget( _rep2L, 2, 1 );
        gl->addWidget( _rep2,  2, 2 ); 
        gl->addWidget( _rep2B, 2, 3 ); 

        connect( _type, SIGNAL(toggled(bool)), SLOT(changedDiffType(bool)) );
        connect( _rep2, SIGNAL(activated(const QString&)),
          SLOT(checkButtons(const QString&)) );
        connect( _rep2, SIGNAL(highlighted(const QString&)),
          SLOT(checkButtons(const QString&)) );
        connect( _rep2, SIGNAL(textChanged(const QString&)),
          SLOT(checkButtons(const QString&)) );

        connect( _rep2B, SIGNAL(clicked()), SLOT(selectRep2Url()) );

        QToolTip::add( _rep2, _q("new target for a diff between two paths") );

        _rep2L->setEnabled(false);
        _rep2->setEnabled(false);
        _rep2B->setEnabled(false);
      }

      QHBoxLayout* h0 = new QHBoxLayout;
      gl->addMultiCellLayout( h0, 3, 3, 0, 3 );
      {
        _rwPeg = new RevisionWidget(true,"NDS","HBCP",0,this);
        _rwPeg->setTitle( _q("target peg revision:") );
        h0->addWidget(_rwPeg);

        QToolTip::add( _rwPeg, _q("the targets name was target in this revision") );
      }

      QHBoxLayout* h1 = new QHBoxLayout;
      gl->addMultiCellLayout( h1, 4, 4, 0, 3 );
      {
        _rw1 = new RevisionWidget(false,"SDN","HBWCP",0,this);
        _rw1->setTitle( _q("target from revision:") );
        h1->addWidget(_rw1);

        _rw2 = new RevisionWidget(false,"SDN","HBWCP",0,this);
        _rw2->setTitle( _q("target to revision:") );
        h1->addWidget(_rw2);
      }
    }

    QHBoxLayout* h1 = new QHBoxLayout;
    vbl->addLayout(h1);
    {
      _recurse = new QCheckBox(_q("&recursive"),this);
      _recurse->setChecked(model->isCmdRecursive());
      h1->addWidget(_recurse);

      _ancestry = new QCheckBox(_q("&ancestry"),this);
      _ancestry->setChecked(true);
      h1->addWidget(_ancestry);

      _deleted = new QCheckBox(_q("&show deleted"),this);
      _deleted->setChecked(true);
      h1->addWidget(_deleted);
    }

    _summarize = new  QListView(this);
    _summarize->setItemMargin( 2 );
    _summarize->setShowToolTips( true );
    _summarize->setAllColumnsShowFocus(true);
    _summarize->addColumn( _q("text") );
    _summarize->addColumn( _q("prop") );
    _summarize->addColumn( _q("type") );
    _summarize->addColumn( _q("file/folder") );
    _summarize->setColumnAlignment( 0, Qt::AlignCenter );
    _summarize->setColumnAlignment( 1, Qt::AlignCenter );
    _summarize->setColumnAlignment( 2, Qt::AlignCenter );
    _summarize->setColumnAlignment( 3, Qt::AlignLeft );
    _summarize->setSortColumn( 3 );
    _summarize->setResizeMode(QListView::LastColumn);
    _summarize->setSelectionMode(QListView::Extended);
    _summarize->setHidden(!_folder);
    vbl->addWidget(_summarize);

    connect( _summarize, SIGNAL(selectionChanged()), SLOT(enableButtons()) );
    connect( _summarize, SIGNAL(doubleClicked(QListViewItem*,const QPoint&, int)),
      SLOT(doubleClicked(QListViewItem*,const QPoint&, int)) );

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _diff = new QPushButton(this);
      _diff->setText( _q("&Diff") );
      hu->addWidget(_diff);

      _patch = new QPushButton(this);
      _patch->setText( _q("P&atch") );
      _patch->setDefault(true);
      hu->addWidget(_patch);

      _sum = new QPushButton(this);
      _sum->setText( _q("&Summarize") );
      _sum->setHidden(!_folder);
      hu->addWidget(_sum);

      QPushButton* ca = new QPushButton(this);
      ca->setText( _q("&Close") );
      hu->addWidget(ca);

      hu->addSpacing(getSizeGripSpacing());

      connect( _diff, SIGNAL(clicked()), SLOT(xdiff()) );
      connect( _patch, SIGNAL(clicked()), SLOT(xpatch()));
      connect( _sum, SIGNAL(clicked()), SLOT(xsum()));
      connect( ca, SIGNAL(clicked()), SLOT(xclose()) );
    }
  }
}

DiffDialog::~DiffDialog()
{
  delete _p;
}

void DiffDialog::customEvent( QCustomEvent* ce )
{
  switch( ce->type() )
  {
  case ScDiffSummarizeEvent:
    {
      DiffSummarizeEvent* pe = (DiffSummarizeEvent*)ce;
      new DiffSummarizeLvi( _summarize, pe->getDiffSummarize() );
      break;
    }
  case ScParameterEvent:
    {
      DiffDialogResultVisitor v(this);
      ScParamEvent* pe = dynamic_cast<ScParamEvent*>(ce);
      if( ! pe->getError() )
      {
        pe->getParam()->accept(&v);
      }
      checkButtons("");
      setEnabled(true);
      break;
    }
  default:
    {
      printf( "DiffDialog: unknown custom event type %d!\n", ce->type() );
    }
  }
}

void DiffDialog::setPathOrUrl1( const QString& pathOrUrl )
{
  _rep1->insertItem( pathOrUrl, 0 );
  enableButtons();
}

void DiffDialog::setPathOrUrl2( const QString& pathOrUrl )
{
  _rep2->insertItem( pathOrUrl, 0 );
  enableButtons();
}

void DiffDialog::enablePathOrUrl1( bool enable )
{
  _rep1->setEnabled(enable);
  _rep1B->setEnabled(enable);
}

void DiffDialog::enablePathOrUrl2( bool enable )
{
  _rep2L->setEnabled(enable);
  _rep2->setEnabled(enable);
  _rep2B->setEnabled(enable);
  _type->setChecked(enable);
}

void DiffDialog::setRevision1( const svn::Revision* rev )
{
  _rw1->setRevision(rev);
}

void DiffDialog::setRevision2( const svn::Revision* rev )
{
  _rw2->setRevision(rev);
}

void DiffDialog::enableButtons()
{
  if( _folder )
  {
    QListViewItemIterator it( _summarize, QListViewItemIterator::Selected );
    _diff->setEnabled(it.current());
    _patch->setEnabled(true);
    _sum->setEnabled(true);
  }
  else
  {
    _diff->setEnabled(true);
    _patch->setEnabled(true);
    _sum->setEnabled(false);
  }
}

void DiffDialog::changedDiffType( bool )
{
  if( ! _type->isChecked() )
  {
    _rep2L->setEnabled(false);
    _rep2->setEnabled(false);
    _rep2B->setEnabled(false);

    _rwPeg->setEnabled(true);

    _rep1L->setText( _q("&target:") );
    QToolTip::add( _rep1, _q("target for a diff between two revisions") );
    _rw1->setTitle( _q("target from revision:") );
    _rw2->setTitle( _q("target to revision:") );
  }
  else if( _type->isChecked() )
  {
    _rep2L->setEnabled(true);
    _rep2->setEnabled(true);
    _rep2B->setEnabled(true);

    _rwPeg->setEnabled(false);

    _rep1L->setText( _q("&old target:") );
    QToolTip::add( _rep1, _q("old target for a diff between two paths") );
    _rw1->setTitle( _q("old target revision:") );
    _rw2->setTitle( _q("new target revision:") );
  }
}

FontSettings* DiffDialog::getFontSettings() const
{
  return _model->getModel()->getFontSettings();
}

bool DiffDialog::isFolder() const
{
  return _folder;
}

QListView* DiffDialog::getListView() const
{
  return _summarize;
}

void DiffDialog::xdiff()
{
  _diff->setDisabled(true);
  _sum->setDisabled(true);
  _patch->setDisabled(true);

  emit diff();
}

void DiffDialog::xpatch()
{
  _diff->setDisabled(true);
  _sum->setDisabled(true);
  _patch->setDisabled(true);

  emit patch();
}

void DiffDialog::xsum()
{
  _diff->setDisabled(true);
  _sum->setDisabled(true);
  _patch->setDisabled(true);

  _summarize->clear();

  emit summarize();
}

void DiffDialog::xclose()
{
  close();
  emit closed();
}

void DiffDialog::doubleClicked( QListViewItem* item, const QPoint& p, int c )
{
  xdiff();
}

void DiffDialog::selectRep1Url()
{
  sc::String res;

  if( _p->selectUrl( this, sc::String(_rep1->currentText().utf8()), res, ExternProvider::Dir) )
  {
    _rep1->insertItem( QString::fromUtf8(res), 0 );
  }
}

void DiffDialog::selectRep2Url()
{
  sc::String res;

  if( _p->selectUrl( this, sc::String(_rep2->currentText().utf8()), res, ExternProvider::Dir) )
  {
    _rep2->insertItem( QString::fromUtf8(res), 0 );
  }
}

void DiffDialog::checkButtons( const QString& /*ignored*/ )
{
  QString rep1 = _rep1->currentText();
  QString rep2 = _rep2->currentText();

  bool ok;

  if( _type->isChecked() ) 
  {
    // normal
    ok = ! rep1.isEmpty() && ! rep2.isEmpty();
  }
  else
  {
    // peg
    ok = ! rep1.isEmpty();
  }

  if( ok )
  {
    enableButtons();
  }
  else
  {
    _diff->setEnabled(false);
    _patch->setEnabled(false);
    _sum->setEnabled(false);
  }
}

DiffParam* DiffDialog::getParameters( DiffType type )
{
  bool patch     = false;
  bool summarize = false;
  SummarizeBaton* baton = NULL;

  switch( type )
  {
  case Patch:
    {
      patch = true;
      break;
    }
  case Summarize:
    {
      summarize = true;
      baton     = new SummarizeBaton(this);
      break;
    }
  default:
    break;
  }

  return new DiffParam(
    sc::String(_rep1->currentText().utf8()),
    _rw1->getRevision(),
    sc::String(_rep2->currentText().utf8()),
    _rw2->getRevision(),
    (!_type->isChecked()) ? _rwPeg->getRevision() : 0,
    _recurse->isChecked(),
    _ancestry->isChecked(),
    _deleted->isChecked(),
    summarize,
    patch,
    baton );
}

void DiffDialog::getParameters( DiffParams& params )
{
  QListViewItemIterator it( _summarize, QListViewItemIterator::Selected );
  while( it.current() )
  {
    DiffSummarizeLvi*     lvi = dynamic_cast<DiffSummarizeLvi*>(it.current());
    svn::DiffSummarizePtr p   = lvi->getDiffSummarize();
    
    if( ! p->isDir() )
    {
      DiffParam* param;

      if( _type->isChecked() ) // no peg
      {
        QString path1 = _rep1->currentText();
        QString path2 = _rep2->currentText();
        QString name  = QString::fromUtf8(p->getName());

        if( ! path1.contains(name) )
        {
          path1 += "/"; path1 += name;
          path2 += "/"; path2 += name;
        }

        param = new DiffParam(
          sc::String(path1.utf8()),
          _rw1->getRevision(),
          sc::String(path2.utf8()),
          _rw2->getRevision(),
          NULL,
          _recurse->isChecked(),
          _ancestry->isChecked(),
          _deleted->isChecked(),
          false,
          false,
          NULL );
      }
      else // peg
      {
        QString path = _rep1->currentText();
        QString name = QString::fromUtf8(p->getName());

        if( ! path.contains(name) )
        {
          path += "/"; path += name;
        }

        param = new DiffParam(
          sc::String(path.utf8()),
          _rw1->getRevision(),
          sc::String(path.utf8()),
          _rw2->getRevision(),
          _rwPeg->getRevision(),
          _recurse->isChecked(),
          _ancestry->isChecked(),
          _deleted->isChecked(),
          false,
          false,
          NULL);
      }

      params.push_back(param);
    }

    ++it;
  }
}
