#
# std.pm - Standard Perl subroutines
#
# Copyright (C) 2004 Russ W. Knize
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# e-mail: rknize@yahoo.com
#
# 09/16/2004 - rknize: Created
#
################################################################################

################################################################################
# Globals
################################################################################
$debug = 0;
$logging = 0;
$verbose = 0;


################################################################################
# printMsg(message) - print and/or log a message if in verbose/logging mode
#
#   message - message to print
################################################################################
sub printMsg
{
    if ($verbose) { print "$_[0]"; }
    if ($logging) { print LOG "$_[0]"; }
}


################################################################################
# printDbg(level, message) - print a message if at the appropriate debug level
#
#   message - message to print
################################################################################
sub printDbg
{
    return if ($_[0] > $debug);
    print $_[1];
}


################################################################################
# printError(message) - print an error to STDERR
#
#   message - message to print and log if logging is activated
################################################################################
sub printError
{
    print STDERR "ERROR: $_[0]\n";
    if ($logging) { print LOG "$_[0]\n"; }
}


################################################################################
# exitError(message) - print an error to STDERR and exit
#
#   message - message to print and log if logging is activated
################################################################################
sub exitError
{
    print STDERR "ERROR: $_[0]\n";
    if ($logging) { print LOG "$_[0]\n"; }
    exit 1;
}


################################################################################
# loadConfig(file, config) - read a configuration file
#
#   file - path to the configuration file
#   config - reference to configuration hash
#
# The file format requires one key/value pair per line, seperated by an equal
# sign (=) or colon (:).  All leading and trailing white space is stripped
# unless they are enclosed by double quotes (").  The quotes themselves will be
# removed.
#
# Returns: 1 on success, 0 on failure.
################################################################################
sub loadConfig
{
    my $file = shift(@_);
    my $hash = shift(@_);
    my $line = "";
    my $key;
    my $value;

    open(CONFIG, "$file") or return 0;

    $line = <CONFIG>;
    while ($line ne "")
    {
        if ($line =~ /^#/) { next; }
        ($key, $value) = split(/^\s*=\s*|^\s*:\s*/, $line, 2);
        if ($key eq "") { next; }
        $value =~ s/^\s|\s$|\n//g;
        $value =~ s/"//g;
        printDbg(2, "$key->$value\n");
        $hash->{$key} = $value;

        $line = <CONFIG>;
    }

    @l = keys(%config);
    printDbg(3, "@l\n");
    close(CONFIG);
    return 1;
}


################################################################################
# fileParts(path) - parse a path into its components (like parsefile(), but it
#                   works)
#
#   path - path to parse
#
# The path is broken down into its directory path, file name, and file
# extension.  The result is returned as a 3 element list of strings.  Any
# component not found in the input path is returned as an empty string.
#
# Returns: (file name, directory path, file extension)
################################################################################
sub fileParts
{
    my $path = $_[0];
    my $name;
    my $type;
    my $slash = 0;
    my $dot = 0;
    my $i;

    $slash = rindex($path, "/");
    $dot = rindex($path, ".");
    if ($slash > -1)
    {
        if ($dot > -1)
        {
            $name = substr($path, $slash + 1, $dot - $slash - 1);
            $type = substr($path, $dot + 1);
        }
        else
        {
            $name = substr($path, $slash + 1);
            $type = "";
        }
        $path = substr($path, 0, $slash + 1);
    }
    else
    {
        if ($dot > -1)
        {
            $name = substr($path, 0, $dot);
            $type = substr($path, $dot + 1);
        }
        else
        {
            $name = $path;
            $type = "";
        }
        $path = "";
    }

    return ($name, $path, $type);
}

################################################################################
# getHomePath() - get the user's home directory
#
# Returns: path of user's home directory on current platform
################################################################################
sub getHomePath
{
    if ($^O =~ /Win32/)
    {
        return $ENV{HOMEDRIVE} . $ENV{HOMEPATH};
    }
    else
    {
        return $ENV{HOME};
    }
}

1
