/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This file uses GNU getline() for safety */


#include "discover.h"
#include "utils.h"

#ifdef HAVE_LIBISAPNP
struct isa_dev {
  int board_num;
  char *board_id;
  int dev_num;
  char *dev_id;
} isa[100];

static void prepare_to_inspect(void);
static resource_inspector inspect_resource;
static void finish_inspection(void);


int free_irqs_dmas_detect(int irqs[], int dmas[]){
  long l_irq, l_dma;
  int i, j;
  int l_ret;
  interrogate_args l_interrog_args;
  interrogate_results l_interrog_results;

  l_interrog_args.m_non_fatal_error_callback_p = NULL;
  l_interrog_args.m_progress_report_callback_p = NULL;
  l_interrog_args.m_numcards = -1;        /* Default = autodetect     */
  l_interrog_args.m_readport = -1;        /* Default = autodetect     */
  l_interrog_args.m_reset_flag = 0;
  l_interrog_args.m_ignore_csum_flag = 0;
  l_interrog_args.m_debug_flag = 0;

  l_ret = interrogate_isapnp((interrogate_args *)&l_interrog_args, 
                            (interrogate_results *)&l_interrog_results);
  if(l_ret != 0){
    fprintf(stderr, _("!!! Error while trying to interrogate the ISA PnP cards !!!\n"));
    return -1;
  }/*endif*/

  l_ret = get_free_irqs_dmas((long *)&l_irq, (long *)&l_dma);
  if(l_ret != 0){
    fprintf(stderr, _("!!! Error while trying to get IRQ/DMA information !!!\n"));
    return -2;
  }/*endif*/
  
  for(i = 0, j = 0; i < 16; i++){
    if(l_irq & (1<<i)) {
      irqs[j] = i;
      j++;
    }/*endif*/
  }/*endif*/
  irqs[j] = -1;

  for(i = 0, j = 0; i < 8; i++){
    if(l_dma & (1<<i)){
      dmas[j] = i;
      j++;
    }/*endif*/
  }/*endif*/
  dmas[j] = -1;
  return 0;
}/*endfunc free_irqs_dmas_detect*/


extern struct isa_info *isa_detect(struct cards_lst *lst){
  char *line;
  char *board_id = NULL;
  char *dev_id = NULL;
  int board_num, dev_num;
  int found = 0;
  int i;
  size_t len = 0;
  
  FILE *f;
  
  /* I/O to interrogate_isapnp*/
  interrogate_args l_interrog_args;
  interrogate_results l_interrog_results;
  int l_ret;
  int l_alloc_result;
  
  struct cards_lst *bkup_lst;
  struct isa_info *result = (struct isa_info *)NULL;
  static struct isa_info *first = (struct isa_info *)NULL;
  
  if(first){
    return first;
  }/*endif*/

  if(debug){
    fprintf(stdout, "\nProbing ISA cards...\n");
  }/*endif*/
  
  /********************************************************************/
  /* First were're trying with isapnp if available...                 */
  /********************************************************************/
  if((!(exec_modprobe("isapnp", NULL)) || !(exec_modprobe("isa-pnp", NULL))) &&
     (f = fopen(PATH_PROC_ISAPNP, "r"))){
    if(debug){
      fprintf(stdout, "\tProbing ISAPNP via module ...\n");
    }/*endif*/
    while(getline(&line, &len, f) >= 0){
      if(!strncmp("Card ", line, 5)){
        board_id = (char *)my_malloc(8);
        sscanf(line, "Card %d '%7s[^']", &board_num, board_id);
      }else if (!strncmp("  Logical device", line, 16)) {
        if(!first){
          first = result = (struct isa_info *)
                                     my_malloc(sizeof(struct isa_info));
        }else{
          result->next = (struct isa_info *)
                                     my_malloc(sizeof(struct isa_info));
          result = result->next;
        }/*endif*/
        result->next = (struct isa_info *)NULL;
        found = 0;
        
        dev_id = (char *)my_malloc(8);
        sscanf(line, "  Logical device %d '%7s[^:]", &dev_num, dev_id);
        for(bkup_lst = lst; bkup_lst; bkup_lst = bkup_lst->next){
          if(bkup_lst->bus == ISA){
            if(!strcmp(dev_id, bkup_lst->dev_id)){
              result->board_id = (char *)my_malloc(8);
              result->dev_id = (char *)my_malloc(8);
              result->board_num = board_num - 1;        /*seems board isn't the same as card...*/
              result->board_id = strdup(board_id);
              result->dev_num = dev_num;
              result->dev_id = strdup(dev_id);
              result->vendor = bkup_lst->vendor;
              result->model = bkup_lst->model;
              result->type = bkup_lst->type;
              result->options = bkup_lst->options;
              result->modulename = bkup_lst->modulename;
              found = 1;
            }/*endif*/
          }/*endif*/
        }/*next bkup_lst*/
        if(!found){
          result->board_id = (char *)my_malloc(8);
          result->dev_id = (char *)my_malloc(8);
          result->board_id =strdup(board_id);
          result->board_num = board_num - 1;        /*seems board isn't the same as card...*/
          result->dev_id = strdup(dev_id);
          result->dev_num = dev_num;
          result->vendor = s_unknown;
          result->model = s_unknown;
          result->type = UNKNOWN_DEVICE;
          result->modulename = s_unknown;
        }/*endif*/
        if(debug){
          fprintf(stdout, "\t\tFound %s %s (%s)\n", 
               result->vendor, result->model, device2str(result->type));
        }/*endif*/
      }/*endif*/
      if(!strncmp("    Active port", line, 15)){
        sscanf(line, "    Active port %x", &result->io);
      }else if(!strncmp("    Active IRQ", line, 14)){
        sscanf(line, "    Active IRQ %d", &result->irq);
      }else if(!strncmp("    Active DMA", line, 14)){
        if(result->options && HAS_DMA16) {
          sscanf(line, "    Active DMA %d,%d", &result->dma8, &result->dma16);
        }else{
          sscanf(line, "    Active DMA %d", &result->dma8);
        }/*endif*/
      }/*endif*/
    }/*endwhile*/
    
    free(line);
    len = 0;
    fclose(f);
    return first;
  }/*endif*/


  /*However try with old method via libisapnp....*/
  /********************************************************************/
  /********************* ISA CARDS DETECTION **************************/
  /********************************************************************/
  if(debug){
    fprintf(stdout, "\tProbing ISAPNP via libispnp ...\n");
  }/*endif*/
  l_interrog_args.m_non_fatal_error_callback_p = NULL;
  l_interrog_args.m_progress_report_callback_p = NULL;
  l_interrog_args.m_numcards = -1;      /* autodetect                 */
  l_interrog_args.m_readport = -1;      /* autodetect                 */
  l_interrog_args.m_reset_flag = 0;     /* Don't reset cards          */
  l_interrog_args.m_ignore_csum_flag = 0; /* Take checksum into account */
#ifdef REALTIME
  l_interrog_args.m_realtime_timeout = 5L;
#endif
  l_interrog_args.m_debug_flag = 0;     /* No debug mode              */
  l_ret = interrogate_isapnp((interrogate_args *)&l_interrog_args,
                            (interrogate_results *)&l_interrog_results);
  if(l_ret != 0){
    fprintf(stderr, _("!!! Error while trying to interrogate the ISA PnP cards !!!\n"));
    return first;
  }/*endif*/
  l_alloc_result = 0;
  prepare_to_inspect();
  for_each_resource(l_interrog_results.m_resource_p,
            l_interrog_results.m_resource_count,
            inspect_resource, stdout, l_alloc_result);
  finish_inspection();

  for(i = 0; isa[i].board_id != NULL;i++) {
    if(!first){
      first = result = (struct isa_info *)
        my_malloc(sizeof(struct isa_info));
    }else{
      result->next = (struct isa_info *)
        my_malloc(sizeof(struct isa_info));
      result = result->next;
    }/*endif*/
    result->next = (struct isa_info *)NULL;
    found = 0;
    
    for(bkup_lst = lst; bkup_lst; bkup_lst = bkup_lst->next){
      if(bkup_lst->bus == ISA) {
        if(!strcmp(isa[i].dev_id, bkup_lst->dev_id)) {
          result->board_num = isa[i].board_num;
          result->board_id = isa[i].board_id;
          result->dev_num = isa[i].dev_num;
          result->dev_id = isa[i].dev_id;
          result->vendor = bkup_lst->vendor;
          result->model = bkup_lst->model;
          result->type = bkup_lst->type;
          result->options = bkup_lst->options;
          result->modulename = bkup_lst->modulename;
          found = 1;
        }/*endif*/
      }/*endif*/
    }/*next bkup_lst*/
    if(!found){
      result->board_num = isa[i].board_num;
      result->board_id = isa[i].board_id;
      result->dev_num = isa[i].dev_num;
      result->dev_id = isa[i].dev_id;
      result->vendor = s_unknown;
      result->model = s_unknown;
      result->type = UNKNOWN_DEVICE;
      result->modulename = s_unknown;
    }/*endif*/
  }/*next i*/
  return first;
}/*endfunc isa_detect*/


/**********************************************************************/
/* Auxiliary procedures - used by inspect_resource().                 */
/**********************************************************************/
static char *devidstr(unsigned char d1, unsigned char d2,
               unsigned char d3, unsigned char d4){
  static char resstr[] = "PNP0000";
  sprintf(resstr, "%c%c%c%x%x%x%x",
     'A' + (d1 >> 2) - 1, 'A' + (((d1 & 3) << 3) | (d2 >> 5)) - 1,
     'A' + (d2 & 0x1f) - 1, d3 >> 4, d3 & 0x0f, d4 >> 4, d4 & 0x0f);
  return resstr;
}/*endfunc devidstr*/


/**********************************************************************/
/* Illegal value, to catch failure to call prepare_to_inspect()       */
/**********************************************************************/
int st_res_index = -1;


/**********************************************************************/
/* Definitions of procedures for inspecting the resources, which are  */
/* called after interrogation of the ISA PnP cards.                   */
/**********************************************************************/
static void prepare_to_inspect(){
  st_res_index = 0;
}/*endfunc prepare_to_inspect*/


char *board_id;
int board_count = -1;
int dev_count = -1;
int count = -1;

static void inspect_resource(FILE *out_file,
                             struct resource *in_res_ptr,
                             int in_selected){
  st_res_index++;
  switch(in_res_ptr->type){
  case NewBoard_PSEUDOTAG:
    {
      int ll_csn = in_res_ptr->start;
      board_count++;
      dev_count = -1;
      board_id = (char *) my_malloc(8);
      strcpy(board_id, devidstr(serial_identifier[ll_csn][0],
                                serial_identifier[ll_csn][1],
                                serial_identifier[ll_csn][2],
                                serial_identifier[ll_csn][3]));
    }/*end block*/
    break;
  case LogDevId_TAG:
    count++;
    dev_count++;
    isa[count].board_num = board_count;
    isa[count].board_id = board_id;
    isa[count].dev_num = dev_count;
    isa[count].dev_id = (char *)my_malloc(8);
    strcpy(isa[count].dev_id, devidstr(in_res_ptr->data[0],
                                       in_res_ptr->data[1], 
                                       in_res_ptr->data[2], 
                                       in_res_ptr->data[3]));
    break;
  case StartDep_TAG:
    break;
  case EndDep_TAG:
    for_each_resource(in_res_ptr->alternatives[in_res_ptr->value].resources,
                      in_res_ptr->alternatives[in_res_ptr->value].len,
                      inspect_resource, out_file, in_selected);
    break;
  case ANSIstr_TAG:
    break;
  case End_TAG:
    count++;
    isa[count].board_id = NULL;
    isa[count].dev_id = NULL;
    break;
  }/*endselect*/
}/*endfunc */


static void finish_inspection(void){
}/*endfunc finish_inspection*/

#else

extern struct isa_info *isa_detect(struct cards_lst *lst){
  return NULL;
}/*endfunc isa_detect*/

#endif
