/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include <QtCore/QList>
#include <QtCore/QString>
#include "QtTapioca/ContactBase"
#include "QtTapioca/Avatar"
#include "QtTapioca/Handle"

namespace QtTapioca {

class ContactBasePrivate {
public:
    ContactBasePrivate()
        : currentToken("")
    {
    }
    ~ContactBasePrivate()
    {
    }

    QString currentToken;
    QString alias;
};

};

using namespace QtTapioca;

ContactBase::ContactBase(org::freedesktop::Telepathy::Connection *telepathyConn,
                         org::freedesktop::Telepathy::ConnectionAvatarsInterface  *iAvatar,
                         org::freedesktop::Telepathy::ConnectionPresenceInterface *iPresence,
                         org::freedesktop::Telepathy::ConnectionAliasingInterface *iAliasing,
                         org::freedesktop::Telepathy::ConnectionCapabilitiesInterface *iCapabilities,
                         Handle *contact_handle,
                         QObject *parent)
    : ChannelTarget(contact_handle, parent),
      d(new ContactBasePrivate()),
      telepathyIAvatar(iAvatar),
      telepathyIPresence(iPresence),
      telepathyIAliasing(iAliasing),
      telepathyICapabilities(iCapabilities),
      m_presence(ContactBase::Offline)
{
    if (telepathyIAvatar)
        QObject::connect(telepathyIAvatar, SIGNAL(AvatarUpdated(uint,const QString&)), this, SLOT(onAvatarUpdated(uint, const QString&)));

    if (telepathyIPresence) {
        QObject::connect(telepathyIPresence, SIGNAL(PresenceUpdate(org::freedesktop::Telepathy::PresenceStateInTimeMap)), this,
                         SLOT(onPresenceUpdated(org::freedesktop::Telepathy::PresenceStateInTimeMap)));
        telepathyIPresence->RequestPresence(QList<uint>() << handle()->id());
    }

    if (telepathyIAliasing)
        QObject::connect(telepathyIAliasing, SIGNAL(AliasesChanged(org::freedesktop::Telepathy::AliasInfoList)), this,
                         SLOT(onAliasChanged(org::freedesktop::Telepathy::AliasInfoList)));

//    if (telepathyICapabilities)
//        QObject::connect(telepathyICapabilities, SIGNAL(capabilitiesChanged(uint)), this, SLOT(onCapabilitiesChanged(uint)));
}

ContactBase::~ContactBase()
{
    delete d;
}

ContactBase::Presence ContactBase::presence() const
{
    return m_presence;
}

QString ContactBase::presenceMessage() const
{
    return m_presenceMessage;
}

QString ContactBase::alias() const
{
    if ((d->alias.isEmpty()) && (telepathyIAliasing)) {
        QStringList aliases = telepathyIAliasing->RequestAliases(QList<uint>() << handle()->id());
        d->alias = aliases.first();
    }

    return d->alias;
}

//TODO
ContactBase::Capability ContactBase::capabilities() const
{
    return None;
}

void ContactBase::requestAvatar()
{
    if (!telepathyIAvatar)
        return;

    QString newToken;
    QStringList tokens = telepathyIAvatar->GetAvatarTokens(QList<uint>() << handle()->id());

    if (!tokens.isEmpty()) {
        newToken = tokens.first();
        if (d->currentToken != newToken)
            d->currentToken = newToken;

        QString mime;
        QByteArray data = telepathyIAvatar->RequestAvatar(handle()->id(), mime);

        Avatar *avatar = new Avatar(data, mime, newToken);

        if (avatar)
            emit avatarReceived(this, avatar);
    }
}

QString ContactBase::currentAvatarToken()
{
    if ((d->currentToken.isEmpty()) && (telepathyIAvatar)) {
        QStringList tokens = telepathyIAvatar->GetAvatarTokens(QList<uint>() << handle()->id());
        if (!tokens.isEmpty())
            d->currentToken = tokens.first();
    }

    return d->currentToken;
}

void ContactBase::onPresenceUpdated(org::freedesktop::Telepathy::PresenceStateInTimeMap newPresence)
{
    if (newPresence.contains(handle()->id())) {
        org::freedesktop::Telepathy::PresenceStateInTime stateInTime = newPresence[handle()->id()];
        m_presence = strToPresenceEnum(stateInTime.state.begin().key());
        m_presenceMessage = (stateInTime.state.begin().value())["message"].toString();

        emit presenceUpdated(this, m_presence, m_presenceMessage);
    }
}

ContactBase::Presence ContactBase::strToPresenceEnum(const QString &presenceStr)
{
    ContactBase::Presence presence;
    if (presenceStr == "available")
        presence = ContactBase::Available;
    else if (presenceStr == "away" ||
             presenceStr == "brb")
        presence = ContactBase::Away;
    else if (presenceStr == "xa")
        presence = ContactBase::XA;
    else if (presenceStr == "busy" ||
             presenceStr == "dnd")
        presence = ContactBase::Busy;
    else if (presenceStr == "hidden")
        presence = ContactBase::Hidden;
    else
        presence = ContactBase::Offline;

    return presence;
}

const QString ContactBase::presenceEnumToStr(ContactBase::Presence num)
{
    QString presenceStr;
    switch (num) {
        case ContactBase::Available:
            presenceStr = "available";
            break;
        case ContactBase::Away:
            presenceStr = "away";   //Could be "brb"
            break;
        case ContactBase::XA:
            presenceStr = "xa";
            break;
        case ContactBase::Busy:
            presenceStr = "dnd";   //Could be "busy"
            break;
        case ContactBase::Hidden:
            presenceStr = "hidden";
            break;
        default:
            presenceStr = "offline";
            break;
    }

    return presenceStr;
}

void ContactBase::onAvatarUpdated(uint contact, const QString &newAvatarToken)
{
    if (handle()->id() == contact)
        if (d->currentToken != newAvatarToken) {
            d->currentToken = newAvatarToken;
            emit avatarUpdated(this, d->currentToken);
    }
}

void ContactBase::onAliasChanged(org::freedesktop::Telepathy::AliasInfoList aliases)
{
    org::freedesktop::Telepathy::AliasInfo aliasInfo;

    foreach (aliasInfo, aliases)
        if (aliasInfo.contactHandle == handle()->id()) {
            d->alias = aliasInfo.newAlias;
            emit aliasChanged(this, d->alias);
        }
}

void ContactBase::onCapabilitiesChanged(org::freedesktop::Telepathy::CapabilitiesChangedInfoList caps)
{
    //TODO
}
