/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include <QtCore/QPointer>

#include "QtTapioca/TextChannel"
#include "QtTapioca/Contact"

#include <QtTelepathy/Client/ChannelText>

namespace QtTapioca {

class TextChannelPrivate {
public:
    TextChannelPrivate(Connection *conn, const QString &serviceName, const QString &objPath)
    {
        ch = new org::freedesktop::Telepathy::ChannelText(serviceName, objPath, QDBusConnection::sessionBus());
    }
    ~TextChannelPrivate()
    {
        delete ch;
    }

    Connection *conn;
    org::freedesktop::Telepathy::ChannelText *ch;
};

}

using namespace QtTapioca;

TextChannel::TextChannel(Connection *conn, const QString &serviceName, const QString &objPath, ChannelTarget *target, QObject *parent)
    : Channel(conn, serviceName, objPath, Channel::Text, target, parent),
      d(new TextChannelPrivate(conn, serviceName, objPath))
{
    QObject::connect(d->ch, SIGNAL(Received(uint, uint, uint, uint, uint, const QString &)),
                     this, SLOT(onMessageReceived(uint, uint, uint, uint, uint, const QString &)));
    QObject::connect(d->ch, SIGNAL(SendError(uint, uint, uint, const QString &)),
                     this, SLOT(onSendError(uint, uint, uint, const QString &)));
    QObject::connect(d->ch, SIGNAL(Sent(uint, uint, const QString &)),
                     this, SLOT(onSent(uint, uint, const QString &)));
}

TextChannel::~TextChannel()
{
}

void TextChannel::sendMessage(const QString &contents)
{
    d->ch->Send(static_cast<uint>(TextChannel::Message::Normal), contents);
}

void TextChannel::sendMessage(const Message &message)
{
    d->ch->Send(static_cast<uint>(message.type()), message.contents());
}

QList<TextChannel::Message> TextChannel::pendingMessages() const
{
    QList<TextChannel::Message> ret;
    org::freedesktop::Telepathy::TextMessageInfo msgInfo;
    org::freedesktop::Telepathy::TextMessageInfoList msgs = d->ch->ListPendingMessages(false);

    foreach (msgInfo, msgs)
        ret << Message(msgInfo.message, msgInfo.timestamp, static_cast<TextChannel::Message::Type>(msgInfo.type), msgInfo.id, true);

    return ret;
}

void TextChannel::acknowledge(const Message &message)
{
    if ((message.m_pending) && (message.m_id >= 0)) {
        QList<uint> ids;
        ids << message.m_id;
        d->ch->AcknowledgePendingMessages(ids);
        message.m_pending = false;
    }
}

void TextChannel::onMessageReceived(uint id, uint timestamp, uint sender, uint type, uint flags, const QString &text)
{
    Message msg(text, timestamp, static_cast<TextChannel::Message::Type>(type), id, true);
    emit messageReceived(msg);
}

void TextChannel::onSendError(uint error, uint timestamp, uint type, const QString &text)
{
    Message msg(text, timestamp, static_cast<TextChannel::Message::Type>(type));
    emit messageDeliveryError(msg, static_cast<TextChannel::Message::DeliveryError>(error));
}

void TextChannel::onSent(uint timestamp, uint type, const QString &text)
{
    Message msg(text, timestamp, static_cast<TextChannel::Message::Type>(type));
    emit messageSent(msg);
}

TextChannel::Message::Message(const QString &contents)
    : m_contents(contents),
      m_timestamp(0),
      m_type(TextChannel::Message::Normal),
      m_id(-1),
      m_pending(false)
{
}

TextChannel::Message::Message(const QString &contents, uint timestamp, TextChannel::Message::Type type, int id, bool pending)
    : m_contents(contents),
      m_timestamp(timestamp),
      m_type(type),
      m_id(id),
      m_pending(pending)
{
}

TextChannel::Message::~Message()
{
}

