/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-channel.h:
 * GObject wrapper for D-Bus method calls to org.freedesktop.Telepathy.Channel
 * 
 * Copyright (C) 2006, 2007 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef __TI_CHANNEL_H__
#define __TI_CHANNEL_H__

#include <glib-object.h>

/*
 * Type macros.
 */

typedef struct _TIChannel TIChannel;
typedef struct _TIChannelClass TIChannelClass;

#define TI_TYPE_CHANNEL               ti_channel_get_type()
#define TI_CHANNEL(object)            (G_TYPE_CHECK_INSTANCE_CAST ((object), TI_TYPE_CHANNEL, TIChannel))
#define TI_CHANNEL_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST ((klass), TI_TYPE_CHANNEL, TIChannelClass))
#define TT_IS_CHANNEL(object)         (G_TYPE_CHECK_INSTANCE_TYPE ((object), TI_TYPE_CHANNEL))
#define TT_IS_CHANNEL_CLASS(klass)    (G_TYPE_CHECK_CLASS_TYPE ((klass), TI_TYPE_CHANNEL))
#define TI_CHANNEL_GET_CLASS(object)  (G_TYPE_INSTANCE_GET_CLASS ((object), TI_TYPE_CHANNEL, TIChannelClass))

struct _TIChannel {
    GObject parent;
};

GType ti_channel_get_type (void);

/**
 * Creates a new channel
 */
TIChannel* ti_channel_new (const gchar* service_name, const gchar* obj_path);

/**
 * Get Service Name
 *
 */
const gchar* ti_channel_get_service_name (TIChannel* channel);

/**
 * Get Object Path
 *
 */
const gchar* ti_channel_get_object_path (TIChannel* channel);

/**
 * Get Interfaces
 *
 * @return A NULL-terminated array of strings containing interfaces' names. Must be freed after use.
 */
gchar** ti_channel_get_interfaces (TIChannel* channel);

/**
 * Get Channel Type
 *
 * @return Channel type. Must be freed after use.
 */
gchar* ti_channel_get_channel_type (TIChannel* channel, GError** error);

/**
 * Get Handle
 *
 */
void ti_channel_get_handle (TIChannel* channel, guint* handle_type, guint* handle_number, GError** error);

/**
 * Close
 *
 */
void ti_channel_close (TIChannel* channel, GError** error);

/**
 * AddMembers method from Group interface
 *
 */
void ti_channel_igroup_add_members (TIChannel* channel, GArray* members, const gchar* message, GError** error);

/**
 * GetGroupFlags method from Group interface
 *
 * @return Group flags.
 */
guint ti_channel_igroup_get_group_flags (TIChannel* channel, GError** error);

/**
 * GetMembers method from Group interface
 *
 * @return Array of handles (guint*). Must be freed after use.
 */
GArray* ti_channel_igroup_get_members (TIChannel* channel, GError** error);

/**
 * GetLocalPendingMembers method from Group interface
 *
 * @return Array of handles (guint*). Must be freed after use.
 */
GArray* ti_channel_igroup_get_local_pending_members (TIChannel* channel, GError** error);

/**
 * GetRemotePendingMembers method from Group interface
 *
 * @return Array of handles (guint*). Must be freed after use.
 */
GArray* ti_channel_igroup_get_remote_pending_members (TIChannel* channel, GError** error);

/**
 * RemoveMembers method from Group interface
 *
 */
void ti_channel_igroup_remove_members (TIChannel* channel, GArray* members, const gchar* message, GError** error);

/**
 * Send method from Type.Text interface
 *
 */
void ti_channel_typetext_send (TIChannel* channel, guint type, const gchar* text, GError** error);

/**
 * ListPendingMessages method from Type.Text interface
 *
 */
void ti_channel_typetext_list_pending_messages (TIChannel* channel, gboolean clear, GPtrArray** messages, GError** error);

/**
 * AcknowledgePendingMessages method from Type.Text interface
 *
 */
void ti_channel_typetext_acknowledge_pending_messages (TIChannel* channel, GArray* ids, GError** error);

/**
 * ListRooms method from Type.RoomList interface
 *
 */
void ti_channel_typeroomlist_list_rooms (TIChannel* channel, GError** error);

/**
 * GetListingRooms method from Type.RoomList interface
 *
 */
gboolean ti_channel_typeroomlist_get_listing_rooms (TIChannel* channel, GError** error);

#endif
