/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-add-members.c:
 * Dialog to add members to a Channel.Interface.Group
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-add-members.h"
#include "ti-util.h"
#include "ti-config.h"
#include "ti-handle-list-editor.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgAddMembers, ti_dlg_add_members, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgAddMembersPrivate {
    gboolean disposed;

    GtkWindow* parent;

    GladeXML* glade_xml;

    TIHandleListEditor* handle_list_editor;

    GtkWidget* dialog;

    GtkTextBuffer* textbuffer_message;
};
typedef struct _TIDlgAddMembersPrivate TIDlgAddMembersPrivate;

#define TI_DLG_ADD_MEMBERS_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_ADD_MEMBERS, TIDlgAddMembersPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);
static gchar* _ti_dlg_add_members_get_message_text (TIDlgAddMembers* dlg_add_members);
static void _ti_dlg_add_members_setup (TIDlgAddMembers* dlg_add_members);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_add_members_dispose (GObject *object)
{
    TIDlgAddMembers *dlg_add_members = TI_DLG_ADD_MEMBERS (object);
    TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (dlg_add_members);

    if (priv->disposed) {
        return;
    } else {
        priv->disposed = TRUE;
    }

    if (priv->glade_xml != NULL) {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_add_members_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_add_members_class_init (TIDlgAddMembersClass *ti_dlg_add_members_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_add_members_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_add_members_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_add_members_class, sizeof (TIDlgAddMembersPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_add_members_init (TIDlgAddMembers *ti_dlg_add_members)
{
    TIDlgAddMembersPrivate* priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (ti_dlg_add_members);

    priv->disposed = FALSE;
}

/**
 * Returns a new instance.
 */
TIDlgAddMembers*
ti_dlg_add_members_new (GtkWindow* parent, TIHandleMapper* handle_mapper) 
{
    TIDlgAddMembers* dialog = NULL;
    TIDlgAddMembersPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	dialog = g_object_new (TI_TYPE_DLG_ADD_MEMBERS, NULL);

    priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (dialog);
    priv->parent = parent;
    priv->handle_list_editor = TI_HANDLE_LIST_EDITOR (ti_handle_list_editor_new (handle_mapper));

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-add-members.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    _ti_dlg_add_members_setup (dialog);

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean ti_dlg_add_members_run (TIDlgAddMembers* dlg_add_members, GArray* in_handles, GArray** out_handles, gchar** message)
{
    TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (dlg_add_members);
    gint result;

    g_assert (out_handles != NULL);
    g_assert (message != NULL);

    ti_handle_list_editor_set (priv->handle_list_editor, in_handles);

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    *message = _ti_dlg_add_members_get_message_text (dlg_add_members);

    *out_handles = ti_handle_list_editor_get (priv->handle_list_editor);

    if (*out_handles == NULL)
    {
        return FALSE;
    }
    else
    {
        return result == GTK_RESPONSE_OK;
    }
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

/**
 * Get Message Text
 */
static gchar*
_ti_dlg_add_members_get_message_text (TIDlgAddMembers* dlg_add_members)
{
    TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (dlg_add_members);
    GtkTextIter start_iter;
    GtkTextIter end_iter;

    gtk_text_buffer_get_start_iter (priv->textbuffer_message, &start_iter);
    gtk_text_buffer_get_end_iter (priv->textbuffer_message, &end_iter);

    return gtk_text_buffer_get_text (priv->textbuffer_message, &start_iter, &end_iter, FALSE);
}

/**
 * Setup
 */
static void
_ti_dlg_add_members_setup (TIDlgAddMembers* dlg_add_members)
{
    TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (dlg_add_members);
    GtkWidget* widget;

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_add_members");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    // Handle List Editor
    widget = glade_xml_get_widget (priv->glade_xml, "container_handle_list_editor");
    g_assert (widget != NULL && GTK_IS_CONTAINER (widget));
    gtk_container_add (GTK_CONTAINER (widget), GTK_WIDGET (priv->handle_list_editor));

    // Text View Message
    widget = glade_xml_get_widget (priv->glade_xml, "textview_message");
    g_assert (widget != NULL && GTK_IS_TEXT_VIEW (widget));
    priv->textbuffer_message = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));
}
