/*
 * QtTelepathy, the Tapioca Qt4 Telepathy Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <QtCore/QtCore>
#include <QtCore/QString>
#include <QtTelepathy/Client/Connection>
#include <QtTelepathy/Client/ConnectionManager>
#include <QtTelepathy/Client/ChannelGroupInterface>

#include "test.h"

using namespace org::freedesktop::Telepathy;

static void registerTypes()
{
    static bool registered = false;
    if (!registered) {
        qDBusRegisterMetaType<org::freedesktop::Telepathy::NVPairs>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::ParameterDefinition>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::ParameterDefinitionList>();
        registered = true;
    }
}

int main(int argc, char **argv)
{
    if (argc < 3) {
        return 1;
    }

    // create the QApplication object
    QCoreApplication app(argc, argv);

    if (!QDBusConnection::sessionBus().isConnected()) {
        fprintf(stderr, "Cannot connect to the D-BUS session bus.\n"
                "To start it, run:\n"
                "\teval `dbus-launch --auto-syntax`\n");
        return 1;
    }

    org::freedesktop::Telepathy::NVPairs params;
    QString account(argv[1]);
    if (account.endsWith("@gmail.com") || account.endsWith("@googlemail.com")) {
        params["server"] = QVariant("talk.google.com");
        params["old-ssl"] = QVariant(true);
        params["ignore-ssl-errors"] = QVariant(true);
        params["port"] = QVariant((uint) 5223);
    }
    else {
        if (argc < 5) {
            qDebug() << "usage: test username@host password server port";
            return 1;
        }
        params["server"] = QVariant(argv[3]);
        params["port"] = QVariant(QString(argv[4]).toUInt());
    }

    registerTypes();

    QDBusConnection bus = QDBusConnection::sessionBus();
    QDBusObjectPath objPath;
    params["account"] = QVariant(argv[1]);
    params["password"] = QVariant(argv[2]);

    ConnectionManager connMgr("org.freedesktop.Telepathy.ConnectionManager.gabble",
        "/org/freedesktop/Telepathy/ConnectionManager/gabble", bus);
    QString service = connMgr.RequestConnection("jabber", params, objPath);
    if (service.isEmpty())
        return 1;

    qDebug() << "created connection with service" << service << "and object path" << objPath.path() << "";
    Connection conn(service, objPath.path(), bus);
    MessageFilter messageFilter(&conn, bus);
    QObject::connect(&conn, SIGNAL(StatusChanged(uint,uint)), &messageFilter, SLOT(OnStatusChanged(uint,uint)));
    conn.Connect();

    return app.exec();
}

MessageFilter::MessageFilter(Connection *c, const QDBusConnection &b, QObject *parent)
    : QObject(parent),
      conn(c),
      bus(b)
{
}

MessageFilter::~MessageFilter()
{
}

void MessageFilter::OnStatusChanged(uint status, uint reason)
{
    qDebug() << "Connection state changed to" << status << "- reason" << reason;
    if (status == CONNECTION_STATUS_CONNECTED) {
        QStringList args;
        args << "subscribe" /* << "publish" */;
        QList<uint> handles = conn->RequestHandles(HANDLE_TYPE_LIST, args);
        uint handle;
        foreach (handle, handles) {
            QDBusObjectPath objPath = conn->RequestChannel("org.freedesktop.Telepathy.Channel.Type.ContactList", HANDLE_TYPE_LIST, handle, true);
            ChannelGroupInterface contactList(conn->service(), objPath.path(), bus);
            QList<uint> members = contactList.GetMembers();
            QStringList members_str = conn->InspectHandles (HANDLE_TYPE_CONTACT, members);
            QString member;
            foreach (member, members_str)
                qDebug() << "member:" << member;
        }
    }
}

