/*
 * dvi.h - a heder file for DVIlib
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-2001  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __DVI_H__
#define __DVI_H__



/* 
 * DATA TYPES
 */
typedef struct dvi_s_file          *DVI;
typedef struct dvi_s_device        *DVI_DEVICE;
typedef struct dvi_s_bitmap        *DVI_BITMAP;
typedef struct dvi_s_pixmap_rgb    *DVI_PIXMAP_RGB;
typedef struct dvi_s_pixmap_cmyk   *DVI_PIXMAP_CMYK;
typedef struct dvi_s_graymap       *DVI_GRAYMAP;
typedef struct dvi_s_context       *DVI_CONTEXT;
typedef struct dvi_s_property      *DVI_PROPERTY;
typedef struct dvi_s_color         *DVI_COLOR;
typedef struct dvi_s_frame_buffer  *DVI_FRAME_BUFFER;


/* Return Code of draw_page/draw_context  
 * (It is guaranteed that no error occured if return value is non-negative.
 */
#define DVI_DRAW_ERROR        -2
#define DVI_DRAW_INTERRUPTED  -1
#define DVI_DRAW_OK            0
#define DVI_DRAW_LAST_PAGE     1

/*
 * ERROR CODE
 */
enum dvi_error {
  DVI_ERR_INTERNAL = 1,     /* internal error */
  DVI_ERR_NO_MEMORY,        /* No memory */
  DVI_ERR_CANT_OPEN,        /* Can't open a file */
  DVI_ERR_ILL_FORMAT,       /* Illegal DVI file format */
  DVI_ERR_ILL_VERSION,      /* Illeganl DVI file version */
  DVI_ERR_NAME_LONG,        /* Font name is too long */
  DVI_ERR_ILL_PAGE_NO,      /* Illegal page number */
  DVI_ERR_STACK_UNDERFLOW,  /* Stack underflow at DVI interpreter */
  DVI_ERR_STACK_OVERFLOW,   /* Stack overflow at DVI interpreter */
  DVI_ERR_NO_METRIC,        /* No font metric */
  DVI_ERR_NO_GLYPH,         /* No font glyph */
  DVI_ERR_CHECKSUM,         /* Fontfile checksum inconsistency */
  DVI_ERR_NOT_ACTIVE,       /* Non-active DVI object */
  DVI_ERR_FONT_INIT,        /* Font: Initialization error */
  DVI_ERR_FONT_NOT_FOUND,   /* Font: not found */
  DVI_ERR_FONT_CANNOT_OPEN, /* Font: cannot opened */
  DVI_ERR_FONT_NO_GLYPH,    /* Font: no glyph */
  DVI_ERR_INTERRUPTED       /* Interrupted */
};


/*
 * DVI PROPERTY
 */
enum dvi_property_id {
  /* Print font names */
  DVI_PROP_PRINT_FONTS = 0,
  /* Print font names that are not found */
  DVI_PROP_PRINT_MISSING_FONTS,
  /* DVI extension for Japanese TeX by ASCII Coop. */
  DVI_PROP_ASCII_JTEX,
  /* Enlarge/shrink font if not a font of correct size is not found. */
  DVI_PROP_SCALE_FONT_IF_NOT_EXIST,
  /* LaTeX 2.09 epsf.sty or LaTeX 2e graphic[sx].sty for EPS figures.
     LaTeX 2e graphic[sx].sty is default property. */
  DVI_PROP_LATEX2E_GRAPHICS_STY,
  DVI_PROP_LATEX209_EPSF_STY,
  /* Skip rendering characters on output devices. */
  DVI_PROP_SKIP_RENDERING,
  /* Skip processing special DVI instructions (e.g., rendering EPS files). */
  DVI_PROP_SKIP_SPECIALS,
  /* Report fonts. */
  DVI_PROP_LIST_FONTS,
  /* Report missing fonts. */
  DVI_PROP_LIST_MISSING_FONTS,
  /* Defer opening fonts until it is necessary. */
  DVI_PROP_DELAYED_FONT_OPEN,
  /* Asynchronous Ghostscript invocation */
  DVI_PROP_ASYNC_GS_INVOCATION,
  /* Incremental EPS display */
  DVI_PROP_INCREMENTAL_EPS_DISPLAY
};
struct dvi_s_property {
  char   props[32];
};
extern DVI_PROPERTY  DVI_property_alloc(void);
extern DVI_PROPERTY  DVI_property_alloc_default(void);
extern void          DVI_property_release(DVI_PROPERTY);
extern int           DVI_property_update(DVI,DVI_DEVICE,DVI_PROPERTY);
#define  DVI_PROPERTY_ALLOC()           DVI_property_alloc()
#define  DVI_PROPERTY_ALLOC_DEFAULT()   DVI_property_alloc_default()
#define  DVI_PROPERTY_RELEASE(p)        DVI_property_release(p)
#define  DVI_PROPERTY_TEST(p,i)         ((p)->props[i] == 1)
#define  DVI_PROPERTY_SET(p,i)          ((p)->props[i] = 1)
#define  DVI_PROPERTY_UNSET(p,i)        ((p)->props[i] = 0)
#define  DVI_PROPERTY_UPDATE(dvi,dev,p) DVI_property_update(dvi,dev,p)


/*
 * BITMAP
 */
struct dvi_s_bitmap {
  long           width, height;
  long           raster;
  unsigned char  *bitmap;
};

/*
 * PIXMAP (RGB)
 */
struct dvi_s_pixmap_rgb {
  long           width, height;
  long           raster;
  int            max;
  unsigned char  *bitmap;
};

/*
 * PIXMAP (CMYK)
 */
struct dvi_s_pixmap_cmyk {
  long           width, height;
  long           raster;
  int            max;
  unsigned char  *bitmap;
};

/*
 * GRAYMAP
 */
struct dvi_s_graymap {
  long           width, height;
  long           raster;
  int            max;
  unsigned char  *bitmap;
};

/*
 * Color
 */
struct dvi_s_color {
  int     model;
  double  c1, c2, c3, c4;
  char    *cname;
  /*   model  |  c1 c2 c3 c4   cname
       -------+--------------------
       RBG    |  R  G  B  -     - 
       CYMK   |  C  Y  M  K     -
       GRAY   |  GS -  -  -     -
       NAMED  |  -  -  -  -    NAME      */ 
  char        *color_name;
  DVI_COLOR   next; 
};



/*
 * DEVICE OBJECT
 */
struct dvi_s_device {
  /** DEVICE PARAMETERS **/ 
  double      h_dpi;
  double      v_dpi;
  /** PAPER SIZE (in inch) **/ 
  double      paper_h;
  double      paper_w;
  /** DEVICE CONTOL METHODS **/ 
  /* drawing */
  void  (*put_bitmap)(DVI_DEVICE,DVI,DVI_BITMAP,int,long,long,long,long);
  void  (*put_rectangle)(DVI_DEVICE,DVI,long,long,long,long);
  void  (*put_pixmap_rgb)(DVI_DEVICE,DVI,DVI_PIXMAP_RGB,
			  int,long,long,long,long);
  void  (*put_pixmap_cmyk)(DVI_DEVICE,DVI,DVI_PIXMAP_CMYK,
			   int,long,long,long,long);
  void  (*put_graymap)(DVI_DEVICE,DVI,DVI_GRAYMAP,
		       int,long,long,long,long);
  /* special */
  int   (*special_command_hook)(DVI_DEVICE,DVI,char*,long,long);
  int   (*special_command_undef)(DVI_DEVICE,DVI,char*,long,long);
  int   (*draw_ps_figures)(DVI_DEVICE,DVI);
  void  (*before_ps_figure)(DVI_DEVICE,DVI,char*,long,long);
  void  (*after_ps_figure)(DVI_DEVICE,DVI);
  /* incremental eps display */
  int   (*ps_figure_display_step)(DVI_DEVICE,DVI,long,long);
  /* ghostscript control */
  char *(*gs_program_path)(DVI_DEVICE,DVI);
  int   (*gs_timeout_give_up)(DVI_DEVICE,DVI);
  int   (*gs_timeout_value)(DVI_DEVICE,DVI);
  /* color */
  void  (*color_rgb)(DVI_DEVICE,DVI,int,double,double,double);
  void  (*color_cmyk)(DVI_DEVICE,DVI,int,double,double,double,double);
  void  (*color_gray)(DVI_DEVICE,DVI,int,double);
  void  (*color_named)(DVI_DEVICE,DVI,int,char*);
  void  (*bgcolor_rgb)(DVI_DEVICE,DVI,int,double,double,double);
  void  (*bgcolor_cmyk)(DVI_DEVICE,DVI,int,double,double,double,double);
  void  (*bgcolor_gray)(DVI_DEVICE,DVI,int,double);
  void  (*bgcolor_named)(DVI_DEVICE,DVI,int,char*);
  /* message */
  void  (*message_advice)(DVI_DEVICE,DVI,char*,...);
  void  (*message_warning)(DVI_DEVICE,DVI,char*,...);
  void  (*message_error)(DVI_DEVICE,DVI,char*,...);
  void  (*message_fatal)(DVI_DEVICE,DVI,char*,...);
  /* font list */
  void  (*font_list)(DVI_DEVICE,DVI,char*,long,double,long,double,long,double);
  void  (*font_list_missing)(DVI_DEVICE,DVI,char*,long,double,
			     long,double,long,double);
  /* working directory */
  char  *(*temp_dir)(DVI_DEVICE,DVI);
  /* device polling */
  int   (*device_polling)(DVI_DEVICE,DVI,int);
  /*** Device Private Data ***/
  void  *private1;
  void  *private2;
  void  *private3;
  void  *private4;
};
extern DVI_DEVICE                        DVI_device_alloc(void);
#define DVI_DEVICE_ALLOC()               DVI_device_alloc()
#define DEV_CALL(dev,method)             (dev)->method
#define DEV_METHOD_SET(dev,method,func)  ((dev)->method = (func))
#define DEV_METHOD_DEF(dev,method)       ((dev)->method != NULL)
#define DEV_METHOD_UNDEF(dev,method)     ((dev)->method == NULL)

#define DVI_POLL_FILE      1   /* polling during reading a dvi file */
#define DVI_POLL_FONT      2   /* polling during opening fonts */
#define DVI_POLL_PAGE      3   /* polling during drawing a page */
#define DVI_POLL_SPECIAL   4   /* polling during executing a special */

#define DVI_SHRINK_MIN   0.01



/*
 * GRAPHICS CONTEXT 
 */

#define DVI_GSTATE_STACK_SIZE        64
struct s_gstate_stack_elem {
  int     type;
  double  scale_h, scale_v;
  int     angle;
};
typedef struct s_gstate_stack_elem   DVI_GSTATE_STACK;



/*
 * CONTEXT 
 */

struct dvi_s_context {
  int                 current_page;
  long                offset;
  unsigned long       fnt;
  int                 fid;
  int                 depth;
  struct stack_frame *stack;
  DVI_COLOR           color_stack;
  char               *color_bg;
  DVI_GSTATE_STACK    gstate_stack[DVI_GSTATE_STACK_SIZE];
  int                 gstate_stack_ptr;
  int                 gstate_transformed;
  double              gstate_scale_h;
  double              gstate_scale_v;
  int                 gstate_angle;
};


/*
 * page number in BOP instruction
 */
struct dvi_s_page_no {
  long   c[10];
};


/* 
 * DVI FILE OBJECT 
 */
struct dvi_s_file {
  /** METHODS **/
  int       (*open_font)(DVI,DVI_DEVICE);
  int       (*dispose)(DVI,DVI_DEVICE);
  int       (*draw_page)(DVI,DVI_DEVICE,int,double);
  int       (*file_modified)(DVI,char*);
  int       (*continue_to_context)(DVI,DVI_DEVICE,DVI_CONTEXT,double);
  DVI_CONTEXT  (*get_current_context)(DVI,DVI_DEVICE);
  int          (*free_context)(DVI,DVI_DEVICE,DVI_CONTEXT);
  /** PUBLIC VARIABLES **/
  /** CURRENT PAGE NO **/ 
  int                   current_page;
  char                  job_name[257];
  struct dvi_s_page_no *page_no;             /* c[0],...,c[9] in BOP instr */
  char                  curr_page_name[16];  /* c[0] of current page */
  int                   pages;     /* total number of pages */
  enum dvi_error        error;     /* Error code by method invokation */
  int                   ptex_dvi;  /* DVI with vertical writing by pTeX */
  /** DVILIB INTERNAL USE **/
  void                 *privates;
};


/* 
 * FRAME BUFFER OBJECT 
 */
#define DVI_FB_TYPE_MONO       0
#define DVI_FB_TYPE_RGB        1
struct dvi_s_frame_buffer {
  int             type;
  unsigned char  *data;
  long            width;
  long            height;
  long            w_bytes;
  long            raster;
  unsigned char   rgb_fg_r, rgb_fg_g, rgb_fg_b;
  unsigned char   rgb_bg_r, rgb_bg_g, rgb_bg_b;
};



/*
 * DVIlib Functions 
 */

extern char  *DVI_version(void);
extern void   DVI_setup(void);
extern int    DVI_init(char*,char*);
extern DVI    DVI_create(DVI_DEVICE,char*,DVI_PROPERTY);

extern char  *DVI_find_dvi_path(char*);
extern char  *DVI_find_dvi_path2(char*,char**);
extern int    DVI_read_paper_size_db(void);

extern int    DVI_parse_paper_size(char*,double*,double*);
extern char **DVI_get_paper_size_list(void);
extern int    DVI_read_device_mode_db(void);
extern int    DVI_parse_device_mode(char*,double*,double*,char**);
extern char **DVI_get_device_mode_list(void);
extern int    DVI_rgb_lookup(char *name, 
			     unsigned char*, unsigned char*, unsigned char*);


#define DVI_VERSION()                   DVI_version()
#define DVI_SETUP()                     DVI_setup()
#define DVI_INIT(vflibcap,vflib_params) DVI_init(vflibcap,vflib_params)
#define DVI_CREATE(dev,f,prop)          DVI_create(dev,f,prop)
#define DVI_OPEN_FONT(dvi,dev)          (dvi->open_font)(dvi,dev)
#define DVI_DISPOSE(dvi,dev)            (dvi->dispose)(dvi,dev)
#define DVI_CHANGE_SHRINK(dvi,dev,n)    (dvi->change_shrink)(dvi,dev,n)
#define DVI_FILE_MODIFIED(dvi, f)       (dvi->file_modified)(dvi,f)

#define DVI_DRAW_PAGE(dvi,dev,p,shrink) \
                           (dvi->draw_page)(dvi,dev,p,shrink)
#define DVI_CURRENT_PAGE_NAME(dvi,dev)  (dvi->curr_page_name)
#define DVI_CONTINUE_TO_CONTEXT(dvi,dev,c,shrink) \
                           (dvi->continue_to_context)(dvi,dev,c,shrink)
#define DVI_GET_CURRENT_CONTEXT(dvi,dev)  \
                           (dvi->get_current_context)(dvi,dev)
#define DVI_FREE_CONTEXT(dvi,dev,c)     \
                           (dvi->free_context)(dvi,dev,c)

extern DVI_FRAME_BUFFER DVI_fb_create(long,long);
extern DVI_FRAME_BUFFER DVI_fb_create_mono(long,long);
extern DVI_FRAME_BUFFER DVI_fb_create_rgb(long,long);
extern void DVI_fb_dispose(DVI_FRAME_BUFFER);
extern void DVI_fb_clear(DVI_FRAME_BUFFER);
extern int DVI_fb_put_bitmap(DVI_FRAME_BUFFER,DVI_BITMAP,long,long);
extern int DVI_fb_put_pixmap_rgb(DVI_FRAME_BUFFER,DVI_PIXMAP_RGB,long,long);
extern int DVI_fb_put_rectangle(DVI_FRAME_BUFFER,long,long,long,long);
extern int DVI_fb_raster_zero(DVI_FRAME_BUFFER,long);
extern int DVI_fb_raster_nonzero(DVI_FRAME_BUFFER,long);
extern unsigned char* DVI_fb_raster_pointer(DVI_FRAME_BUFFER,long);
extern unsigned short* DVI_fb_antialias(DVI_FRAME_BUFFER,int,
					long,long,long,long,long*,long*);
extern int DVI_fb_boundingbox(DVI_FRAME_BUFFER fb,
			      long *xminp, long *xmaxp,
			      long *yminp, long *ymaxp);
extern void DVI_fb_rgb_foreground(DVI_FRAME_BUFFER fb, int r, int g, int b);
extern void DVI_fb_rgb_background(DVI_FRAME_BUFFER fb, int r, int g, int b);


#endif /*__DVI_H__*/

/*EOF*/
