/*
 * lpconf.c -  A part of xgdvi
 *
 *  Copyright (C) 1999-2001  Hirotsugu Kakugawa. All rights reserved. 
 *  See "COPYING" for distribution of this software. 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
  
#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#if HAVE_PWD_H
#  include <pwd.h>
#endif
#include <ctype.h>

#include "lprconf.h"


#define MAX_NPRINTERS  64
struct lprconfig {
  char  *name;
  char  *spec_doc;
  char  *spec_page;
  char  *spec_queue;
  char  *spec_cancel;
};
struct lprconfig  lprconf[MAX_NPRINTERS];

static char  *lprconf_strchr(char*p, char ch);
static char  *substr_dup(char *s1, char *s2);


int
lprconf_read(char *sysdir, char *file)
{
  int    line, kline, knlines, nerrs, debug;
  char   path[1024];
  struct passwd  *pw;
  FILE  *fp;
  char   buff[4*1024], lbuff[1024], kbuff[1024];
  char   *p1, *p2, *p3, *p4, *p5;
  int    i, j, k, esc;

  debug = 0;
  if (getenv("XGDVI_DEBUG_LPR") != NULL)
    debug = 1;

#if defined(HAVE_GETPWUID) && defined(HAVE_GETUID)
  /* check ~/ */ 
  if ((pw = getpwuid(getuid())) != NULL){
    if (strcmp(pw->pw_dir, "/") == 0)
      sprintf(path, "/.%s", file);
    else
      sprintf(path, "%s/.xgdvi/%s", pw->pw_dir, file);
    if (debug == 1)
      printf("Checking PrintSpec: %s\n", path);
    if ((fp = fopen(path, "r")) != NULL)
      goto Found;
  }
#endif

  /* check site file in sysdir */ 
  if (strcmp(sysdir , "/") == 0)
    sprintf(path, "/%s", file);
  else 
    sprintf(path, "%s/%s", sysdir, file);
  if (debug == 1)
    printf("Checking PrintSpec: %s\n", path);
  if ((fp = fopen(path, "r")) != NULL)
    goto Found;

  /* check default file in sysdir */ 
  if (strcmp(sysdir , "/") == 0)
    sprintf(path, "/%s.default", file);
  else
    sprintf(path, "%s/%s.default", sysdir, file);
  if (debug == 1)
    printf("Checking PrintSpec: %s\n", path);
  if ((fp = fopen(path, "r")) != NULL)
    goto Found;

  /* not found. install defaults. */
  if (debug == 1)
    printf("PrintSpec not found. Use built-in definition.\n");
  lprconf[0].name        = "PostScript Print: (dvips)"; 
  lprconf[0].spec_doc    = "cd %d; dvips -f %F | lpr ";
  lprconf[0].spec_page   = "cd %d; dviselect -i %F =%Rb:%Re  > %t; \
                            dvips -f %t | lpr; \
                            rm -f %t ";
  lprconf[0].spec_queue  = "lpq"; 
  lprconf[0].spec_cancel = "lprm"; 
  lprconf[1].name        = "Direct DVI Print: lpr -d"; 
  lprconf[1].spec_doc    = "lpr -d"; 
  lprconf[1].spec_page   = "dviselect -s =%Rb:%Re | lpr -d"; 
  lprconf[1].spec_queue  = "lpq"; 
  lprconf[1].spec_cancel = "lprm"; 
  lprconf[2].name        = NULL; 
  lprconf[2].spec_doc    = NULL; 
  lprconf[2].spec_page   = NULL; 
  lprconf[2].spec_queue  = NULL; 
  lprconf[2].spec_cancel = NULL; 
  return 0; 


Found:

  if (debug == 1)
    printf("PrintSpec: %s\n", path);

  line = 0;
  knlines = 1;
  nerrs = 0;
  i = 0;
  buff[0] = '\0';

  while (i < MAX_NPRINTERS-1){
    line++;
    if (fgets(lbuff, sizeof(lbuff), fp) == NULL)
      break;
    if ((p1 = strchr(lbuff, '\n')) != NULL)
      *p1 = '\0';
    if (strlen(buff) == 0)
      strcpy(kbuff, lbuff);
    esc = 0;
    for (j = 0; lbuff[j] != '\0' ; j++){
      if (esc == 1){
	esc = 0;
      } else {
	if (lbuff[j] == '#'){
	  lbuff[j] = '\0';
	  break;
	} else if ((lbuff[j] == '\\') && (lbuff[j+1] == '\0')){
	  esc = 1;
	}
      }
    }

    k = strlen(lbuff);
    if (lbuff[k-1] == '\\'){
      lbuff[k-1] = '\0';
      strcat(buff, lbuff);
      knlines++;
      continue;
    } else {
      kline = line;
      strcat(buff, lbuff);
    }

    j = 0;
    while (isspace((int)(buff[j])) || iscntrl((int)(buff[j]))){
      if (buff[j] == '\0')
	break;
      j++;
    }
    if (buff[j] == '\0')
      continue;

    p1 = p2 = p3 = NULL;
    if (    ((p1 = lprconf_strchr(buff, ':')) == NULL)
	 || ((p2 = lprconf_strchr(p1+1, ':')) == NULL)
	 || ((p3 = lprconf_strchr(p2+1, ':')) == NULL) 
	 || ((p4 = lprconf_strchr(p3+1, ':')) == NULL) 
	 || ((p5 = lprconf_strchr(p4+1, ':')) == NULL) ){
      nerrs++;
      if (nerrs == 1)
	fprintf(stderr, "XGDVI: Broken lpr config file: %s\n", path);
      fprintf(stderr, "  Line % 2d-: %s\n", kline, kbuff);
      continue;
    }
    lprconf[i].name        = substr_dup(buff, p1);
    lprconf[i].spec_doc    = substr_dup(p1+1, p2);
    lprconf[i].spec_page   = substr_dup(p2+1, p3);
    lprconf[i].spec_queue  = substr_dup(p3+1, p4);
    lprconf[i].spec_cancel = substr_dup(p4+1, p5);
#ifdef DEBUG 
    printf("** %s\n   %s\n   %s\n   %s\n   %s\n", 
	   lprconf[i].name, lprconf[i].spec_doc, lprconf[i].spec_page,
	   lprconf[i].spec_queue, lprconf[i].spec_cancel);
#endif
    i++;
    knlines = 1;
    buff[0] = '\0';
  }
  lprconf[i].name        = NULL; 
  lprconf[i].spec_doc    = NULL; 
  lprconf[i].spec_page   = NULL; 
  lprconf[i].spec_queue  = NULL; 
  lprconf[i].spec_cancel = NULL; 
  return 0;
}

static char*
substr_dup(char *s1, char *s2)
{
  char  *p;
  int   i;

  while (isspace((int)*s1))
    s1++;
  s2--;
  while (isspace((int)*s2) && (s2 != s1))
    s2--;
  for (i = 0; &s1[i-1] != s2; i++)
    ;
  p = (char*)malloc(i+1);
  for (i = 0; &s1[i-1] != s2; i++){
    if (iscntrl((int)s1[i]))
      p[i] = ' ';
    else
      p[i] = s1[i];
  }
  p[i] = '\0';

  return p;
}

static  char*
lprconf_strchr(char *p, char ch)
{
  if (p == NULL)
    return NULL;

  while (*p != '\0'){
    if (*p == '\\'){
      p++;
      if (*p == '\0')
	return NULL;
      p++;
      if (*p == '\0')
	return NULL;
    }
    if (*p == ch)
      return p;
    p++;
  }

  return NULL;
}


int
lprconf_nlprs(void)
{
  int  i;

  i = 0;
  while (lprconf[i].name != NULL){
    i++;
  }
  return i;

}

char*
lprconf_get_lpr_name(int i)
{
  return lprconf[i].name;
}

char*
lprconf_get_spec_doc(int i)
{
  return lprconf[i].spec_doc;
}

char*
lprconf_get_spec_page(int i)
{
  return lprconf[i].spec_page;
}

char*
lprconf_get_spec_queue(int i)
{
  return lprconf[i].spec_queue;
}

char*
lprconf_get_spec_cancel(int i)
{
  return lprconf[i].spec_cancel;
}



#ifdef DEBUG
int
main(int argc, char **argv)
{
  int  n, i;
  char *p1, *p2, *p3;

  lprconf_read(".", "xgdvi-lpr.default");
  n = lprconf_nlprs();
  for (i = 0; i < n; i++){
    p1 = lprconf_get_lpr_name(i);
    p2 = lprconf_get_spec_doc(i);
    p3 = lprconf_get_spec_page(i);
    printf("** %s\n   %s\n   %s\n", p1, p2, p3);
  }	   
}
#endif
