/***************************************************************************
 *   copyright           : (C) 2003 by Hendrik Sattler                     *
 *   mail                : pingos@hendrik-sattler.de                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ContestRankingDlg.H"
#include <algorithm>
#include <wx/textfile.h>
#include <wx/statline.h>

bool operator< (const ContestEntry& a, const ContestEntry& b) {
  return a.getRankingPoints() < b.getRankingPoints();
}

#define rateTotal 100.0
double ContestEntry::promilleMax = 0.0;
double ContestEntry::apmMax = 0.0;
double ContestEntry::getRankingPoints () const {
  /* getPromille(): less is better
   * promilleMax: highest value of all contest entries
   * That's why we map the contest entries value to
   * a value between 0 and 1 and invert the result.
   */
  double ipromille = 1.0;
  if (promilleMax != 0.0) {
    ipromille =  1 - (entry.getPromille() / promilleMax);
  }
  /* touches per minutes must only mapped to values between 0 and 1
   */
  double apm = 0.0;
  if (apmMax != 0.0) {
    apm = entry.getApM() / apmMax;
  }
  /* weight was specified by the user
   * higher values: give more weight to promille
   * lower values: give more weight to apm
   */
  return (ipromille * weight) + (apm * (rateTotal - weight));
}

ContestRankingDlg::ContestRankingDlg(wxWindow* parent,
				     const wxString& title,
				     const wxFileName& dataFile,
				     const double weight)
  : wxDialog(parent, -1, _("Contest Ranking: ")+title,
	     wxDefaultPosition, wxDefaultSize,
	     wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER)
{
  wxListCtrl* listingCtrl = getListingControl(dataFile,weight);

  wxBoxSizer* topSizer = new wxBoxSizer(wxVERTICAL);
  if (listingCtrl!=0) {
    topSizer->Add(listingCtrl,1,wxEXPAND|wxALL,10);
  } else{
    topSizer->Add(0,0,1);
    topSizer->Add(new wxStaticText(this,-1,_("Noone participated in this contest, yet.")),
		  0,wxALL|wxALIGN_CENTER,10);
    topSizer->Add(0,0,1);
  }
  topSizer->Add(new wxStaticLine(this,-1),0,wxEXPAND|wxLEFT|wxRIGHT,10);
  topSizer->Add(CreateButtonSizer(wxOK),0,wxALIGN_CENTER|wxALL,10);

  SetSizer(topSizer);
  topSizer->SetSizeHints(this);
  Centre();
}

#include <wx/file.h>
#include <wx/txtstrm.h>
#include "commonElements.H"

wxListCtrl* ContestRankingDlg::getListingControl (const wxFileName& dataFile,
						  const double weight)
{
  wxFile* file;
  std::vector<ContestEntry>* listing;
  wxListCtrl* listingCtrl;
  {
    wxLogNull logTarget; //ignore and don't display error message
    file = new wxFile(dataFile.GetFullPath());
  }
  if (!file->IsOpened()) {
    delete file;
    return 0;
  } else {
    wxFileInputStream filestream(*file);
    if (!filestream.Ok()) {
      delete file;
      return 0;
    } else {
      listing = getEntries(filestream,weight);
      delete file;
      if (listing != NULL) {
	 listingCtrl = createListingControl(listing);
	 delete listing;
	 return listingCtrl;
      } else {
	return 0;
      }
    }
  }
}

std::vector<ContestEntry>* ContestRankingDlg::getEntries (wxFileInputStream& filestream,
							  const double weight)
{
  std::vector<ContestEntry>* listing;
  wxArrayString lines;
  ContestEntry temp(weight);

  listing = new std::vector<ContestEntry>;
  lines = common::getIndexFile(filestream);
  wxString line;
  for (unsigned int i=0; i<lines.GetCount();++i) {
    line = lines[i];
    if (line.Len() > 0) {
      ContestEntry::promilleMax = 0.0;
      ContestEntry::apmMax = 0.0;
      temp.name = wxString(line.AfterFirst('"').BeforeFirst('"'));
      line = line.AfterFirst('"').AfterFirst('"').AfterFirst(',');
      long tempval;
      line.BeforeFirst(',').ToLong(&tempval);
      temp.entry.setRichtig((time_t)tempval);
      line.AfterFirst(',').BeforeLast(',').ToLong(&tempval);
      temp.entry.setFehler((time_t)tempval);
      temp.entry.setZeit(line.AfterLast(','));
    } else {
      temp.name = _("error in entry");
    }
    if (ContestEntry::promilleMax < temp.entry.getPromille()) {
      ContestEntry::promilleMax = temp.entry.getPromille();
    }
    if (ContestEntry::apmMax < temp.entry.getApM()) {
      ContestEntry::apmMax = temp.entry.getApM();
    }
    listing->push_back(temp);
      
    temp.name.Empty();
    temp.entry.setFehler(0);
    temp.entry.setRichtig(0);
    temp.entry.setZeit(0);
  }
  stable_sort(listing->begin(), listing->end());
  return listing;
}

wxListCtrl* ContestRankingDlg::createListingControl (std::vector<ContestEntry>* listing) {
  wxListCtrl* listingCtrl = new wxListCtrl(this,-1,
					   wxDefaultPosition,
					   wxDefaultSize,
					   wxLC_REPORT
					   |wxLC_VRULES
					   |wxSUNKEN_BORDER);
  wxListItem item;
  item.m_mask = wxLIST_MASK_TEXT;

  //inserting the column headers
  item.m_text = _("place");
  listingCtrl->InsertColumn(0,item);
  item.m_text = _("name");
  listingCtrl->InsertColumn(1,item);
  item.m_text = _("correct");
  listingCtrl->InsertColumn(2,item);
  item.m_text = _("mistakes");
  listingCtrl->InsertColumn(3,item);
  item.m_text = _("time");
  listingCtrl->InsertColumn(4,item);
  item.m_text = _("points");
  listingCtrl->InsertColumn(5,item);
    
  //filling the list
  std::vector<ContestEntry>::const_iterator runner;
  long index = 0;
  long place = listing->size()+1;
  wxString s;

  wxListItem placeCol;
  placeCol.SetMask(wxLIST_MASK_TEXT|wxLIST_MASK_FORMAT|wxLIST_MASK_WIDTH);
  placeCol.SetAlign(wxLIST_FORMAT_RIGHT);

  wxListItem nameCol;
  nameCol.SetMask(wxLIST_MASK_TEXT|wxLIST_MASK_FORMAT|wxLIST_MASK_WIDTH);
  nameCol.SetAlign(wxLIST_FORMAT_LEFT);
  nameCol.SetColumn(1);
  nameCol.SetId(0);

  wxListItem dataCol;
  dataCol.SetMask(wxLIST_MASK_TEXT|wxLIST_MASK_FORMAT|wxLIST_MASK_WIDTH);
  dataCol.SetAlign(wxLIST_FORMAT_RIGHT);
  dataCol.SetId(0);

  for (runner=listing->begin();runner!=listing->end();++runner) {
    //column 0
    place--;
    s.Empty();
    s << place;
    placeCol.m_text=s;
    if (place%2) {
      placeCol.SetBackgroundColour(*wxLIGHT_GREY);
      nameCol.SetBackgroundColour(*wxLIGHT_GREY);
      dataCol.SetBackgroundColour(*wxLIGHT_GREY);
    } else {
      placeCol.SetBackgroundColour(listingCtrl->GetBackgroundColour());
      nameCol.SetBackgroundColour(listingCtrl->GetBackgroundColour());
      dataCol.SetBackgroundColour(listingCtrl->GetBackgroundColour());
    }
    index=listingCtrl->InsertItem(placeCol);
    if (index != -1) {
      nameCol.SetId(index);
      dataCol.SetId(index);
      //column 1
      nameCol.SetText(runner->name);
      listingCtrl->SetItem(nameCol);
      //column 2
      s.Empty();
      s << runner->entry.getRichtig();
      dataCol.SetColumn(2);
      dataCol.SetText(s);
      listingCtrl->SetItem(dataCol);
      //column 3
      s.Empty();
      s << runner->entry.getFehler();
      dataCol.SetColumn(3);
      dataCol.SetText(s);
      listingCtrl->SetItem(dataCol);
      //column 4
      dataCol.SetColumn(4);
      dataCol.SetText(runner->entry.getZeitString());
      listingCtrl->SetItem(dataCol);
      //column 5
      dataCol.SetColumn(5);
      dataCol.SetText(wxString::Format(_T("%.3f"),runner->getRankingPoints()));
      listingCtrl->SetItem(dataCol);
    }
  }

  //setting the column widths
  listingCtrl->SetColumnWidth(0,wxLIST_AUTOSIZE_USEHEADER);
  listingCtrl->SetColumnWidth(1,wxLIST_AUTOSIZE);
  listingCtrl->SetColumnWidth(2,wxLIST_AUTOSIZE_USEHEADER);
  listingCtrl->SetColumnWidth(3,wxLIST_AUTOSIZE_USEHEADER);
  listingCtrl->SetColumnWidth(4,wxLIST_AUTOSIZE);
  listingCtrl->SetColumnWidth(5,wxLIST_AUTOSIZE);

  //setting the control's width
  wxSize listingSize = listingCtrl->GetSize();
  int viewRows = 10;
  if (listingCtrl->GetItemCount() <= viewRows) {
    listingSize.SetWidth(listingSize.GetWidth()
			 - wxSystemSettings::GetMetric(wxSYS_VSCROLL_X));
    viewRows = listingCtrl->GetItemCount();
  }
  for (long i=0;i<listingCtrl->GetColumnCount();++i) {
    listingSize.SetWidth(listingSize.GetWidth()
#ifdef __WXMSW__
			 +1
#endif
			 +listingCtrl->GetColumnWidth(i));
  }
    
  //setting the control's height
  listingCtrl->SetSize(listingSize);
  while (listingCtrl->GetCountPerPage() < viewRows){
    listingSize.SetHeight(listingSize.GetHeight()+2);
    listingCtrl->SetSize(listingSize);
  }
  listingSize.SetHeight(listingSize.GetHeight()
			+ wxSystemSettings::GetMetric(wxSYS_HSCROLL_Y));
  listingCtrl->SetSize(listingSize);
  
  return listingCtrl;
}
