/* Tracker
 * routines for emails with Evolution
 * Copyright (C) 2006, Laurent Aguerreche (laurent.aguerreche@free.fr)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <glib/gstdio.h>

#include "tracker-email-evolution.h"
#include "tracker-email-utils.h"
#include "tracker-db-email.h"

#ifdef HAVE_INOTIFY
#   include "tracker-inotify.h"
#else
#   ifdef HAVE_FAM
#      include "tracker-fam.h"
#   endif
#endif


#define EVOLUTION_MAIL_DIR_S ".evolution/mail"


typedef struct {
	char		*name;		/* laurent.aguerreche@free.fr */
	char		*uid;		/* uinique ID generated by Evolution, looks like "1134161347.7985.16@foo.bar" */
	gboolean	enabled;	/* enabled account or not */
	char		*source_url;	/* mbox:///var/mail/laurent (used to find where root directory for email files is) */
} EvolutionAccount;

typedef struct {
	char		*mail_dir;	/* something like "/home/laurent.evolution/mail" */
	GSList		*accounts;	/* list of EvolutionAccount elements */
	char		*dir_local;	/* something like "/home/laurent.evolution/mail/local" */
	GSList		*imap_dirs;	/* a directory looks like "/home/laurent.evolution/mail/imap/laurent.aguerreche@free.fr".
					   Email address acts as an account name. */
	GSList		*mh_dirs;	/* list of MH directories */
	GSList		*maildir_dirs;	/* list of maildir directories */
} EvolutionConfig;

typedef struct {
	GQueue		 *elements;	/* Stack of elements in XML files.
					   Eg. <source><url>foo</url></source> where "url" and "source" will appear in that stack. */
	EvolutionAccount *account;	/* Parsing of an XML entity feeds that account. */
} ParsingAccountState;

typedef enum {
	EVOLUTION_MAIL_PROTOCOL_MBOX,
	EVOLUTION_MAIL_PROTOCOL_IMAP,
	EVOLUTION_MAIL_PROTOCOL_MH,
	EVOLUTION_MAIL_PROTOCOL_MAILDIR
} EvolutionMailProtocol;

enum {
	EVOLUTION_MESSAGE_ANSWERED	= 1 << 0,
	EVOLUTION_MESSAGE_DELETED	= 1 << 1,
	EVOLUTION_MESSAGE_DRAFT		= 1 << 2,
	EVOLUTION_MESSAGE_FLAGGED	= 1 << 3,
	EVOLUTION_MESSAGE_SEEN		= 1 << 4,
	EVOLUTION_MESSAGE_ATTACHMENTS	= 1 << 5,
	EVOLUTION_MESSAGE_ANSWERED_ALL	= 1 << 6,
	EVOLUTION_MESSAGE_JUNK		= 1 << 7,
	EVOLUTION_MESSAGE_SECURE	= 1 << 8
};

typedef struct {
	char		*path;
	FILE		*f;
} SummaryFile;

typedef struct {
	gint32		version;
	gboolean	legacy;
	gint32		flags;
	gint32		nextuid;
	time_t		time;
	gint32		saved_count;
	gint32		unread_count;
	gint32		deleted_count;
	gint32		junk_count;
} SummaryFileHeader;

/* Some infos are only accessible throw a deep code path but we need to retreive them. */
typedef struct {
	char		*mail_uid;
} EvolutionAdHocInfos;


extern Tracker		*tracker;

static EvolutionConfig	*evolution_config = NULL;


static gboolean	load_evolution_config			(EvolutionConfig **conf);
static void	free_evolution_config			(EvolutionConfig *conf);

static EvolutionAccount * copy_evolution_account	(EvolutionAccount *account);
static void	free_evolution_account			(EvolutionAccount *account);
static GSList *	find_accounts_by_gconf			(GSList *found_accounts);
static void	account_start_element_handler		(GMarkupParseContext	*context,
							 const gchar		*element_name,
							 const gchar		**attr_names,
							 const gchar		**attr_values,
							 gpointer			user_data,
							 GError			**error);
static void	account_text_handler			(GMarkupParseContext	*context,
							 const gchar		*text,
							 gsize			text_len,
							 gpointer			user_data,
							 GError			**error);
static void	account_end_element_handler		(GMarkupParseContext	*context,
							 const gchar		*element_name,
							 gpointer			user_data,
							 GError			**error);

static gboolean	is_in_dir_local				(const char *dir);
static gboolean	is_in_dir_imap				(const char *dir);
static gboolean	is_in_dir_imap4				(const char *dir);

static void	watch_files_for_mbox			(DBConnection *db_con, const char *dir_path);
static void	watch_files_for_imap_account		(DBConnection *db_con, const char *dir_path);
static void	watch_files_for_mh_dir			(DBConnection *db_con, const char *dir_path);
static void	watch_files_for_maildir_dir		(DBConnection *db_con, const char *dir_path);
static void	watch_ev_summary_files			(DBConnection *db_con, const char *dir_path);

static char *	get_account_name_in_imap_path		(const char *path);
static char *	make_uri				(EvolutionMailProtocol mail_protocol,
							 const char *account_name, const char *path, const char *uid);

static void	load_uri_and_status_of_mbox_mail_message (GMimeMessage *g_m_message, MailMessage *msg);

static void	watch_summary_file			(DBConnection *db_con, const char *path);
static gboolean	open_summary_file			(const char *path, SummaryFile **summary);
static void	free_summary_file			(SummaryFile *summary);
static gboolean	load_summary_file_header		(SummaryFile *summary, SummaryFileHeader **header);
static void	free_summary_file_header		(SummaryFileHeader *header);
static gboolean	load_summary_file_meta_header_for_local	(SummaryFile *summary, SummaryFileHeader *header);
static gboolean	load_summary_file_meta_header_for_imap	(SummaryFile *summary, SummaryFileHeader *header);

//static gboolean	load_mail_message_for_local 		(SummaryFile *summary, MailMessage **mail_msg);
//static gboolean	skip_mail_message_for_local		(SummaryFile *summary);
static gboolean	load_mail_message_for_imap		(SummaryFile *summary, MailMessage **mail_msg);
static gboolean	skip_mail_message_for_imap		(SummaryFile *summary);
static gboolean	load_mail_message_for_imap4		(SummaryFile *summary, MailMessage **mail_msg);
static gboolean	skip_mail_message_for_imap4		(SummaryFile *summary);
static gboolean	do_load_mail_message_for_imap		(SummaryFile *summary, MailMessage **mail_msg, gboolean do_skipping_of_content_info);
static gboolean	do_skip_mail_message_for_imap		(SummaryFile *summary, gboolean do_skipping_of_content_info);
static gboolean	load_mail_message			(SummaryFile *summary, MailMessage *mail_msg, EvolutionAdHocInfos *adhoc_infos);
static gboolean	skip_mail_message			(SummaryFile *summary);
static void	free_evolution_adhoc_infos		(EvolutionAdHocInfos *adhoc_infos);
static gboolean	try_to_save_ondisk_email_message	(DBConnection *db_con, const char *path);
static GSList *	add_persons_from_internet_address_list_string_parsing	(GSList *list, const char *s);
static gboolean	skip_loading_content_info		(SummaryFile *summary);
static gboolean	do_skip_loading_content_info		(SummaryFile *summary);

static inline gboolean	decode_gint32		(FILE *f, gint32 *n);
static inline gboolean	skip_gint32_decoding	(FILE *f);
static inline gboolean	decode_guint32		(FILE *f, guint32 *n);
static inline gboolean	skip_guint32_decoding	(FILE *f);
static inline gboolean	decode_time_t		(FILE *f, time_t *t);
static inline gboolean	skip_time_t_decoding	(FILE *f);
static inline gboolean	decode_off_t		(FILE *f, off_t *t);
static inline gboolean	skip_off_t_decoding	(FILE *f);
static inline gboolean	decode_string		(FILE *f, char **str);
static inline gboolean	skip_string_decoding	(FILE *f);
static inline gboolean	skip_token_decoding	(FILE *f);




/********************************************************************************************
 Public functions
*********************************************************************************************/

gboolean
evolution_init_module (void)
{
	EvolutionConfig *conf;

	if (evolution_config) {
		return TRUE;
	}

	conf = NULL;

	if (load_evolution_config (&conf)) {
		evolution_config = conf;
	}

	return evolution_module_is_running ();
}


gboolean
evolution_module_is_running (void)
{
	return evolution_config != NULL;
}


gboolean
evolution_finalize_module (void)
{
	if (!evolution_config) {
		return TRUE;
	}

	free_evolution_config (evolution_config);
	evolution_config = NULL;

	return !evolution_module_is_running ();
}


void
evolution_watch_emails (DBConnection *db_con)
{
	EvolutionConfig *conf;

	g_return_if_fail (db_con);
	g_return_if_fail (evolution_config);

	conf = evolution_config;

	watch_files_for_mbox (db_con, conf->dir_local);

	#define WATCH_MY_LIST(list,watch_fct) {			\
		const GSList *tmp;				\
		for (tmp = list; tmp; tmp = tmp->next) {	\
			const char *dir_path;			\
			dir_path = tmp->data;			\
			watch_fct (db_con, dir_path);		\
		}						\
	}

	WATCH_MY_LIST (conf->imap_dirs, watch_files_for_imap_account);
	WATCH_MY_LIST (conf->mh_dirs, watch_files_for_mh_dir);
	WATCH_MY_LIST (conf->maildir_dirs, watch_files_for_maildir_dir);

	#undef WATCH_MY_LIST
}


gboolean
evolution_file_is_interesting (DBConnection *db_con, FileInfo *info)
{
	const GSList *dir;

	g_return_val_if_fail (db_con, FALSE);
	g_return_val_if_fail (info, FALSE);
	g_return_val_if_fail (evolution_config, FALSE);
	g_return_val_if_fail (evolution_config->mail_dir, FALSE);

	/* We want to take any file that could interest Evolution.
	   So we consider as interesting any file into directories of Evolution. */
	if (g_str_has_prefix (info->uri, evolution_config->mail_dir)) {
		return TRUE;
	}

	/* But emails into MH dirs need to be treat separately since they can
	   be anywhere! */
	for (dir = evolution_config->mh_dirs; dir; dir = dir->next) {
		const char *dir_path;

		dir_path = dir->data;
		if (g_str_has_prefix (info->uri, dir_path)) {
			return TRUE;
		}
	}

	/* and same thing for maildir dirs */
	for (dir = evolution_config->maildir_dirs; dir; dir = dir->next) {
		const char *dir_path;

		dir_path = dir->data;
		if (g_str_has_prefix (info->uri, dir_path)) {
			return TRUE;
		}
	}

	return FALSE;
}


void
evolution_index_file (DBConnection *db_con, FileInfo *info)
{
	char *file_name;

	g_return_if_fail (db_con);
	g_return_if_fail (info);

	file_name = g_path_get_basename (info->uri);

	if (is_in_dir_local (info->uri) && g_str_has_suffix (file_name, ".ev-summary")) {
		/* a MBox file changed */
		char		*mbox_file;
		SummaryFile	*summary;

		mbox_file = tracker_get_radix_by_suffix (info->uri, ".ev-summary");

		summary = NULL;

		if (open_summary_file (mbox_file, &summary)) {
			SummaryFileHeader	*header;
			guint			tmp_nb_emails_in_db;
			gint32			nb_emails_in_db;

			header = NULL;

			if (!load_summary_file_header (summary, &header)) {
				free_summary_file (summary);
				goto end_index;
			}

			if (!load_summary_file_meta_header_for_local (summary, header)) {
				free_summary_file_header (header);
				free_summary_file (summary);
				goto end_index;
			}

			tmp_nb_emails_in_db = tracker_db_email_get_nb_emails_in_mbox (db_con, mbox_file);
			if (tmp_nb_emails_in_db > G_MAXINT32) {
				tracker_log ("ERROR: number of emails in MBox file is too big (> G_MAXINT32)");
				free_summary_file_header (header);
				free_summary_file (summary);
				goto end_index;
			}

			nb_emails_in_db = (gint32) tmp_nb_emails_in_db, 0, G_MAXINT32;

			if (header->saved_count == nb_emails_in_db) {
				/* some emails changed their status */
				//gint32 i;

				/* It is time consuming to find the changed email, so it is prefered
				   to update status of each email.
				for (i = 0; i < header->saved_count; i++) {
					MailMessage *mail_msg;

					mail_msg = NULL;
					if (load_mail_message_for_local (summary, &mail_msg)) {
						tracker_db_email_update_email (db_con, mail_msg);
						email_free_mail_message (mail_msg);
					}
				}
				*/

			} else if (header->saved_count < nb_emails_in_db) {
				/* some emails removed. It is easiest to do a new full MBox indexing */
				tracker_db_email_delete_emails_of_mbox (db_con, mbox_file);
				email_parse_mail_file_and_save_new_emails (db_con, MAIL_APP_EVOLUTION, mbox_file, load_uri_and_status_of_mbox_mail_message);
			} else {
				/* new emails received */
				email_parse_mail_file_and_save_new_emails (db_con, MAIL_APP_EVOLUTION, mbox_file, load_uri_and_status_of_mbox_mail_message);
			}

			free_summary_file_header (header);
			free_summary_file (summary);
		}

		g_free (mbox_file);

	} else if (is_in_dir_imap (info->uri) || is_in_dir_imap4 (info->uri)) {

		if (strcmp (file_name, "summary") == 0) {
			/* Evolution has received a new email, modified status of an old one, or deleted some of them in IMAP */
			SummaryFile *summary;

			summary = NULL;

			if (open_summary_file (info->uri, &summary)) {
				SummaryFileHeader	*header;
				gboolean		in_imap, in_imap4;
				gint32			nb_emails_in_db;
				guint			tmp_nb_emails_in_db;
				char			*imap_dir;

				header = NULL;

				if (!load_summary_file_header (summary, &header)) {
					free_summary_file (summary);
					goto end_index;
				}

				if (!load_summary_file_meta_header_for_imap (summary, header)) {
					free_summary_file_header (header);
					free_summary_file (summary);
					goto end_index;
				}

				if (is_in_dir_imap4 (info->uri)) {
					in_imap = FALSE;
					in_imap4 = TRUE;
				} else {
					in_imap = TRUE;
					in_imap4 = FALSE;
				}

				tmp_nb_emails_in_db = tracker_db_email_get_nb_emails_in_dir (db_con, imap_dir);
				if (tmp_nb_emails_in_db > G_MAXINT32) {
					tracker_log ("ERROR: number of emails in IMAP directory is too big (> G_MAXINT32)");
					free_summary_file_header (header);
					free_summary_file (summary);
					goto end_index;
				}

				nb_emails_in_db = (gint32) tmp_nb_emails_in_db, 0, G_MAXINT32;

				imap_dir = g_path_get_dirname (info->uri);

				if (header->saved_count == nb_emails_in_db) {
					/* some emails changed their status */
					//gint32 i;

					/* It is time consuming to find the changed email, so it is prefered
					   to update status of each email. 
					for (i = 0; i < header->saved_count; i++) {
						MailMessage *mail_msg;

						mail_msg = NULL;
						if ((in_imap && load_mail_message_for_imap (summary, &mail_msg)) ||
						    (in_imap4 && load_mail_message_for_imap4 (summary, &mail_msg))) {

							tracker_db_email_update_email (db_con, mail_msg);
							email_free_mail_message (mail_msg);
						}
					}

					*/
				/* we should only do this if we are sure count is accurate */	
				} else if (header->saved_count < nb_emails_in_db) {
					/* some emails removed. It is easiest to do a new full summary indexing */
					gint32 i;

					tracker_db_email_delete_emails_of_dir (db_con, imap_dir);

					for (i = 0; i < header->saved_count; i++) {
						MailMessage *mail_msg;

						mail_msg = NULL;
						if ((in_imap && load_mail_message_for_imap (summary, &mail_msg)) ||
						    (in_imap4 && load_mail_message_for_imap4 (summary, &mail_msg))) {

							if (!try_to_save_ondisk_email_message (db_con, mail_msg->path)) {
								/* we do not have all infos but we still save them */
								tracker_db_email_save_email (db_con, mail_msg);
							}
							email_free_mail_message (mail_msg);
						}
					}

				} else {
					/* new emails received */
					gint32 i;

					for (i = 0; i < nb_emails_in_db; i++) {
						if ((in_imap && !skip_mail_message_for_imap (summary)) ||
						    (in_imap4 && !skip_mail_message_for_imap4 (summary))) {
							return;
						}
					}

					/* now we will read the new emails */
					for (; i < header->saved_count; i++) {
						MailMessage *mail_msg;

						mail_msg = NULL;
						if ((in_imap && !load_mail_message_for_imap (summary, &mail_msg)) ||
						    (in_imap4 && !load_mail_message_for_imap4 (summary, &mail_msg))) {
							return;
						}

						if (!try_to_save_ondisk_email_message (db_con, mail_msg->path)) {
							/* we do not have all infos but we still save them */
							tracker_db_email_save_email (db_con, mail_msg);
						}
						email_free_mail_message (mail_msg);
					}
				}

				free_summary_file (summary);
				free_summary_file_header (header);

				g_free (imap_dir);
			}

		} else if (info->mime && strcmp (info->mime, "message/rfc822") == 0 && g_str_has_suffix (file_name, ".")) {
			/* Evolution has downloaded body of an email */
			if (tracker_file_is_indexable (info->uri)) {
				MailMessage *ondisk_msg;

				ondisk_msg = email_parse_mail_message_by_path (MAIL_APP_EVOLUTION, info->uri, NULL);
				if (ondisk_msg) {
					//tracker_db_email_update_email (db_con, ondisk_msg);
					email_free_mail_message (ondisk_msg);
				}
			}
		}

	} else if (info->mime && strcmp (info->mime, "message/rfc822") == 0) {
		/* for MH and maildir we can only check mime type of each file to know what we have */

		if (email_mh_is_in_a_mh_dir (info->uri)) {
			/* a new mail message in an MH directory */
			try_to_save_ondisk_email_message (db_con, info->uri);

		} else if (email_maildir_is_in_a_maildir_dir (info->uri)) {
			/* a new mail message in an maildir directory */
			try_to_save_ondisk_email_message (db_con, info->uri);
		}
	}

 end_index:
	g_free (file_name);
}




/********************************************************************************************
 Private functions
*********************************************************************************************/

static gboolean
load_evolution_config (EvolutionConfig **conf)
{
	char		*dir_imap, *dir_imap4;
	EvolutionConfig	*m_conf;
	GSList		*accounts;
	const GSList	*account;

	if (*conf) {
		free_evolution_config (*conf);
	}

	*conf = g_new0 (EvolutionConfig, 1);
	m_conf = *conf;

	m_conf->mail_dir = g_build_filename (g_get_home_dir (), EVOLUTION_MAIL_DIR_S, NULL);
	m_conf->dir_local = g_build_filename (m_conf->mail_dir, "local", NULL);

	dir_imap = g_build_filename (m_conf->mail_dir, "imap", NULL);
	dir_imap4 = g_build_filename (m_conf->mail_dir, "imap4", NULL);

	accounts = find_accounts_by_gconf (NULL);

	for (account = accounts; account; account = account->next) {
		EvolutionAccount *evo_acc;

		evo_acc = account->data;

		if (evo_acc->enabled) {

			m_conf->accounts = g_slist_prepend (m_conf->accounts, copy_evolution_account (evo_acc));

			if (evo_acc->source_url) {

				if (g_str_has_prefix (evo_acc->source_url, "mbox://")) {
					/* currently nothing */

				} else if (g_str_has_prefix (evo_acc->source_url, "imap://")) {
					char *account_name;

					/* Assume url schema is:
					   imap://foo@imap.free.fr/;etc

					   We try to get "foo@imap.free.fr".
					*/

					account_name = g_strndup (evo_acc->source_url + 7,
								  (strchr (evo_acc->source_url + 7, ';') - 1) - (evo_acc->source_url + 7));

					m_conf->imap_dirs = g_slist_prepend (m_conf->imap_dirs, g_build_filename (dir_imap, account_name, NULL));
					g_free (account_name);

				} else if (g_str_has_prefix (evo_acc->source_url, "mh://")) {

					m_conf->mh_dirs = g_slist_prepend (m_conf->mh_dirs, g_strdup (evo_acc->source_url + 5));

				} else if (g_str_has_prefix (evo_acc->source_url, "maildir://")) {

					m_conf->maildir_dirs = g_slist_prepend (m_conf->maildir_dirs, g_strdup (evo_acc->source_url + 10));
				}
			}
		}
	}

	g_slist_foreach (accounts, (GFunc) free_evolution_account, NULL);
	g_slist_free (accounts);

	g_free (dir_imap);
	g_free (dir_imap4);

	return TRUE;
}


static void
free_evolution_config (EvolutionConfig *conf)
{
	if (!conf) {
		return;
	}

	if (conf->mail_dir) {
		g_free (conf->mail_dir);
	}

	if (conf->dir_local) {
		g_free (conf->dir_local);
	}

	#define FREE_MY_LIST(list, free_fct)				\
		g_slist_foreach (list, (GFunc) free_fct, NULL);		\
		g_slist_free (list);

	FREE_MY_LIST (conf->accounts, free_evolution_account);
	FREE_MY_LIST (conf->imap_dirs, g_free);
	FREE_MY_LIST (conf->mh_dirs, g_free);
	FREE_MY_LIST (conf->maildir_dirs, g_free);

	#undef FREE_MY_LIST

	g_free (conf);
}


static EvolutionAccount *
copy_evolution_account (EvolutionAccount *account)
{
	EvolutionAccount *ret;

	g_return_val_if_fail (account, NULL);

	ret = g_new0 (EvolutionAccount, 1);

	#define DEEP_COPY_STR(val)				\
		if (account->val) {				\
			ret->val = g_strdup (account->val);	\
		}

	DEEP_COPY_STR (name);
	DEEP_COPY_STR (uid);
	DEEP_COPY_STR (source_url);

	#undef DEEP_COPY_STR

	return ret;
}


static void
free_evolution_account (EvolutionAccount *account)
{
	if (!account) {
		return;
	}

	if (account->name) {
		g_free (account->name);
	}

	if (account->uid) {
		g_free (account->uid);
	}

	if (account->source_url) {
		g_free (account->source_url);
	}

	g_free (account);
}


static GSList *
find_accounts_by_gconf (GSList *found_accounts)
{
	char *argv[4];
	char *text;

	argv[0] = "gconftool";
	argv[1] = "--get";
	argv[2] = "/apps/evolution/mail/accounts";
	argv[3] = NULL;

	if (g_spawn_sync (NULL,
			  argv,
			  NULL,
			  G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL,
			  NULL,
			  NULL,
			  &text,
			  NULL,
			  NULL,
			  NULL)) {

		if (text) {
			char			*to_parse;
			GMarkupParser		*parser;
			GMarkupParseContext	*parser_context;
			char			**accounts;
			ParsingAccountState	*state;

			if (text[0] == '[') {
				size_t len;

				/* Usually a string returned by gconftool begins by character '['.
				   We remove this character. */

				to_parse = text + 1;

				/* and then it probably ends with string "]\n"... */
				len = strlen (text);
				if (text[len - 1] == '\n' && text[len - 2] == ']') {
					text[len - 2] = '\0';
				}
			} else {
				to_parse = text;
			}

			parser = g_new0 (GMarkupParser, 1);
			parser->start_element = account_start_element_handler;
			parser->text = account_text_handler;
			parser->end_element = account_end_element_handler;

			state = g_new0 (ParsingAccountState, 1);
			state->elements = g_queue_new ();

			parser_context = g_markup_parse_context_new (parser, 0, state, NULL);

			/* Accounts look that way:

			   <?xml version="1.0"?>
			   <account name="laurent.aguerreche@free.fr" uid="1134161347.7985.16@foo.bar" enabled="true"><identity><...etc...></account>
			   ,<?xml version="1.0"?>
			   <account name="whrahahaaa@free.fr" uid="1147098156.5252.0@foo.bar" enabled="true"><identity><...etc...></account>

			   so we split accounts around string "\n,".
			 */

			accounts = g_strsplit (to_parse, "\n,", -1);

			if (accounts) {
				char **account;

				for (account = accounts; *account; account++) {
					state->account = g_new0 (EvolutionAccount, 1);

					g_markup_parse_context_parse (parser_context, *account, -1, NULL);

					found_accounts = g_slist_prepend (found_accounts, state->account);
				}

				g_strfreev (accounts);
			}

			g_markup_parse_context_free (parser_context);

			g_queue_free (state->elements);
			g_free (state);

			g_free (parser);

			g_free (text);
		}
	}

	return found_accounts;
}


#define ELEMENT_ACCOUNT		"account"
#define ELEMENT_SOURCE		"source"

#define ELEMENT_URL		"url"

#define ELEMENT_IS(name) (element_name != NULL && strcmp (element_name, (name)) == 0)
#define PREV_ELEMENT_IS(name) (prev_element_name != NULL && strcmp (prev_element_name, (name)) == 0)

#define ATTRIBUTE_NAME		"name"
#define ATTRIBUTE_UID		"uid"
#define ATTRIBUTE_ENABLED	"enabled"

#define ATTRIBUTE_IS(name) (strcmp (attr_names[i], (name)) == 0)


static void
account_start_element_handler (GMarkupParseContext	*context,
			       const gchar		*element_name,
			       const gchar		**attr_names,
			       const gchar		**attr_values,
			       gpointer			user_data,
			       GError			**error)
{
	ParsingAccountState	*state;
	EvolutionAccount	*account;

	state = user_data;

	g_return_if_fail (state);

	account = state->account;

	g_queue_push_head (state->elements, g_strdup (element_name));

	if (ELEMENT_IS (ELEMENT_ACCOUNT)) {
		guint i;
		for (i = 0; attr_names[i]; i++) {
			if (ATTRIBUTE_IS (ATTRIBUTE_NAME)) {
				account->name = g_strdup (attr_values[i]);

			} else if (ATTRIBUTE_IS (ATTRIBUTE_UID)) {
				account->uid = g_strdup (attr_values[i]);

			} else if (ATTRIBUTE_IS (ATTRIBUTE_ENABLED)) {
				account->enabled = (strcmp (attr_values[i], "true") == 0);
			}
		}
	}
}


static void
account_text_handler (GMarkupParseContext	*context,
		      const gchar		*text,
		      gsize			text_len,
		      gpointer			user_data,
		      GError			**error)
{
	ParsingAccountState	*state;
	const char		*element_name, *prev_element_name;

	state = user_data;

	g_return_if_fail (state);

	element_name = g_queue_peek_head (state->elements);
	prev_element_name = g_queue_peek_nth (state->elements, 1);

	if (ELEMENT_IS (ELEMENT_URL) && PREV_ELEMENT_IS (ELEMENT_SOURCE)) {
		state->account->source_url = g_strndup (text, text_len);
	}
}


static void
account_end_element_handler (GMarkupParseContext	*context,
			     const gchar		*element_name,
			     gpointer			user_data,
			     GError			**error)
{
	ParsingAccountState	*state;
	char			*my_element_name;

	state = user_data;

	g_return_if_fail (state);

	my_element_name = g_queue_pop_head (state->elements);

	g_free (my_element_name);
}


static gboolean
is_in_dir_local (const char *path)
{
	g_return_val_if_fail (path, FALSE);

	return strstr (path, G_DIR_SEPARATOR_S "local") != NULL;
}


static gboolean
is_in_dir_imap (const char *path)
{
	g_return_val_if_fail (path, FALSE);

	return strstr (path, G_DIR_SEPARATOR_S "imap") != NULL;
}


static gboolean
is_in_dir_imap4 (const char *path)
{
	g_return_val_if_fail (path, FALSE);

	return strstr (path, G_DIR_SEPARATOR_S "imap4" G_DIR_SEPARATOR_S) != NULL;
}


static void
watch_files_for_mbox (DBConnection *db_con, const char *dir_path)
{
	g_return_if_fail (db_con);
	g_return_if_fail (dir_path);

	watch_ev_summary_files (db_con, dir_path);
}


static void
watch_files_for_imap_account (DBConnection *db_con, const char *dir_path)
{
	char *tmp_name, *inbox_dir, *path_of_summary_file;

	g_return_if_fail (db_con);
	g_return_if_fail (dir_path);

	if (tracker_file_is_no_watched (dir_path)) {
		return;
	}

	/* We are currently in directory of an account.
	   Into this directory we find directory "folders" which contains directory "INBOX".
	   Directory "INBOX" has a summary file and mail messages. */

	tmp_name = g_build_filename (dir_path, "folders", NULL);

	if (!tracker_is_directory (tmp_name)) {
		g_free (tmp_name);
		return;
	}

	inbox_dir = g_build_filename (tmp_name, "INBOX", NULL);
	g_free (tmp_name);

/* 	if (!tracker_is_directory (tmp_name)) { */
/* 		g_free (inbox_dir); */
/* 		return; */
/* 	} */

	if (!tracker_is_directory_watched (inbox_dir, db_con)) {
		tracker_add_watch_dir (inbox_dir, db_con);
	}

	path_of_summary_file = g_build_filename (inbox_dir, "summary", NULL);
	g_free (inbox_dir);

	watch_summary_file (db_con, path_of_summary_file);
	g_free (path_of_summary_file);
}


static void
watch_files_for_mh_dir (DBConnection *db_con, const char *dir_path)
{
	g_return_if_fail (db_con);
	g_return_if_fail (dir_path);

	email_mh_watch_mail_messages (db_con, dir_path);
}


static void
watch_files_for_maildir_dir (DBConnection *db_con, const char *dir_path)
{
	g_return_if_fail (db_con);
	g_return_if_fail (dir_path);

	email_maildir_watch_mail_messages (db_con, dir_path);
}


static void
watch_ev_summary_files (DBConnection *db_con, const char *dir_path)
{
	GSList		*files;
	const GSList	*tmp_list;

	g_return_if_fail (db_con);
	g_return_if_fail (dir_path);

	if (tracker_file_is_no_watched (dir_path)) {
		return;
	}

	files = tracker_get_files (dir_path, FALSE);

	if (!tracker_is_directory_watched (dir_path, db_con)) {
		tracker_add_watch_dir (dir_path, db_con);
	}

	for (tmp_list = files; tmp_list; tmp_list = tmp_list->next) {
		const char *path;

		path = tmp_list->data;

		if (g_str_has_suffix (path, ".ev-summary")) {
			if (tracker_file_is_indexable (path)) {
				tracker_db_insert_pending_file (db_con, 0, path, NULL, 0, TRACKER_ACTION_CHECK, FALSE, FALSE, -1);
			}
		}
	}

	g_slist_foreach (files, (GFunc) g_free, NULL);
	g_slist_free (files);
}


static char *
get_account_name_in_imap_path (const char *path)
{
	const char *pos_imap;

	g_return_val_if_fail (path, NULL);

	pos_imap = strstr (path, G_DIR_SEPARATOR_S "imap");

	if (pos_imap) {
		const char *s, *s_end;

		s = pos_imap + 5;	/* strlen (G_DIR_SEPARATOR_S "imap") == 5 */

		if (s[0] == '4') {
			/* we are reading the end of word "imap4" */
			s++;

			if (s[0] == '\0') {
				return NULL;
			}
		}

		/* we are currently on a G_DIR_SEPARATOR character. Let's go further. */
		s++;

		if (s[0] == G_DIR_SEPARATOR || s[0] == '\0') {
			return NULL;
		}

		s_end = strchr (s, G_DIR_SEPARATOR);

		return (s_end) ? g_strndup (s, s_end - s) : g_strdup (s);
	}

	return NULL;
}


static char *
make_uri (EvolutionMailProtocol mail_protocol,
	  const char *account_name, const char *path, const char *uid)
{
	g_return_val_if_fail (path, NULL);

	switch (mail_protocol) {

		case EVOLUTION_MAIL_PROTOCOL_MBOX:
			return g_strdup_printf ("email://local@local/%s;uid=%s", path, uid);

		case EVOLUTION_MAIL_PROTOCOL_IMAP:
			return g_strdup_printf ("email://%s/%s;uid=%s", account_name, path, uid);

		case EVOLUTION_MAIL_PROTOCOL_MH:
			return g_strdup ("MH");

		case EVOLUTION_MAIL_PROTOCOL_MAILDIR:
			return g_strdup ("Maildir");

		default:
			g_return_val_if_reached (NULL);
	}
}


static void
load_uri_and_status_of_mbox_mail_message (GMimeMessage *g_m_message, MailMessage *msg)
{
	const char		*field;
	char			**parts;
	unsigned long int	uid, flags;
	char			*mbox_path;

	g_return_if_fail (g_m_message);
	g_return_if_fail (msg);

	field = g_mime_message_get_header (g_m_message, "X-Evolution");

	g_return_if_fail (field);

	/* we want to split lines with that form: 00001fd3-0100 into 00001fd3 and 0100 */
	parts = g_strsplit (field, "-", -1);

	if (!parts || !parts[0] || !parts[1]) {
		g_strfreev (parts);
		return;
	}


	/* parts[0] contains UID to contruct mail URI */

	uid = strtoul (parts[0], NULL, 16);

	mbox_path = g_path_get_basename (msg->parent_mail_file->path);

	if (msg->uri) {
		g_free (msg->uri);
	}

	if (uid <= G_MAXUINT32) {
		char *str_uid;

		str_uid = g_strdup_printf ("%" G_GUINT32_FORMAT, (guint32) uid);
		msg->uri = make_uri (EVOLUTION_MAIL_PROTOCOL_MBOX, NULL, mbox_path, str_uid);
		g_free (str_uid);

	} else {
		tracker_log ("******ERROR**** UID of evolution email is too big. X-Evolution field is probably malformed.");
		msg->uri = NULL;
	}

	g_free (mbox_path);


	/* parts[1] contains flags for status */

	flags = strtoul (parts[1], NULL, 16);

	if ((flags & EVOLUTION_MESSAGE_DELETED) == EVOLUTION_MESSAGE_DELETED) {
		msg->deleted = TRUE;
	}

	if ((flags & EVOLUTION_MESSAGE_JUNK) == EVOLUTION_MESSAGE_JUNK) {
		msg->junk = TRUE;
	}

	g_strfreev (parts);

}


static void
watch_summary_file (DBConnection *db_con, const char *path)
{
	g_return_if_fail (db_con);
	g_return_if_fail (path);

	tracker_db_insert_pending_file (db_con, 0, path, NULL, 0, TRACKER_ACTION_CHECK, FALSE, FALSE, -1);
}


static gboolean
open_summary_file (const char *path, SummaryFile **summary)
{
	g_return_val_if_fail (path, FALSE);

	if (*summary) {
		free_summary_file (*summary);
	}

	*summary = g_new0 (SummaryFile, 1);
	(*summary)->f = g_fopen (path, "rb");
	(*summary)->path = g_strdup (path);

	if (!(*summary)->f) {
		free_summary_file (*summary);
		*summary = NULL;

		return FALSE;
	} else {
		return TRUE;
	}
}


static void
free_summary_file (SummaryFile *summary)
{
	if (!summary) {
		return;
	}

	fclose (summary->f);
	g_free (summary->path);

	g_free (summary);
}


static gboolean
load_summary_file_header (SummaryFile *summary, SummaryFileHeader **header)
{
	SummaryFileHeader	*h;
	FILE			*f;

	g_return_val_if_fail (summary, FALSE);

	if (*header) {
		free_summary_file_header (*header);
	}

	*header = g_new0 (SummaryFileHeader, 1);
	h = *header;

	f = summary->f;

	if (!decode_gint32 (f, &h->version)) {
		goto error;
	}

	if (h->version > 0xff && (h->version & 0xff) < 12) {
		tracker_log ("******ERROR**** Summary file header version too low");
		goto error;
	}

	h->legacy = !(h->version < 0x100 && h->version >= 13);

	if (!decode_gint32 (f, &h->flags) ||
	    !decode_gint32 (f, &h->nextuid) ||
	    !decode_time_t (f, &h->time) ||
	    !decode_gint32 (f, &h->saved_count)) {
		goto error;
	}

	if (!h->legacy) {
		if (!decode_gint32 (f, &h->unread_count) ||
		    !decode_gint32 (f, &h->deleted_count) ||
		    !decode_gint32 (f, &h->junk_count)) {
			goto error;
		}
	}

	return TRUE;

 error:
	free_summary_file_header (*header);
	*header = NULL;

	return FALSE;
}


static void
free_summary_file_header (SummaryFileHeader *header)
{
	if (!header) {
		return;
	}

	g_free (header);
}


static gboolean
load_summary_file_meta_header_for_local (SummaryFile *summary, SummaryFileHeader *header)
{
	FILE	*f;
	gint32	dummy;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	if (!decode_gint32 (f, &dummy) ||	/* major */
	    !decode_gint32 (f, &dummy) ||	/* minor */
	    !decode_gint32 (f, &dummy)) {	/* uid_len */
		return FALSE;
	}

	return TRUE;
}


static gboolean
load_summary_file_meta_header_for_imap (SummaryFile *summary, SummaryFileHeader *header)
{
	FILE	*f;
	guint32	dummy0;

	g_return_val_if_fail (summary, FALSE);
	g_return_val_if_fail (header, FALSE);

	f = summary->f;

	/* check for legacy version */
	if (header->version != 0x30c) {
		gint32 version, dummy1;

		if (!decode_gint32 (f, &version)) {
			return FALSE;
		}

		if (version < 0) {
			tracker_log ("******ERROR**** summary file version too low");
			return FALSE;
		}

		/* Right now we only support summary versions 1 through 3 */
		if (version > 3) {
			tracker_log ("******ERROR**** Reported summary version (%" G_GINT32_FORMAT ") is too new", version);
			return FALSE;
		}

		if (version == 2) {
			if (!skip_gint32_decoding (f)) {
				return FALSE;
			}
		}

		/* validity */
		if (!decode_gint32 (f, &dummy1)) {
			return FALSE;
		}
	} else {
		/* validity */
		if (!decode_guint32 (f, &dummy0)) {
			return FALSE;
		}
	}

	return TRUE;
}


static gboolean
try_to_save_ondisk_email_message (DBConnection *db_con, const char *path)
{
	g_return_val_if_fail (db_con, FALSE);
	g_return_val_if_fail (path, FALSE);

	if (tracker_file_is_indexable (path)) {
		/* we have downloaded the mail message on disk so we can fully index it. */
		MailMessage *msg_on_disk;

		msg_on_disk = email_parse_mail_message_by_path (MAIL_APP_EVOLUTION, path, NULL);

		if (msg_on_disk) {
			tracker_db_email_save_email (db_con, msg_on_disk);
			email_free_mail_message (msg_on_disk);

			return TRUE;
		}
	}

	return FALSE;
}

/*
static gboolean
load_mail_message_for_local (SummaryFile *summary, MailMessage **mail_msg)
{
	off_t offset;

	g_return_val_if_fail (summary, FALSE);

	if (*mail_msg) {
		email_free_mail_message (*mail_msg);
	}

	*mail_msg = email_allocate_mail_message ();

	if (!load_mail_message (summary, *mail_msg, NULL)) {
		goto error;
	}

	if (!decode_off_t (summary->f, &offset)) {
		goto error;
	}

	return TRUE;

 error:
	email_free_mail_message (*mail_msg);
	*mail_msg = NULL;
	return FALSE;
}


static gboolean
skip_mail_message_for_local (SummaryFile *summary)
{
	g_return_val_if_fail (summary, FALSE);

	return (skip_mail_message (summary) &&
		skip_off_t_decoding (summary->f));
}
*/

static gboolean
load_mail_message_for_imap (SummaryFile *summary, MailMessage **mail_msg)
{
	return do_load_mail_message_for_imap (summary, mail_msg, TRUE);
}


static gboolean
skip_mail_message_for_imap (SummaryFile *summary)
{
	return do_skip_mail_message_for_imap (summary, TRUE);
}


static gboolean
load_mail_message_for_imap4 (SummaryFile *summary, MailMessage **mail_msg)
{
	return do_load_mail_message_for_imap (summary, mail_msg, FALSE);
}


static gboolean
skip_mail_message_for_imap4 (SummaryFile *summary)
{
	return do_skip_mail_message_for_imap (summary, TRUE);
}


static gboolean
do_load_mail_message_for_imap (SummaryFile *summary, MailMessage **mail_msg, gboolean do_skipping_of_content_info)
{
	EvolutionAdHocInfos	*adhoc_infos;
	guint32			server_flags;

	g_return_val_if_fail (summary, FALSE);

	if (*mail_msg) {
		email_free_mail_message (*mail_msg);
	}

	*mail_msg = email_allocate_mail_message ();

	adhoc_infos = g_new0 (EvolutionAdHocInfos, 1);

	if (!load_mail_message (summary, *mail_msg, adhoc_infos)) {
		goto error;
	}

	if (!decode_guint32 (summary->f, &server_flags)) {
		goto error;
	}

	if (do_skipping_of_content_info) {
		if (!skip_loading_content_info (summary)) {
			goto error;
		}
	}

	if (adhoc_infos->mail_uid) {
		char *dir_sum_file, *account_name;

		/* First we make path to mail message which is a path to mail message UID which ends by a dot. */
		dir_sum_file = g_path_get_dirname (summary->path);
		(*mail_msg)->path = g_strconcat (dir_sum_file, G_DIR_SEPARATOR_S, adhoc_infos->mail_uid, ".", NULL);
		g_free (dir_sum_file);

		/* Then we make mail message URI. */
		account_name = get_account_name_in_imap_path (summary->path);

		if (account_name) {
			const GSList *account;

			for (account = evolution_config->accounts; account; account = account->next) {
				const EvolutionAccount *evo_acc;

				evo_acc = account->data;

				if (strstr (evo_acc->source_url, account_name)) {
					(*mail_msg)->uri = make_uri (EVOLUTION_MAIL_PROTOCOL_IMAP, evo_acc->uid, "INBOX", adhoc_infos->mail_uid);
					break;
				}
			}

			g_free (account_name);
		}
	}

	free_evolution_adhoc_infos (adhoc_infos);

	return TRUE;

 error:
	free_evolution_adhoc_infos (adhoc_infos);
	email_free_mail_message (*mail_msg);
	*mail_msg = NULL;
	return FALSE;
}


static gboolean
do_skip_mail_message_for_imap (SummaryFile *summary, gboolean do_skipping_of_content_info)
{
	g_return_val_if_fail (summary, FALSE);

	if (!skip_mail_message (summary)) {
		return FALSE;
	}

	if (!skip_guint32_decoding (summary->f)) {
		return FALSE;
	}

	if (do_skipping_of_content_info) {
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}
	}

	return TRUE;
}


static gboolean
load_mail_message (SummaryFile *summary, MailMessage *mail_msg, EvolutionAdHocInfos *adhoc_infos)
{
	FILE	*f;
	guint32	flags, size, count;
	time_t	date_sent, date_received;
	char	*uid, *to, *cc, *mlist;

	g_return_val_if_fail (summary, FALSE);
	g_return_val_if_fail (mail_msg, FALSE);

	f = summary->f;

	if (!decode_string (f, &uid) ||
	    !decode_guint32 (f, &flags) ||
	    !decode_guint32 (f, &size) ||
	    !decode_time_t (f, &date_sent) ||
	    !decode_time_t (f, &date_received) ||
	    !decode_string (f, &mail_msg->subject) ||
	    !decode_string (f, &mail_msg->from) ||
	    !decode_string (f, &to) ||
	    !decode_string (f, &cc) ||
	    !decode_string (f, &mlist)) {

		return FALSE;
	}

	g_free (mlist);

	if (adhoc_infos) {
		adhoc_infos->mail_uid = g_strdup (uid);
	}

	g_free (uid);

	if ((flags & EVOLUTION_MESSAGE_DELETED) == EVOLUTION_MESSAGE_DELETED) {
		mail_msg->deleted = TRUE;
	}

	if ((flags & EVOLUTION_MESSAGE_JUNK) == EVOLUTION_MESSAGE_JUNK) {
		mail_msg->junk = TRUE;
	}

	mail_msg->date = (long) date_received;

	mail_msg->to = add_persons_from_internet_address_list_string_parsing (NULL, to);
	mail_msg->cc = add_persons_from_internet_address_list_string_parsing (NULL, cc);

	skip_gint32_decoding (f);	/* mi->message_id.id.part.hi */
	skip_gint32_decoding (f);	/* mi->message_id.id.part.lo */

	/* references */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.hi */
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.lo */
		}
	} else {
		return FALSE;
	}

	/* user flags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* a flag */
		}
	} else {
		return FALSE;
	}

	/* user tags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* tag name */
			skip_string_decoding (f);	/* tag value */
		}
	} else {
		return FALSE;
	}

	return TRUE;
}


static gboolean
skip_mail_message (SummaryFile *summary)
{
	FILE	*f;
	guint32	count;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	if (!skip_string_decoding (f) ||	/* uid */
	    !skip_guint32_decoding (f) ||	/* flags */
	    !skip_guint32_decoding (f) ||	/* size */
	    !skip_time_t_decoding (f) ||	/* date_sent */
	    !skip_time_t_decoding (f) ||	/* date_received */
	    !skip_string_decoding (f) ||	/* subject */
	    !skip_string_decoding (f) ||	/* from */
	    !skip_string_decoding (f) ||	/* to */
	    !skip_string_decoding (f) ||	/* cc */
	    !skip_string_decoding (f)) {	/* mlist */

		return FALSE;
	}

	skip_gint32_decoding (f);	/* mi->message_id.id.part.hi */
	skip_gint32_decoding (f);	/* mi->message_id.id.part.lo */

	/* references */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.hi */
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.lo */
		}
	} else {
		return FALSE;
	}

	/* user flags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* a flag */
		}
	} else {
		return FALSE;
	}

	/* user tags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* tag name */
			skip_string_decoding (f);	/* tag value */
		}
	} else {
		return FALSE;
	}

	return TRUE;
}


static void
free_evolution_adhoc_infos (EvolutionAdHocInfos *adhoc_infos)
{
	if (!adhoc_infos) {
		return;
	}

	if (adhoc_infos->mail_uid) {
		g_free (adhoc_infos->mail_uid);
	}

	g_free (adhoc_infos);
}


static GSList *
add_persons_from_internet_address_list_string_parsing (GSList *list, const char *s)
{
	InternetAddressList *addrs_list, *tmp;

	g_return_val_if_fail (s, NULL);

	addrs_list = internet_address_parse_string (s);

	for (tmp = addrs_list; tmp; tmp = tmp->next) {
		MailPerson *mp;

		mp = email_allocate_mail_person ();

		mp->name = g_strdup (addrs_list->address->name);
		mp->addr = g_strdup (addrs_list->address->value.addr);

		list = g_slist_prepend (list, mp);
	}

	return list;
}


static gboolean
skip_loading_content_info (SummaryFile *summary)
{
	guint32 count, i;

	g_return_val_if_fail (summary, FALSE);

	if (!do_skip_loading_content_info (summary)) {
		return FALSE;
	}

	if (!decode_guint32 (summary->f, &count)) {
		return FALSE;
	}

	if (count > 500) {
		return FALSE;
	}

	for (i = 0; i < count; i++) {
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}
	}

	return TRUE;
}


static gboolean
do_skip_loading_content_info (SummaryFile *summary)
{
	FILE	*f;
	gint8	n;
	guint32	count;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	fread (&n, 1, 1, f);	/* one byte to read... */

	if (n == 0) {
		return TRUE;
	}

	skip_token_decoding (f);	/* type */
	skip_token_decoding (f);	/* subtype */

	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_token_decoding (f);	/* name */
			skip_token_decoding (f);	/* value */
		}
	} else {
		return FALSE;
	}

	if (!skip_token_decoding (f) ||		/* id */
	    !skip_token_decoding (f) ||		/* description */
	    !skip_token_decoding (f) ||		/* encoding */
	    !decode_guint32 (f, &count)) {	/* size */
		return FALSE;
	}

	return TRUE;
}


/***
  Functions to decode summary (or ev-summary) files.
***/

#define MAKE_SKIP_DECODING_FCT(type,size)		\
	static inline gboolean				\
	skip_##type##_decoding (FILE *f)		\
	{						\
		return fseek (f, size, SEEK_CUR) > 0;	\
	}


static inline gboolean
decode_gint32 (FILE *f, gint32 *n)
{
	if (fread (n, 4, 1, f) == 0) {
		return FALSE;
	}

#	if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
	*n = GINT32_FROM_BE (*n);
#	endif

	return TRUE;
}


MAKE_SKIP_DECODING_FCT (gint32, 4);


static inline gboolean
decode_guint32 (FILE *f, guint32 *n)
{
	guint32	value;
	int	v;

	value = 0;

	/* until we get the last byte, keep decoding 7 bits at a time */
	while ( ((v = fgetc (f)) & 0x80) == 0 && v != EOF) {
		value |= v;
		value <<= 7;
	}

	if (v == EOF) {
		*n = value >> 7;
		return FALSE;
	}

	*n = value | (v & 0x7f);

	return TRUE;
}


static inline gboolean
skip_guint32_decoding (FILE *f)
{
	guint32 n;

	/* FIXME: use only fseek() */
	return decode_guint32 (f, &n);
}


static inline gboolean
decode_time_t (FILE *f, time_t *t)
{
	if (fread (t, sizeof (time_t), 1, f) == 0) {
		return FALSE;
	}

#	if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
	/* It is ugly but we need to check that sizeof(time_t)==sizeof(long)
	   since we use the bit swapping function for glong numbers */
	g_assert (sizeof (time_t) == sizeof (glong));
	*t = GLONG_FROM_BE (*t);
#	endif

	return TRUE;
}


MAKE_SKIP_DECODING_FCT (time_t, sizeof (time_t));


static inline gboolean
decode_off_t (FILE *f, off_t *t)
{
	if (fread (t, sizeof (off_t), 1, f) == 0) {
		return FALSE;
	}

#	if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
	/* It is ugly but we need to check that sizeof(off_t)==sizeof(long)
	   since we use the bit swapping function for glong numbers */
	g_assert (sizeof (off_t) == sizeof (glong));
	*t = GLONG_FROM_BE (*t);
#	endif

	return TRUE;
}


MAKE_SKIP_DECODING_FCT (off_t, sizeof (off_t));


static inline gboolean
decode_string (FILE *f, char **str)
{
	guint32		len;
	register char	*ret;

	if (!decode_guint32 (f, &len)) {
		*str = NULL;
		return FALSE;
	}

	len--;

	if (len > 65536) {
		*str = NULL;
		return FALSE;
	}

	ret = g_new0 (char, len + 1);

	if (len > 0 && fread (ret, len, 1, f) != 1) {
		g_free (ret);
		*str = NULL;
		return FALSE;
	}

	ret[len] = '\0';
	*str = ret;

	return TRUE;
}


static inline gboolean
skip_string_decoding (FILE *f)
{
	guint32 len;

	if (!decode_guint32 (f, &len)) {
		return FALSE;
	}

	return fseek (f, len - 1, SEEK_CUR) > 0;
}


static inline gboolean
skip_token_decoding (FILE *f)
{
	guint32 len;

	if (!decode_guint32 (f, &len)) {
		return FALSE;
	}

	if (len < 32) {
		return TRUE;
	}

	len -= 32;

	return fseek (f, len, SEEK_CUR) > 0;
}
