#!/usr/bin/env python

#****************************************************************************
# optiondefaults.py, provides initial config option values
#
# TreeLine, an information storage program
# Copyright (C) 2005, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

import globalref

class OptionDefaults:
    """Creates translated lists of initial default values"""
    modKeys = ['Shift', 'Ctrl', 'Alt', 'Meta']
    keyNames = ['Ins', 'Del', 'Home', 'End', 'PgUp', 'PgDown', \
                'Left', 'Right', 'Up', 'Down']
    # Limits:
    minPrintMargin = 0.25
    maxPrintMargin = 3.0
    maxNumCol = 9
    maxNumLines = 200

    def __init__(self, keyTranslateFunc):
        """keyTranslateFunc is used to translate key names only"""
        self.keyTranslations = {}
        self.revKeyTranslations = {}  # needed only by Qt2
        for key in OptionDefaults.modKeys + OptionDefaults.keyNames:
            self.keyTranslations[key] = keyTranslateFunc(key)
            self.revKeyTranslations[keyTranslateFunc(key)] = key
        self.keyBindList = [\
            (N_('FileNew'),            'Ctrl N'),
            (N_('FileOpen'),           'Ctrl O'),
            (N_('FileOpenSample'),     ''),
            (N_('FileSave'),           'Ctrl S'),
            (N_('FileSaveAs'),         ''),
            (N_('FileExport'),         ''),
            (N_('FilePrint'),          'Ctrl P'),
            (N_('FilePrintOpt'),       ''),
            (N_('FileQuit'),           'Ctrl Q'),
            (N_('EditUndo'),           'Ctrl Z'),
            (N_('EditRedo'),           'Ctrl Y'),
            (N_('EditCut'),            'Ctrl X'),
            (N_('EditCopy'),           'Ctrl C'),
            (N_('EditCopyText'),       ''),
            (N_('EditPaste'),          'Ctrl V'),
            (N_('EditPasteText'),      ''),
            (N_('EditRename'),         'Ctrl R'),
            (N_('EditInBefore'),       'Ctrl B'),
            (N_('EditInAfter'),        'Ctrl I'),
            (N_('EditAddChild'),       'Ctrl A'),
            (N_('EditDelete'),         'Del'),
            (N_('EditIndent'),         'Ctrl Shift Right'),
            (N_('EditUnindent'),       'Ctrl Shift Left'),
            (N_('EditMoveUp'),         'Ctrl Shift Up'),
            (N_('EditMoveDown'),       'Ctrl Shift Down'),
            (N_('ViewDataOutput'),     'Ctrl D'),
            (N_('ViewDataEdit'),       'Ctrl E'),
            (N_('ViewTitleList'),      'Ctrl L'),
            (N_('ViewGoTree'),         'Ctrl G'),
            (N_('ViewDataPageUp'),     'Shift PgUp'),
            (N_('ViewDataPageDown'),   'Shift PgDown'),
            (N_('DataSetItemType'),    'Ctrl T'),
            (N_('DataSetDescendType'), ''),
            (N_('DataConfigType'),     ''),
            (N_('DataCopyTypes'),      ''),
            (N_('DataSortChildren'),   ''),
            (N_('DataSortType'),       ''),
            (N_('DataSortTitle'),      ''),
            (N_('DataFilter'),         ''),
            (N_('DataChange'),         ''),
            (N_('DataNumber'),         ''),
            (N_('DataCategoryAdd'),    ''),
            (N_('DataCategoryFlat'),   ''),
            (N_('DataRefArrange'),     ''),
            (N_('DataRefFlat'),        ''),
            (N_('DataRefUpdate'),      ''),
            (N_('ToolsExpand'),        'Ctrl Right'),
            (N_('ToolsCollapse'),      'Ctrl Left'),
            (N_('ToolsFind'),          'Ctrl F'),
            (N_('ToolsSpellCheck'),    ''),
            (N_('ToolsRemXLST'),       ''),
            (N_('ToolsGenOptions'),    ''),
            (N_('ToolsFileOptions'),   ''),
            (N_('ToolsBackColor'),     ''),
            (N_('ToolsTextColor'),     ''),
            (N_('HelpContents'),       'F1'),
            (N_('HelpFullReadMe'),     ''),
            (N_('HelpAbout'),          ''),
            (N_('HelpPlugin'),         ''),
            (N_('TreeSelectPrev'),     'Up'),
            (N_('TreeSelectNext'),     'Down'),
            (N_('TreeOpenItem'),       'Right'),
            (N_('TreeCloseItem'),      'Left'),
            (N_('TreePrevSibling'),    'Ctrl Up'),
            (N_('TreeNextSibling'),    'Ctrl Down'),
            (N_('TreeSelectParent'),   'Ctrl U'),
            (N_('TreeTop'),            'Home'),
            (N_('TreeBottom'),         'End'),
            (N_('TreePageUp'),         'PgUp'),
            (N_('TreePageDown'),       'PgDown'),
            (N_('TreeIncremSearch'),   '/'),
            (N_('TreeIncremNext'),     'F3'),
            (N_('TreeIncremPrev'),     'Shift F3')]

        self.keyBindList = [(cmd, _(cmd), self.translateKey(key)) for \
                            cmd, key in self.keyBindList]

    def translateKey(self, keySeq):
        return ' '.join([self.keyTranslations.get(key, key) for key in \
                         keySeq.split()])

    def defaultOutput(self):
        """Return a list of tranlated defaults in config file format"""
        defaultList = [\
            _("# Options for TreeLine, an information storage program"),
            "#",
            _("# Path to an alternate plugin directory:"),
            "UserPluginDir       ",
            "#",
            "##############################################################",
            "#",
            _("# Keyboard key bindings"),
            "#",
            _("# Note that key names are case-sensitive"),
            "#",
            _("# Available modifiers include:")]
        defaultList.append('# ' + \
                           ', '.join([self.keyTranslations.get(key, key) for \
                                      key in OptionDefaults.modKeys]))
        defaultList.extend([\
            "#",
            _("# Available keys include:")])
        defaultList.append('# ' + \
                           ', '.join([self.keyTranslations.get(key, key) for \
                                      key in OptionDefaults.keyNames]))
        defaultList.extend([\
            _("# and F1-F12 and any printable character or symbol"),
            "#",
            "##############################################################",
            "#"])

        # add spacers added between key bind sections
        if globalref.lang and globalref.lang[:2] in ['fr', 'de']:
            template = '%-35s %s'  # accomodate long French & German names
        else:
            template = '%-20s %s'
        prev = ''
        for englishName, transName, key in self.keyBindList:
            if prev and prev[:4] != englishName[:4]:
                defaultList.append('#')
            defaultList.append(template % (transName, key))
            prev = englishName

        defaultList.extend([\
            "#",
            "##############################################################",
            "#",
            _("# The following options are set from within the program,"),
            _("# editing here is not recommended"),
            "#",
            "##############################################################",
            "#",
            "AutoFileOpen         no",
            "StartShowChildren    yes",
            "PersistTreeState     yes",
            "SaveWindowGeom       yes",
            "ClickRename          yes",
            "DragTree             yes",
            "InsertOnEnter        yes",
            "RenameNewNodes       yes",
            "OpenSearchNodes      yes",
            "ShowTreeIcons        yes",
            "EnableExecLinks      yes",
            "CompressNewFiles     no",
            "EncryptNewFiles      no",
            "HtmlNewFields        no",
            "#",
            "IndentOffset         20",
            "MaxEditLines         7",
            "UndoLevels           8",
            "AutoSaveMinutes      0",
            "RecentFiles          4",
            "#",
            "EditDateFormat       mm/dd/yyyy",
            "EditTimeFormat       H:MM",
            "#",
            "PrintPrev            yes",
            "PrintRoot            yes",
            "PrintLines           no",
            "PrintOpenOnly        no",
            "PrintKeepFirstChild  yes",
            "PrintUseOutputFont   yes",
            "PrintFont            ",
            "PrintFontSize        ",
            "PrintFontBold        no",
            "PrintFontItalic      no",
            "PrintFontUnderline   no",
            "PrintFontStrikeOut   no",
            "PrintPageSize        letter",
            "PrintNumCols         1",
            "PrintColSpace        0.30",
            "PrintOffset          20",
            "HorizMargin          0.50",
            "VertMargin           0.70",
            "PrintUnits           inch",
            "#",
            "BackgroundR          255",
            "BackgroundG          255",
            "BackgroundB          255",
            "ForegroundR          0",
            "ForegroundG          0",
            "ForegroundB          0",
            "#",
            "TreeFont             ",
            "TreeFontSize         ",
            "TreeFontBold         no",
            "TreeFontItalic       no",
            "TreeFontUnderline    no",
            "TreeFontStrikeOut    no",
            "OutputFont           ",
            "OutputFontSize       ",
            "OutputFontBold       no",
            "OutputFontItalic     no",
            "OutputFontUnderline  no",
            "OutputFontStrikeOut  no",
            "EditorFont           ",
            "EditorFontSize       ",
            "EditorFontBold       no",
            "EditorFontItalic     no",
            "EditorFontUnderline  no",
            "EditorFontStrikeOut  no",
            "#",
            "ToolbarData          [1,2,0,0,0],[1,2,1,0,0]",
            "RecentFile1          ",
            "RecentFile2          ",
            "RecentFile3          ",
            "RecentFile4          ",
            "TreeState1           0:0:0:0",
            "TreeState2           0:0:0:0",
            "TreeState3           0:0:0:0",
            "TreeState4           0:0:0:0",
            "SpellCheckPath       ",
            "ExtEditorPath        ",
            "#",
            "WindowXSize          540",
            "WindowYSize          460",
            "WindowXPos           0",
            "WindowYPos           0",
            "TreeSplitPercent     40",
            "OutputSplitPercent   20",
            "EditorSplitPercent   25",
            "TitleSplitPercent    10",
            "ActiveRightView      0",
            "PrintPrevXSize       300",
            "PrintPrevYSize       560",
            "PrintPrevXPos        0",
            "PrintPrevYPos        0"])
        return defaultList
