//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 06/09/2005
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef Tulip_GLGRAPH_H
#define Tulip_GLGRAPH_H

#ifndef NDEBUG
#define TRACE_EXEC()	
//	cerr << __PRETTY_FUNCTION__ << endl;
#else
#define TRACE_EXEC()
#endif

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include <GL/gl.h>
#include <tulip/PluginContext.h>
#include <tulip/Observable.h>
#include <tulip/Coord.h>
#include <tulip/Color.h>
#include <tulip/SuperGraph.h>
#include <tulip/Types.h>
#include <tulip/TemplateFactory.h>
#include <tulip/Reflect.h>
#include <tulip/MutableContainer.h>
#include <tulip/hash_string.h>
#include <tulip/tulipconf.h>
#include "tulip/Camera.h"
#include "tulip/OcclusionTest.h"
#include "tulip/ObservableGlGraph.h"
#include <tulip/Matrix.h>


class LayoutProxy;
class MetricProxy;
class StringProxy;
class IntProxy;
class SelectionProxy;
class SizesProxy;
class ColorsProxy;
class MetaGraphProxy;
class PropertyManager;
class Glyph;
class GlyphFactory;
class GlyphContext;
class OcclusionTest;
class PluginLoader;
class TextRenderer;


enum DrawState { DRAWNODE = 0, DRAWEDGE = 1, DRAWSELECTEDNODELABELS = 2, DRAWSELECTEDEDGELABELS = 3,
		DRAWNODELABELS = 4 , DRAWEDGELABELS = 5 , DRAWEND = 6};

enum EdgeShape {POLYLINESHAPE = 0, BEZIERSHAPE = 4, SPLINESHAPE = 8};

/**
 * This widget enables an incremental 3D drawing of a Tulip SuperGraph.
 *  It also includes 3D navigation capabilities and selection methods.
 *  All the displaying is done with OpenGl.
 */
class TLP_GL_SCOPE GlGraph : public GraphObserver, public ObservableGlGraph {
  
public:
  GlGraph();
  GlGraph(const GlGraph &);
  ~GlGraph();
  //============================================
  /**
   * This function must be call before using the opengl
   * rendering engine.
   */
  void initializeGL();
  //============================================
  DataSet getParameters() const;
  void    setParameters(const DataSet);
  /**
   *  Initialize scale factors according to layout bounding box
   *  and the camera position in order to have the graph at the 
   *  center of the screen
   */
  void centerScene();
  /**
   * Change the dimension of the drawing (screen size), and enable redrawing.
   */
  void changeViewport(const unsigned x, const unsigned y, const unsigned int width, const unsigned int height);
  /**
   * Draw a part of the graph, the number of elements displayed
   * is automatically computed by analysing the time used to draw
   * the previous slice of data. If all the slice have been drawn
   * return false;
   */
  bool drawPart();
  /**
   * Draw the entire graph.
   */
  void draw();
  /**
   * copy current values into given parameters (names are corresponding)
   */
  virtual void getWinParameters(int *winX, int *winY, int *winW, int *winH, GLint *vp[4]);
  //==============================================================================
  // Data centralization in order to optimize memory using
  //==============================================================================
  /** activate Texture with name filename.
   * \param filename the image to be textured
   * \return true on success, false otherwise (format not supported, file not found,
             not enough memory, ...)
   */
  bool activateTexture(const std::string &filename);
  /*
   * Desactivate texturing
   */
  void desactivateTexture();
  //==============================================================================
  // Rendering parameters
  //==============================================================================
  /** return a Camera object which desribes the 
      camera used for he rendering
  */
  Camera getCamera() const;
  /** 
   * set the camera parameter for the rendering
   */
  void setCamera(const Camera &cam);
  /** return the background color
   * \sa setBackgroundColor
   */
  Color getBackgroundColor() const;
  /** set the background color, RGB colorspace.
   * \sa getBackgroundColor
   */
  void setBackgroundColor(const int r,const int g, const int b);
   /** set the background color, RGB colorspace.
   * \sa getBackgroundColor
   */
  void setBackgroundColor(const Color &);
  /** activate or deactivate displaying of arrows along the graph's edges.
   * \sa isViewArrow
   */
  void setViewArrow(const bool state);
  /** return true if arrows drawing is activated else false.
   * \sa setViewArrow
   */
  bool isViewArrow()const;
  /** activate or deactivate displaying of labels on nodes and edges.
   *  \sa isViewLabel
   */
  void setViewLabel(const bool state);
  /** return true if label drawing is on else false.
   *  \sa setViewLabel
   */
  bool isViewLabel() const;
  /** activate or deactivate interpolation of nodes colors along edge
   *  \sa isEdgeColorInterpolate
   */
  void setEdgeColorInterpolate(const bool state);
  /** return true if color interpolation is activated
   * \sa setEdgeColorInterpolate
   */
  bool isEdgeColorInterpolate() const;
  /** activate or deactivate interpolation of nodes colors along edge
   *  \sa isEdgeColorInterpolate
   */
  void setEdgeSizeInterpolate(const bool state);
  /** return true if color interpolation is activated
   * \sa setEdgeColorInterpolate
   */
  bool isEdgeSizeInterpolate() const;

  /** return the type of fonts used in the renderer(0=3D,1=bitmap,2=texture)
   *  \sa setFontsType
   */
  unsigned int fontsType() const;
  /** set fonts type used in the renderer (0=3D,1=bitmap,2=texture)
   * \sa fontsType
   */
  void setFontsType(unsigned int type);
  /** set the path for the fonts loading
   */
  void setFontsPath(std::string path);
  /** if true activate the Orthogonal projection, else perspective is used.
   * \sa isViewOrtho
   */
  void setViewOrtho(const bool state);
  /** return true if orthogonal projection is used.
   *  \sa isViewOrtho
   */
  bool isViewOrtho() const;
  /** if true glgraph use the incremental rendering else display all elements in one pass
   *  \sa isIncrementalRendering
   */
  void setIncrementalRendering(const bool state);
  /** return the incremental rendering state
   *  \sa setIncrementalRendering
   */
  bool isIncrementalRendering() const;

  /*If true and if labels are activated label inside meta nodes will be displayed */
  void setViewMetaLabel(const bool state);
  bool isViewMetaLabel() const;

  void setDisplayEdges(const bool state);
  bool isDisplayEdges() const;
  /** activate Strahler drawing mode. The graph is then drawn in Strahler order
   *  \sa isViewStrahler
   */
  void setViewStrahler(const bool state);
  /** return Strahler mode state
   *  \sa isViewStrahler
   */
  bool isViewStrahler() const;
  /** use 3D displaying of edges instead of 2D lines on ALL edges
   * \sa isEdged3D
   */
  void setEdge3D(const bool state);
  /** return true is 3D drawing of edges is activated
   * \sa setEdge3D
   */
  bool isEdged3D() const;
  /** get current translation applied to the scene
   * \return current tranlation of the scene
   * \sa setSceneTranslation, setTranslation
   */
  Coord getSceneTranslation() const;
  /** set translation to apply to the scene
   *  \sa getSceneTranslation, setTranslation
   */
  void setSceneTranslation(const Coord &translation);
  /** get current rotation applied to the scene
   *  \return current rotation of the scene
   *  \sa setSceneRotation, setRotation
   */  
  Coord getSceneRotation() const;
  /** set rotation to apply to the scene
   *  \sa getSceneRotation, setRotation
   */
  void setSceneRotation(const Coord &rotation);
  /** 
   * Get the size of the border that is added to any label when
   * using bitmap fonts.
   */
  unsigned int getLabelsBorder() const;
  /** 
   * Get the size of the border that is added to any label when
   * using bitmap fonts. Setting this parameter to a high value
   * enables de reduce the label density in the final drawing.
   * Default value is set to 2 which enables readable labels.
   */
  void setLabelsBorder(const unsigned int);
  //=======================================================================
  // Navigation functions: GlGraphNavigate.cpp
  //=======================================================================
  /** set translation to apply to the scene
   * \sa getSceneTranslation, setSceneTranslation
   */
  void translateCamera(const int x, const int y, const int z);
  /** set rotation to apply to the scene
   *  \sa getSceneRotation, setSceneRotation
   */
  void rotateScene(const int rotx, const int roty, const int rotz);
  /** zoom on the center of the screen
   *  \param step positive: zoom in, negative: zoom out
   */
  void zoom(const int step);
  /** \brief zoom to  screen coordinates
   *  a translation is performed while zooming. The farther the point (x,y) is
   *  from the center of the screen, the higher the translation is. The direction
   *  of the translation is reversed when switching from zoom in to zoom out.
   *  \sa zoom
   */
  void zoomXY(const int step, const int x, const int y);
  //=======================================================================
  //Selection of elements on the screen
  //=======================================================================
  /** \brief select nodes and edges in a region of the screen
   *
   *  select all nodes and edges lying in the area of the screen of given width and height,
   *  and with its upper-left corner at (x,y)
   *  \param sNode filled by the method with the nodes found in the region
   *  \param sEdge filled by the method with the edges found in the region
   */
  void doSelect(const int x, const int y, const int width, const int height, std::vector<node> &sNode, std::vector<edge> &sEdge);
   /** \brief select a node or edge at a point
    *  select either a node or edge at point (x,y)
    *  \param type tells what has been found: NODE, EDGE
    *  \return true if something has been found, false otherwise
    */
  bool doSelect(const int x, const int y, tlp::ElementType &type, node &, edge &);
  /** \brief select nodes in a region of the screen
   *  \param selected the nodes found
   *  \return true if at least one node has been found
   */
  bool doNodeSelect(const int x, const int y, const int width, const int height, std::vector<node> &selected, const bool ordered = true);
  /** select edges at a given point, ordered from the closest to the farthest
   *  \param selected the edges found
   *  \return true if at least one edge has been found
   */
  bool doEdgeSelect(const int x, const int y, const int width, const int height, std::vector<edge> &selected, const bool ordered = true);
  //=======================================================================
  //Tools
  //=======================================================================
  //FIXME: mode outputEPS outside of GlGraph
  void outputEPS(int size, int doSort, const char *filename);
  /**
   * Compute coordinate from the 2D screen space to the 
   * 3D space.
   */
  void screenTo3DWorld(float& x, float& y, float& z);
  /**
   * Compute coordinate from the 3D space to the 
   * 2D screen space.
   */
  void worldTo2DScreen(float& x, float& y, float& z);
  /** 
   *  Take a snapshot of the Open GL windows
   *  \return an array of dimension width*height*3 char (8bits per color RGB).
   *          The pointer has to be freed after (with free, not delete)
   **/
  unsigned char *getImage(int &width, int &height);
  //=======================================
  // OpenGl Window
  //=======================================
  /**
   * Compute camera position in order to see the graph
   * in the center of the scene
   */
  void goodScale();
  //=========================================
  //Operation on graph's structure
  //=========================================
  /** Change the graph being viewed
   *  \param a pointer to the graph to view
   *  \sa getSuperGraph
   */
  void setSuperGraph(SuperGraph *supergraph);
  /** get the graph being viewed
   *  \return a pointer to the current graph
   *  \sa setSuperGraph
   */
  SuperGraph* getSuperGraph() const;


  /**
   * Define the name of the layout that is used
   * for rendering the graph (default is "viewLayout")
   */
  void setInputLayout(const std::string &layoutName);

  static TemplateFactory<GlyphFactory,Glyph,GlyphContext *> *glyphFactory;
  static TLP_GL_SCOPE void initFactory() {
    if (!glyphFactory) {
      glyphFactory = new TemplateFactory<GlyphFactory,Glyph,GlyphContext *>;
      glyphFactory->currentLoader = 0;
    }
  }
  static std::string glyphName(int id);
  static int  glyphId(std::string name);
  static void loadPlugins(PluginLoader *plug=0);
  static std::string edgeShapeName(int id);
  static int edgeShapeId(std::string name);
  static const int edgeShapesCount;
  static int edgeShapeIds[];
  GLuint (*selectBuf)[4];
  void initDoSelect(const GLint x, const GLint y, GLint w, GLint h);
  void endSelect();

private:
  static stdext::hash_map<int,std::string>  idToName;
  static stdext::hash_map<std::string, int> nameToId;

protected:
  void addNode (SuperGraph *, const node);
  void addEdge (SuperGraph *, const edge);
  void delNode (SuperGraph *, const node);
  void delEdge (SuperGraph *, const edge);
  void destroy (SuperGraph *);
  void drawPixmapFont(const std::string &str, const Color &col, const Coord &position, bool selected, float width);
  void drawEdge(const Coord &startNode, const Coord &finalNode,
                const Coord &startPoint,const LineType::RealType &bends, const Coord &endPoint,
		const Color &startColor, const Color &endColor, const Size &size, int shape, const bool selected=false);
  void drawEdge(edge ite);
  unsigned int drawEdges(unsigned int , Iterator<edge> *, unsigned int = 0);
  void drawNode(node itv,unsigned int depth);
  unsigned int drawNodes(unsigned int , Iterator<node> *, unsigned int = 0);
  unsigned int drawNodeLabels(unsigned int , Iterator<node> *, bool, unsigned int =0);
  void drawNodeLabel(node itn, bool, unsigned int);
  void drawEdgeLabel(edge e, bool);
  unsigned int drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool, unsigned int =0);
  void drawFanNode(node n) ;
  unsigned int drawFanNodes(unsigned int , Iterator<node> *);
  void drawMetaNode(node n,unsigned int depth, bool = false, bool = false);
  void DrawBitmapString(void *, const char *);
  void DrawStrokeString(void *font, const char *string);
  void makeArrowMatrix(GLfloat *, const Coord , const Coord , GLfloat, GLfloat, GLfloat);
  void initProxies();
  void deleteIterators();
  void initIterators();
  void makeNodeSelect(const int);
  void makeEdgeSelect(const int);
  void initProjection(bool reset=true);
  void initModelView();
  void initLights();
  void initGlParameter();
  void buildOrderedList();
  void buildDisplayLists();
  void deleteDisplayLists();
  void transparentSphere();
  void cube(GLenum type);
  void solidCone();


private:
  // GlGraphStrategy *strategy;
  SuperGraph     *_superGraph;
  Color  backgroundColor;

  bool _viewArrow, _viewLabel, _viewMetaLabel, _viewStrahler;
  bool _viewAutoScale, _incrementalRendering, _edgeColorInterpolate, _edge3D;
  bool _edgeSizeInterpolate;
  unsigned int _viewOrtho;
  unsigned int _FontsType;
  std::string fontsPath;
  bool _displayEdges;
  
  //  stdext::hash_map<int, Glyph *, stdext::hash<int> >  glyphTable;


  stdext::hash_map<std::string, GLuint > texturesMap; //FIXME: make it static?
  

  GLint viewportArray[4];
  GLdouble modelviewMatrix[16];
  GLdouble projectionMatrix[16];
  GLdouble transformMatrix[16];


  //FIXME: made public to allow glyphs to access those proxy
public:
  ColorsProxy    *elementColor;
  ColorsProxy    *elementLabelColor;
  SizesProxy     *elementSize;
  IntProxy       *elementShape;
  MetricProxy    *elementRotation;
  SelectionProxy *elementSelected;
  StringProxy    *elementLabel;
  LayoutProxy    *elementLayout;
  MetaGraphProxy *elementMetaGraph;
  StringProxy	 *elementTexture;

private:

  //we never have to get a copy of that GlGraph.
  //GlGraph objects should be only passed by reference or pointer
  static void loadPlugins(std::string dir, PluginLoader *plug);

private:

  //predefined display list
  GLuint metaGraphDL;
  GLuint selectionDL;
  GLuint arrowDL;
  //glyphs
  MutableContainer<Glyph *> glyphs;
  //Scene information
  Coord sceneTranslation; // Scene translation
  Coord sceneRotation; // Rotation of the scene
  //Camera information
  Coord   cameraEyes,cameraCenter,cameraUp;
  double  cameraZoomFactor;
  GLfloat distCam;   //distance for camera translation
  //viewport position and size
  unsigned int winX, winY; 
  unsigned int winH, winW;

  //+++
  Iterator<node>* drawNodesIterator;
  Iterator<node>* drawLabelsIterator;
  Iterator<node>* drawSelectedLabelsIterator;
  Iterator<edge>* drawEdgesIterator;
  Iterator<edge>* drawEdgeLabelsIterator;
  Iterator<edge>* drawEdgeSelectedLabelsIterator;

  std::list<node> orderedNode;
  std::list<edge> orderedEdge;

  //Incremental rendering parameters
  int maxNumberOfNodeToDraw;
  int maxNumberOfEdgeToDraw;

  //
  MutableContainer<unsigned int> nbElementsFrame;
  MutableContainer<bool> nodesRenderedAsPoint;
  unsigned int currentFrame;
  unsigned drawState;
  unsigned int labelsBorder;

  //Occlusion testing
  OcclusionTest occlusionTest;

  //
  std::string layoutName;
  TextRenderer * fontRenderer;

  //----------------------------
  // from GlGraphStrategy
  //----------------------------
public: 
  //GlGraph has to call this before any GL operation
  virtual void makeCurrent()=0;
  virtual void updateGL()=0;

  //timer methods
  virtual bool timerIsActive()=0;
  virtual int timerStart(int msec, bool sshot=false)=0;
  virtual void timerStop()=0;

  virtual void mPaint()=0;

  bool toUpdate;
};


template<typename T, typename IT> struct stlListIterator:public Iterator<T> {
  IT it,itEnd;
  stlListIterator(const IT &startIt, const IT &endIt):it(startIt),itEnd(endIt){}
  T next(){T tmp=*it;++it;return tmp;}
  bool hasNext(){return (itEnd!=it);}
};

#endif // Tulip_GLGRAPH_H
