#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>

#include <tulip/SuperGraph.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SelectionProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/GlGraphWidget.h>
#include <tulip/DrawingTools.h>
#include <tulip/GlHudRect.h>
#include <tulip/GlHudRect4.h>

#include <tulip/MouseMoveSelection.h>

#if (QT_REL == 3)
#include <qevent.h>
// include below is for compilation purpose only
#include <qcursor.h>
#else
#include <QtGui/qevent.h>
#include "tulip/Qt3ForTulip.h"
#endif

#define EPSILON 1.0
#define EPSILON_SCREEN 50
#define EPSILON_STRETCH_MIN 1 - 1.0e-01
#define EPSILON_STRETCH_MAX 1 + 1.0e-01

using namespace tlp;

//========================================================================================
MouseMoveSelection::MouseMoveSelection() {
  operation = NONE;
  _copyLayout = 0;
  _copySizes = 0;
  _copyRotation = 0;

  ffd.addGlAugmentedDisplay(&controlRects[0], "ControlRect1");
  ffd.addGlAugmentedDisplay(&controlRects[1], "ControlRect2");
  ffd.addGlAugmentedDisplay(&controlRects[2], "ControlRect3");
  ffd.addGlAugmentedDisplay(&controlRects[3], "ControlRect4");
  ffd.addGlAugmentedDisplay(&controlRects[4], "ControlRect5");
  ffd.addGlAugmentedDisplay(&controlRects[5], "ControlRect6");
  ffd.addGlAugmentedDisplay(&controlRects[6], "ControlRect7");
  ffd.addGlAugmentedDisplay(&controlRects[7], "ControlRect8");
  ffd.addGlAugmentedDisplay(&centerRect, "CenterRectangle");
}
//========================================================================================
MouseMoveSelection::~MouseMoveSelection() {
}
//========================================================================================
void MouseMoveSelection::mPressEvent(GlGraphWidget *glGraphWidget, QMouseEvent *qMouseEv) {   
  initProxies(glGraphWidget);
  computeFFD(glGraphWidget);

  editCenter[0] = (centerRect.getPosition(0).getX() + centerRect.getPosition(2).getX()) / 2.0 ;
  editCenter[1] = (centerRect.getPosition(0).getY() + centerRect.getPosition(2).getY()) / 2.0 ;
  editCenter[2] = 0;

  editPostion[0] = qMouseEv->x();
  editPostion[1] = qMouseEv->y();
  editPostion[2] = 0;

  editLayoutCenter = _layoutCenter;

  x = qMouseEv->x();
  y = qMouseEv->y();


  switch(qMouseEv->button()) {
  case Qt::LeftButton : 
    //left <-> right anchor
    // strect_x
    if (controlRects[0].inRect(x,y) || controlRects[4].inRect(x,y)) {
      operation = STRETCH_X;
      glGraphWidget->setCursor(QCursor(Qt::SizeHorCursor));
    }else
    //top <-> bottom anchor
    // strect_y
    if (controlRects[2].inRect(x,y) || controlRects[6].inRect(x,y)) {
      operation = STRETCH_Y;
      glGraphWidget->setCursor(QCursor(Qt::SizeVerCursor));
    }
    else
    //Corner anchor bottom-right top-left
    // rotate
    if (controlRects[3].inRect(x,y) || controlRects[7].inRect(x,y)) {
      glGraphWidget->setCursor(QCursor(Qt::PointingHandCursor));
      operation = ROTATE;
    }
    else
    //Corner anchor top-right bottom-left 
    //strect_xy
    if (controlRects[1].inRect(x,y) || controlRects[5].inRect(x,y)) {
      operation = STRETCH_XY;
      glGraphWidget->setCursor(QCursor(Qt::SizeFDiagCursor));

    }
    else {
      operation = TRANSLATE;
      glGraphWidget->setCursor(QCursor(Qt::SizeAllCursor));
    }
    
    if (qMouseEv->state() & Qt::ShiftButton)
      mode = COORD;
    else
      mode = COORD_AND_SIZE;
    initEdition();
    break;
  case Qt::MidButton :
    undoEdition();
    glGraphWidget->setCursor(QCursor(Qt::ArrowCursor));
    break;
  default: break;
  }
}
//========================================================================================
void MouseMoveSelection:: mReleaseEvent(GlGraphWidget *glGraphWidget,QMouseEvent *qMouseEv) {
  stopEdition();
  glGraphWidget->setCursor(QCursor(Qt::ArrowCursor));
}
//========================================================================================
void MouseMoveSelection:: mPaint(GlGraphWidget *glGraphWidget) {
  computeFFD(glGraphWidget);
  ffd.draw(glGraphWidget);
}
//========================================================================================
void MouseMoveSelection:: mMoveEvent(GlGraphWidget *glGraphWidget,QMouseEvent *qMouseEv)   { 
  switch (operation) {
  case STRETCH_X:
  case STRETCH_Y:
  case STRETCH_XY:
    mMouseStretchAxis(qMouseEv->x(), qMouseEv->y(), glGraphWidget);
    break;
  case ROTATE:
    mMouseRotate(qMouseEv->x(), qMouseEv->y(), glGraphWidget);
    break;
  case TRANSLATE:
    mMouseTranslate(qMouseEv->x(), qMouseEv->y(), glGraphWidget);
    break;
  case NONE:
    cerr << "[Error] : " <<__FUNCTION__ << " should not be call" << endl;
    break;
  }
}
//========================================================================================
void MouseMoveSelection::restoreInfo() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  assert(_copyLayout != 0);
  assert(_copySizes != 0);
  assert(_copyRotation != 0);
  Iterator<node> *itN = _selection->getNodesEqualTo(true, _superGraph);
  while(itN->hasNext()) {
    node n = itN->next();
    _rotation->setNodeValue(n, _copyRotation->getNodeValue(n));
    _layout->setNodeValue(n, _copyLayout->getNodeValue(n));
    _sizes->setNodeValue(n, _copySizes->getNodeValue(n));
  } delete itN;
  Iterator<edge> *itE = _selection->getEdgesEqualTo(true, _superGraph);
  while(itE->hasNext()) {
    edge e = itE->next();
    _rotation->setEdgeValue(e, _copyRotation->getEdgeValue(e));
    _layout->setEdgeValue(e, _copyLayout->getEdgeValue(e));
    _sizes->setEdgeValue(e, _copySizes->getEdgeValue(e));
  } delete itE;
}
//========================================================================================
void MouseMoveSelection::saveInfo() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  assert(_copyLayout == 0);
  assert(_copySizes == 0);
  assert(_copyRotation == 0);
  _copyRotation = new MetricProxy(_superGraph);
  _copyLayout = new LayoutProxy(_superGraph);
  _copySizes = new SizesProxy(_superGraph);
  Iterator<node> *itN = _selection->getNodesEqualTo(true, _superGraph);
  while(itN->hasNext()) {
    node n = itN->next();
    _copyRotation->setNodeValue(n, _rotation->getNodeValue(n));
    _copyLayout->setNodeValue(n, _layout->getNodeValue(n));
    _copySizes->setNodeValue(n, _sizes->getNodeValue(n));
  } delete itN;
  Iterator<edge> *itE = _selection->getEdgesEqualTo(true, _superGraph);
  while(itE->hasNext()) {
    edge e = itE->next();
    _copyRotation->setEdgeValue(e, _rotation->getEdgeValue(e));
    _copyLayout->setEdgeValue(e, _layout->getEdgeValue(e));
    _copySizes->setEdgeValue(e, _sizes->getEdgeValue(e));
  } delete itE;
}
//========================================================================================
void MouseMoveSelection::initEdition() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  saveInfo();
}
//========================================================================================
void MouseMoveSelection::undoEdition() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (operation == NONE) return;
  restoreInfo();
  operation = NONE;
  delete _copyLayout; _copyLayout = 0;
  delete _copySizes; _copySizes = 0;
  delete _copyRotation; _copyRotation = 0;

  
}
//========================================================================================
void MouseMoveSelection::stopEdition() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (operation == NONE) return;
  operation = NONE;
  delete _copyLayout; _copyLayout = 0;
  delete _copySizes; _copySizes = 0;
  delete _copyRotation; _copyRotation = 0;

}
//========================================================================================
void MouseMoveSelection::initProxies(GlGraphWidget *glGraphWidget) {
  _superGraph = glGraphWidget->getSuperGraph();
  _layout = _superGraph->getProperty<LayoutProxy>("viewLayout");
  _selection = _superGraph->getProperty<SelectionProxy>("viewSelection");
  _rotation = _superGraph->getProperty<MetricProxy>("viewRotation");
  _sizes = _superGraph->getProperty<SizesProxy>("viewSize");
}
//========================================================================================
void MouseMoveSelection::mMouseTranslate(double mX, double mY, GlGraphWidget *glGraphWidget) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  Observable::holdObservers();
  initProxies(glGraphWidget);
  Coord v0(0,0,0);
  Coord v1((double)(x - mX), -(double)(y - mY),0);
  glGraphWidget->screenTo3DWorld(v0[0], v0[1], v0[2]);
  glGraphWidget->screenTo3DWorld(v1[0], v1[1], v1[2]);
  v1 -= v0;
  Iterator<node> *itN = _selection->getNodesEqualTo(true, _superGraph);
  Iterator<edge> *itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->translate(v1, itN, itE);
  delete itN;
  delete itE;
  x  = (int)mX;
  y  = (int)mY;
  Observable::unholdObservers();
}
//========================================================================================
void MouseMoveSelection::mMouseStretchAxis(double mX, double mY, GlGraphWidget* glGraphWidget) {
  //  cerr << __PRETTY_FUNCTION__ << "/op=" << operation << ", mod:" << mode << endl;
  Coord curPos(mX, mY, 0);
  Coord strecth(1,1,1);
  if (operation == STRETCH_X || operation == STRETCH_XY) {
    strecth[0] = (curPos[0] - editCenter[0]) / (editPostion[0] - editCenter[0]);
  }
  if (operation == STRETCH_Y || operation == STRETCH_XY) {
    strecth[1] = (curPos[1] - editCenter[1]) / (editPostion[1] - editCenter[1]);
  }
  Observable::holdObservers();  
  restoreInfo();
  //strecth layout
  Coord center(editLayoutCenter);
  center *= -1.;
  Iterator<node> *itN = _selection->getNodesEqualTo(true, _superGraph);
  Iterator<edge> *itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->translate(center, itN, itE);
  delete itN; delete itE;
  itN = _selection->getNodesEqualTo(true, _superGraph);
  itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->scale(strecth, itN, itE);
  delete itN; delete itE;
  center *= -1.;
  itN = _selection->getNodesEqualTo(true, _superGraph);
  itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->translate(center, itN, itE);
  delete itN; delete itE;
  //strecth size
  if (mode == COORD_AND_SIZE) {
    itN = _selection->getNodesEqualTo(true, _superGraph);
    itE = _selection->getEdgesEqualTo(true, _superGraph);
    _sizes->scale(strecth, itN, itE);
    delete itN; delete itE;
  }
  Observable::unholdObservers();
}
//========================================================================================
void MouseMoveSelection::mMouseRotate(double mX, double mY, GlGraphWidget *glGraphWidget) { 
  //  cerr << __PRETTY_FUNCTION__ << endl;

  Coord curPos(mX, mY, 0);

  Coord vCS = editPostion - editCenter;
  vCS /= vCS.norm();
  Coord vCP =  curPos - editCenter;
  vCP /= vCP.norm();

  float sign = (vCS ^ vCP)[2];
  sign /= fabs(sign);
  double cosalpha = vCS.dotProduct(vCP);
  double deltaAngle = sign * acos(cosalpha);
  
  Observable::holdObservers();
  
  initProxies(glGraphWidget);
  restoreInfo();

  double degAngle = (deltaAngle * 180.0 / M_PI);
  //rotate layout
  Coord center(editLayoutCenter);
  center *= -1.;
  Iterator<node> *itN = _selection->getNodesEqualTo(true, _superGraph);
  Iterator<edge> *itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->translate(center, itN, itE);
  delete itN; delete itE;
  itN = _selection->getNodesEqualTo(true, _superGraph);
  itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->rotateZ(-degAngle, itN, itE);
  delete itN; delete itE;
  center *= -1.;
  itN = _selection->getNodesEqualTo(true, _superGraph);
  itE = _selection->getEdgesEqualTo(true, _superGraph);
  _layout->translate(center, itN, itE);
  delete itN; delete itE;

  if (mode == COORD_AND_SIZE) {
    itN = _selection->getNodesEqualTo(true, _superGraph);
    while(itN->hasNext()) {
      node n = itN->next();
      double rotation = _rotation->getNodeValue(n);
      _rotation->setNodeValue(n, rotation - degAngle);
    } delete itN;
  }

  Observable::unholdObservers();
}
//========================================================================================
Coord minCoord(const Coord &v1, const Coord &v2) {
  Coord result;
  for (unsigned int i =0; i<3; ++i)
    result[i] = std::min(v1[i], v2[i]);
  return result;
}
Coord maxCoord(const Coord &v1, const Coord &v2) {
  Coord result;
  for (unsigned int i =0; i<3; ++i)
    result[i] = std::max(v1[i], v2[i]);
  return result;
}
//========================================================================================
void MouseMoveSelection::computeFFD(GlGraphWidget *glGraphWidget) {
  // We calculate the bounding box for the selection :
  initProxies(glGraphWidget);

  pair<Coord, Coord> boundingBox = tlp::computeBoundingBox(_superGraph, _layout, _sizes, _rotation, _selection);

  Coord min2D, max2D;

  _layoutCenter = (boundingBox.first + boundingBox.second) / 2.0;
  //project the 8 points of the cube to obtain the bounding square on the 2D screen
  Coord bbsize = (boundingBox.first - boundingBox.second);
  //v1
  Coord tmp(boundingBox.second);
  glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
  min2D = tmp;
  max2D = tmp;
  //v2, v3, V4
  for (unsigned int i=0; i<3; ++i) {
    tmp = boundingBox.second;
    tmp[i] += bbsize[i];
    glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
    min2D = minCoord(tmp, min2D);
    max2D = maxCoord(tmp, max2D);
  }
  //v4
  tmp = boundingBox.second;
  tmp[0] += bbsize[0];
  tmp[1] += bbsize[1];
  glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
  min2D = minCoord(tmp, min2D);
  max2D = maxCoord(tmp, max2D);
  //v6
  tmp = boundingBox.second;
  tmp[0] += bbsize[0];
  tmp[2] += bbsize[2];
  glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
  min2D = minCoord(tmp, min2D);
  max2D = maxCoord(tmp, max2D);
  //v7
  tmp = boundingBox.second;
  tmp[1] += bbsize[1];
  tmp[2] += bbsize[2];
  glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
  min2D = minCoord(tmp, min2D);
  max2D = maxCoord(tmp, max2D);
  //v8
  tmp = boundingBox.second;
  tmp += bbsize;
  glGraphWidget->worldTo2DScreen(tmp[0], tmp[1], tmp[2]);
  min2D = minCoord(tmp, min2D);
  max2D = maxCoord(tmp, max2D);

  ffdCenter = (boundingBox.first + boundingBox.second) / 2.0;
  
  Color hudColor(128, 128, 128, 255);
 
  int X, Y, W, H;
  GLint *vp;

  vp = new GLint[4];  
  glGraphWidget->getWinParameters(&X, &Y, &W, &H, &vp);
  delete [] vp;

  min2D[1] = (double)H - min2D[1];
  max2D[1] = (double)H - max2D[1];

  Coord tmpCenter(ffdCenter);

  glGraphWidget->worldTo2DScreen(tmpCenter[0], tmpCenter[1], tmpCenter[2]);
   
  Coord recCenter(tmpCenter[0], tmpCenter[1], tmpCenter[2]);

  recCenter[1] = (double)H - recCenter[1];
  
  int x = int(max2D[0] - min2D[0]) / 2;
  int y = int(max2D[1] - min2D[1]) / 2;
  
  Coord center, topLeft, bottomRight;

  Coord positions[8];

  positions[0] = Coord( x,  0, 0) + recCenter; // Right
  positions[1] = Coord( x, -y, 0) + recCenter; // Top Right
  positions[2] = Coord( 0, -y, 0) + recCenter; // Top
  positions[3] = Coord(-x, -y, 0) + recCenter; // Top Left
  positions[4] = Coord(-x,  0, 0) + recCenter; // Left
  positions[5] = Coord(-x,  y, 0) + recCenter; // Bottom Left
  positions[6] = Coord( 0,  y, 0) + recCenter; // Bottom
  positions[7] = Coord( x,  y, 0) + recCenter; // Bottom Right

  //  centerRect = new GlHudRect4(positions[3], positions[1], positions[7], positions[5], hudColor);
  centerRect.setPosition(0, positions[3]);
  centerRect.setPosition(1, positions[1]);
  centerRect.setPosition(2, positions[7]);
  centerRect.setPosition(3, positions[5]);
  centerRect.setColor(0, hudColor);
  centerRect.setColor(1, hudColor);
  centerRect.setColor(2, hudColor);
  centerRect.setColor(3, hudColor);
  centerRect.setRenderState(GlAD_ZEnable, false);
  centerRect.setRenderState(GlAD_Culling, false);
  centerRect.setRenderState(GlAD_Wireframe, true);
  centerRect.setRenderState(GlAD_Solid, false);
  Color hudColor2(0, 255, 0, 255);
  for(unsigned int i=0; i < 8; ++i) {
    center = positions[i];
    // Then we find the position of the topLeft and the bottomRight corner
    Coord delta;
    delta = Coord(4, 4, 0);
    topLeft     = center - delta;
    bottomRight = center + delta;
    controlRects[i].setTopLeftPos(topLeft);
    controlRects[i].setTopLeftColor(hudColor2);
    controlRects[i].setBottomRightPos(bottomRight);
    controlRects[i].setBottomRightColor(hudColor2);
    // We deactivate Backface culling and Z-buffer
    controlRects[i].setRenderState(GlAD_ZEnable, false);
    controlRects[i].setRenderState(GlAD_Culling, false);
    // Then we activate Wireframe rendering
    controlRects[i].setRenderState(GlAD_Wireframe, false);
    controlRects[i].setRenderState(GlAD_Solid, true);
  }

  controlRects[3].setTopLeftColor(Color(255,0,0));
  controlRects[7].setTopLeftColor(Color(255,0,0));
  controlRects[1].setTopLeftColor(Color(0,0,255));
  controlRects[5].setTopLeftColor(Color(0,0,255));
  centerX = (centerRect.getPosition(0).getX() + centerRect.getPosition(2).getX()) / 2 ;
  centerY = (centerRect.getPosition(0).getY() + centerRect.getPosition(2).getY()) / 2 ;
}
//========================================================================================
