/*
 * tuXeyes.cpp - main program code of tuXeyes
 * Copyright (C) 1999 Ivo van der Wijk
 * 
 * The tux image is (c) Larry Ewing (lewing@isc.tamu.edu)
 * The dustpuppy image is (c) Illiad (http://www.userfriendly.org)
 * The bsd daemon ("Chuck") is (c) Eric Green
 * The luxus image is (c) Urs M. E. Streidl (http://www.gigaperls.org/linux/)
 * 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * To do: 
 *	  move when clicked on belly (or image in general)
 *	  small menu when right mouseclick on image (about/exit)
 *	  add mail, websurfing and Operating System capabilities (tux2000!)
 *	  inTUXicated?
 *	  cpu expensive: draw eyes on eyeregion, then scale?
 */
#include <iostream.h>

#include <qapplication.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qevent.h>

#include <X11/Xlib.h>

#include <math.h>
#include <stdlib.h>

#include <string.h>
#include "tuXeyes.h"

#include "tux.h"
#include "luxus.h"
#include "dp.h"
#include "bsd.h"

#undef DEBUG

#define VERSION	"0.0.3"

const int	tuxWidget::timerInterval = 40;

tuxWidget::tuxWidget(figure *fig, QWidget *parent, const char *name): 
	QWidget(parent, name, 0)
{
	/*
	 * I don't know any other way of getting rootwindow events or
	 * rootwindow mouse cursor positions other that the current way:
	 * request them using a timer and by directly asking X.
	 *
	 * This makes tuXeyes difficult to port to non-X systems :(
	 */
	display = XOpenDisplay(NULL);
	root = RootWindow(display, DefaultScreen(display));

	fig->postInit();		// last initialization step
	tuxImg = fig;

	setMask(tuxImg->getMainMask());

	origWidth = tuxImg->getWidth();
	origHeight = tuxImg->getHeight();

	// -geometry? Well, it works..
	resize(tuxImg->getWidth(),tuxImg->getHeight());
	repaint();

	oldX = -1; oldY = -1;
	leftX = tuxImg->getLeftEyeX(); leftY = tuxImg->getLeftEyeY();
	rightX = tuxImg->getRightEyeX(); rightY = tuxImg->getRightEyeY();

	// scaling
	xScale = 1.0;
	yScale = 1.0;
	avg = 1.0;

	// 25 updates per second are sufficient.
	startTimer(timerInterval);
}

void	tuxWidget::paintEyes()
{
	QPixmap		eyes = tuxImg->getEyeRegion();
	QPainter	p2(&eyes);
	QPainter	painter;
	QPixmap		leftPupil = tuxImg->getLeftPupil();	
	QPixmap		rightPupil = tuxImg->getRightPupil();	

	/*
	 * Draw the left and rigtht pupil into the eye pixmap
	 */
	p2.drawPixmap(
		(int)(leftX-(leftPupil.width()/2)-tuxImg->getEyeXOffset()*xScale), 
	        (int)(leftY-(leftPupil.height()/2)-tuxImg->getEyeYOffset()*yScale),
		      leftPupil);
	p2.drawPixmap(
		(int)(rightX-(rightPupil.width()/2)-tuxImg->getEyeXOffset()*xScale), 
	        (int)(rightY-(rightPupil.height()/2)-tuxImg->getEyeYOffset()*yScale),
		      rightPupil);
	/*
	 * Paint the eye pixmap (with pupils) on the tux pixmap
	 */
	painter.begin(this);
	painter.drawPixmap((int)(xScale*tuxImg->getEyeXOffset()), 
			   (int)(yScale*tuxImg->getEyeYOffset()), eyes);
	painter.end();
}

void	tuxWidget::paintEvent(QPaintEvent *)
{
	QPainter 	painter;
		
	/*
	 * Paint the entire tux pixmap
	 */
	painter.begin(this);
	painter.drawPixmap(0, 0, tuxImg->getMainImg());
	painter.end();
		
	/*
	 * Paint the pupils
	 */
	paintEyes();
}

void	tuxWidget::timerEvent(QTimerEvent *)
{
	int 		newX, newY, WinX, WinY;
	unsigned int 	StateMask;
	Window		dummy1, dummy2; // don't need these

	if(!isVisible())
		return;		// don't do anything if we're unmapped

	/*
	 * get the location of the mousepointer. Is there a
	 * qt way to do this, for example using QApplication::desktop()?
	 * Doesn't look like it :(
	 */

	XQueryPointer (display, root, &dummy1, &dummy2, &newX,
	    &newY, &WinX, &WinY, &StateMask);

	// did the mouse move?
	if(oldX == newX && oldY == newY)
		return;

	// store coordinates for later
	oldX = newX;
	oldY = newY;

	// find position of window to get absolute values
	int myX, myY;
	QPoint global = mapToGlobal(QPoint(0,0));
	myX = global.x();
	myY = global.y();

#ifdef DEBUG
	cout << "Avg = " << avg << endl;
#endif
	// get the X and Y differences and calculate the length
	// of the hypotenuse

	/*
	 * calc left eye. We have to scale leftEye[XY] to get
	 * the correct physical location of the eye if the image
	 * is resized.
	 */
	float diffX = (myX + tuxImg->getLeftEyeX()*xScale) - newX;
	float diffY = (myY + tuxImg->getLeftEyeY()*yScale) - newY;
	float hypotenuse = sqrt(diffX*diffX + diffY*diffY);

	// calc the relative size
	float ratio =  hypotenuse/(tuxImg->getLeftPupilRadius());
#if 0
	if(hypotenuse < tuxImg->getLeftPupilRadius()*avg)
	{
#ifdef DEBUG
		cout << "In range" << endl;
#endif
		leftX = tuxImg->getLeftEyeX() - diffX;
		leftY = tuxImg->getLeftEyeY() - diffY;
		leftX *= xScale;
		leftY *= yScale;
	}
	else
#endif
	{
		leftX = tuxImg->getLeftEyeX() - diffX/ratio;
		leftY = tuxImg->getLeftEyeY() - diffY/ratio;

		// properly rescale these values as the image size
		// may have changed
		leftX *= xScale;
		leftY *= yScale;
	}

	/*
	 * calc right eye. We have to scale rightEye[XY] to get
	 * the correct physical location of the eye if the image
	 * is resized.
	 */
	diffX = (myX + tuxImg->getRightEyeX()*xScale) - newX;
	diffY = (myY + tuxImg->getRightEyeY()*yScale) - newY;
	hypotenuse = sqrt(diffX*diffX + diffY*diffY);
	ratio = hypotenuse/(tuxImg->getRightPupilRadius());
		
#if 0
	if(hypotenuse < tuxImg->getRightPupilRadius()*avg)
	{
#ifdef DEBUG
		cout << "In range" << endl;
#endif
		rightX = tuxImg->getRightEyeX() - diffX;
		rightY = tuxImg->getRightEyeY() - diffY;

		// properly rescale these values as the image size
		// may have changed
		rightX *= xScale;
		rightY *= yScale;
	}
	else
#endif
	{
		rightX = tuxImg->getRightEyeX() - diffX/ratio;
		rightY = tuxImg->getRightEyeY() - diffY/ratio;
		rightX *= xScale;
		rightY *= yScale;
	}

#ifdef DEBUG
	cout << "leftX : " << leftX << " leftY " << leftY << endl;
	cout << "rightX : " << rightX << " rightY " << rightY << endl;
#endif

	paintEyes();
}

void	tuxWidget::resizeEvent(QResizeEvent *size)
{
#ifdef DEBUG
	cout << "Resize event: " << size->size().width() 
	     << "x" << size->size().height() << endl;
#endif

	int newWidth = size->size().width();
	int newHeight = size->size().height();

	/* 
	 * Always calculate the scaling based on the original image,
	 * not the previous image. This will compensate for loss that
	 * might occur after lots of resizing.
	 */
	xScale = (float)newWidth/(float)origWidth;
	yScale = (float)newHeight/(float)origHeight;

	// avg is to compensate the difference between xScale and yScale
	avg = (xScale+yScale)/2.0;


#ifdef DEBUG
	cout << "xScale = " << xScale << " yScale = " << yScale << endl;
#endif
	tuxImg->rescale(xScale, yScale);

	// set masks to new scaled mask
	setMask(tuxImg->getMainMask());

	repaint();
}

void	tuxWidget::keyPressEvent(QKeyEvent *e)
{
#ifdef DEBUG
	cout << "Key pressed!" << (char)e->ascii() << endl;
#endif
	if(e->ascii() == 'q' || e->ascii() == 'Q')
		QApplication::exit(0);
}

void	print_license()
{
	cout << "Copyright (c) 1999 Ivo van der Wijk

The tux image is (c) Larry Ewing (lewing@isc.tamu.edu)
The dustpuppy image is (c) Illiad (http://www.userfriendly.org)
The bsd daemon (\"Chuck\") is (c) Eric Green
The luxus image is (c) Urs M. E. Streidl (http://www.gigaperls.org/linux/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE." << endl;
	exit(0);
}

int main( int argc, char **argv )
{
    	QApplication a( argc, argv );
	int	_argc = a.argc();
	char	**_argv = a.argv();
	tuxWidget *w = NULL;
	int	result;

	cout << "tuXeyes version " << VERSION << " is (c) 1999 Ivo van der Wijk"
	     << " (tuxeyes@zero.xs4all.nl)" << endl
	     << "Check http://zero.xs4all.nl/~ivo/tuXeyes/ for more info" 
	     << " and updates." << endl
	     << "Type tuXeyes --help for options and other figures!" << endl;
	
	/*
	 * So you can multiple -tux -luxus parameters..
	 */
	for(int i = 1; i < _argc; i++)
	{
		if(!strcasecmp(_argv[i], "--license"))
			print_license();
		else if(!strcasecmp(_argv[i], "--tux"))
			w = new tuxWidget(new tux());
		else if(!strcasecmp(_argv[i], "--luxus"))
			w = new tuxWidget(new luxus());
		else if(!strcasecmp(_argv[i], "--puppy"))
			w = new tuxWidget(new dp());
		else if(!strcasecmp(_argv[i], "--bsd") ||
			!strcasecmp(_argv[i], "--chuck"))
			w = new tuxWidget(new bsd());
		else if(!strcasecmp(_argv[i], "--help") ||
		        !strcasecmp(_argv[i], "-h"))
		{
			cout << " " << endl;
			cout << "Available options:" << endl
			  << "-h, --help  This help" << endl
			  << "--license   Print tuXeyes' license and exit." << endl
			  << "--tux       Start with standard tux figure"<<endl
			  << "--luxus     Start with luxus figure" << endl
			  << "--chuck     Start with Chuck, the bsd daemon figure" << endl
			  << "--puppy     Start with dustpuppy figure" << endl
			  << endl
			  << "tuXeyes also accepts standard X options such"
			  << " as -geometry, -display, -name, -title, etc." 
			  << endl;
			exit(0);
		}
		else
		{
			cerr << " " << endl
			     << "Unknown parameter " << _argv[i] << endl;
			exit(-1);
		}
	}
	
	if(w == NULL)
		w = new tuxWidget(new tux());

	a.setMainWidget(w);
	w->show();
	result = a.exec();
	delete w;
	return result;
}
