function cholmod_make (metis_path)
%CHOLMOD_MAKE:  compiling the CHOLMOD mexFunctions
%
% CHOLMOD relies on AMD, COLAMD, CCOLAMD, and METIS for its
% ordering options.
%
% All but METIS are distributed with CHOLMOD.  To compile CHOLMOD
% to use METIS you must first place a copy of the metis-4.0
% directory (METIS version 4.0.1) in same directory that contains
% the AMD, COLAMD, CCOLAMD, and CHOLMOD directories.  Next, type
%
%   cholmod_make
%
% in the MATLAB command window.  Alternatively, use this command:
%
%   cholmod_make ('path to your copy of metis-4.0 here') ;
%
% See http://www-users.cs.umn.edu/~karypis/metis for a copy of
% METIS 4.0.1.  If you do not have METIS, use either of:
%
%   cholmod_make ('')
%   cholmod_make ('no metis')
%
% You must type the cholmod_make command while in the
% CHOLMOD/MATLAB directory.

help cholmod_make

if (ispc)

    %---------------------------------------------------------------------------
    % edit this definition, if necessary:
    lapack='libmwlapack.lib' ;
    %---------------------------------------------------------------------------

    fprintf ('Using LAPACK library:\n%s\n', lapack) ;
    fprintf ('If this does not work, edit cholmod_make.m and\n') ;
    fprintf ('modify the definition of lapack.\n') ;
else
    % For other systems, mex can find lapack on its own.
    lapack = '' ;
end

fprintf ('Now compiling CHOLMOD:\n') ;

include = '-I. -I../../AMD/Include -I../../AMD/Source -I../../COLAMD -I../../CCOLAMD -I../Include' ;

 % Determine the METIS path, and whether or not METIS is available
if (nargin == 0)
    metis_path = '../../metis-4.0' ;
end
if (strcmp (metis_path, 'no metis'))
    metis_path = '' ;
end
have_metis = (~isempty (metis_path)) ;

 % fix the METIS 4.0.1 rename.h file
if (have_metis)
    f = fopen ('rename.h', 'w') ;
    if (f == -1)
        error ('unable to create rename.h in current directory') ;
    end
    fprintf (f, '/* do not edit this file; generated by cholmod_make */\n') ;
    fprintf (f, '#undef log2\n') ;
    fprintf (f, '#include "%s/Lib/rename.h"\n', metis_path) ;
    fprintf (f, '#undef log2\n') ;
    fprintf (f, '#define log2 METIS__log2\n') ;
    fprintf (f, '#include "mex.h"\n') ;
    fprintf (f, '#define malloc mxMalloc\n') ;
    fprintf (f, '#define free mxFree\n') ;
    fprintf (f, '#define calloc mxCalloc\n') ;
    fprintf (f, '#define realloc mxRealloc\n') ;
    fclose (f) ;
    include = [include ' -I' metis_path '/Lib'] ;
else
    fprintf ('Compiling CHOLMOD without METIS\n') ;
    include = ['-DNPARTITION ' include] ;
end

include = strrep (include, '/', filesep) ;

amd_src = { ...
    '../../AMD/Source/amd_1', ...
    '../../AMD/Source/amd_2', ...
    '../../AMD/Source/amd_aat', ...
    '../../AMD/Source/amd_control', ...
    '../../AMD/Source/amd_defaults', ...
    '../../AMD/Source/amd_dump', ...
    '../../AMD/Source/amd_global', ...
    '../../AMD/Source/amd_info', ...
    '../../AMD/Source/amd_order', ...
    '../../AMD/Source/amd_postorder', ...
    '../../AMD/Source/amd_post_tree', ...
    '../../AMD/Source/amd_preprocess', ...
    '../../AMD/Source/amd_valid' } ;

colamd_src = {
    '../../COLAMD/colamd', ...
    '../../COLAMD/colamd_global' } ;

ccolamd_src = {
    '../../CCOLAMD/ccolamd', ...
    '../../CCOLAMD/ccolamd_global' } ;

metis_src = {
    'Lib/balance', ...
    'Lib/bucketsort', ...
    'Lib/ccgraph', ...
    'Lib/coarsen', ...
    'Lib/compress', ...
    'Lib/debug', ...
    'Lib/estmem', ...
    'Lib/fm', ...
    'Lib/fortran', ...
    'Lib/frename', ...
    'Lib/graph', ...
    'Lib/initpart', ...
    'Lib/kmetis', ...
    'Lib/kvmetis', ...
    'Lib/kwayfm', ...
    'Lib/kwayrefine', ...
    'Lib/kwayvolfm', ...
    'Lib/kwayvolrefine', ...
    'Lib/match', ...
    'Lib/mbalance2', ...
    'Lib/mbalance', ...
    'Lib/mcoarsen', ...
    'Lib/memory', ...
    'Lib/mesh', ...
    'Lib/meshpart', ...
    'Lib/mfm2', ...
    'Lib/mfm', ...
    'Lib/mincover', ...
    'Lib/minitpart2', ...
    'Lib/minitpart', ...
    'Lib/mkmetis', ...
    'Lib/mkwayfmh', ...
    'Lib/mkwayrefine', ...
    'Lib/mmatch', ...
    'Lib/mmd', ...
    'Lib/mpmetis', ...
    'Lib/mrefine2', ...
    'Lib/mrefine', ...
    'Lib/mutil', ...
    'Lib/myqsort', ...
    'Lib/ometis', ...
    'Lib/parmetis', ...
    'Lib/pmetis', ...
    'Lib/pqueue', ...
    'Lib/refine', ...
    'Lib/separator', ...
    'Lib/sfm', ...
    'Lib/srefine', ...
    'Lib/stat', ...
    'Lib/subdomains', ...
    'Lib/timing', ...
    'Lib/util' } ;

for i = 1:length (metis_src)
    metis_src {i} = [metis_path '/' metis_src{i}] ;
end

cholmod_matlab = { 'cholmod_matlab' } ;

cholmod_src = {
    '../Core/cholmod_aat', ...
    '../Core/cholmod_add', ...
    '../Core/cholmod_band', ...
    '../Core/cholmod_change_factor', ...
    '../Core/cholmod_common', ...
    '../Core/cholmod_complex', ...
    '../Core/cholmod_copy', ...
    '../Core/cholmod_dense', ...
    '../Core/cholmod_error', ...
    '../Core/cholmod_factor', ...
    '../Core/cholmod_memory', ...
    '../Core/cholmod_sparse', ...
    '../Core/cholmod_transpose', ...
    '../Core/cholmod_triplet', ...
    '../Check/cholmod_check', ...
    '../Check/cholmod_read', ...
    '../Cholesky/cholmod_amd', ...
    '../Cholesky/cholmod_analyze', ...
    '../Cholesky/cholmod_colamd', ...
    '../Cholesky/cholmod_etree', ...
    '../Cholesky/cholmod_factorize', ...
    '../Cholesky/cholmod_postorder', ...
    '../Cholesky/cholmod_rcond', ...
    '../Cholesky/cholmod_resymbol', ...
    '../Cholesky/cholmod_rowcolcounts', ...
    '../Cholesky/cholmod_rowfac', ...
    '../Cholesky/cholmod_solve', ...
    '../Cholesky/cholmod_spsolve', ...
    '../MatrixOps/cholmod_drop', ...
    '../MatrixOps/cholmod_horzcat', ...
    '../MatrixOps/cholmod_norm', ...
    '../MatrixOps/cholmod_scale', ...
    '../MatrixOps/cholmod_sdmult', ...
    '../MatrixOps/cholmod_ssmult', ...
    '../MatrixOps/cholmod_submatrix', ...
    '../MatrixOps/cholmod_vertcat', ...
    '../Modify/cholmod_rowadd', ...
    '../Modify/cholmod_rowdel', ...
    '../Modify/cholmod_updown', ...
    '../Supernodal/cholmod_super_numeric', ...
    '../Supernodal/cholmod_super_solve', ...
    '../Supernodal/cholmod_super_symbolic', ...
    '../Partition/cholmod_ccolamd', ...
    '../Partition/cholmod_csymamd', ...
    '../Partition/cholmod_metis', ...
    '../Partition/cholmod_nesdis' } ;

cholmod_mex_src = { ...
    'analyze', ...
    'bisect', ...
    'chol2', ...
    'cholmod', ...
    'etree2', ...
    'lchol', ...
    'ldlchol', ...
    'ldlsolve', ...
    'ldlupdate', ...
    'metis', ...
    'mread', ...
    'nesdis', ...
    'resymbol', ...
    'sparse2', ...
    'symbfact2' } ;

if (ispc)
    % Windows does not have drand48 and srand48, required by METIS.  Use
    % drand48 and srand48 in CHOLMOD/MATLAB/Windows/rand48.c instead.
    obj_extension = '.obj' ;
    cholmod_matlab = [cholmod_matlab {'Windows/rand48'}] ;
    include = [include ' -IWindows'] ;
else
    obj_extension = '.o' ;
end

 % compile each library source file
obj = '' ;
k = 0 ;

source = [amd_src colamd_src ccolamd_src cholmod_src cholmod_matlab] ;
if (have_metis)
    source = [metis_src source] ;
end

for f = source
    f = strrep (f {1}, '/', filesep) ;
    slash = strfind (f, filesep) ;
    if (isempty (slash))
        slash = 1 ;
    else
        slash = slash (end) + 1 ;
    end
    o = f (slash:end) ;
    obj = [obj  ' ' o obj_extension] ;
    s = sprintf ('mex -O %s -c %s.c', include, f);
    fprintf ('.') ;
    % fprintf (' %s\n', s) ;
    k = k + 1 ;
    if (mod (k,50) == 0)
        fprintf ('\n') ;
    end
    eval (s) ;
end

 % compile each mexFunction
for f = cholmod_mex_src
    s = sprintf ('mex -O %s %s.c', include, f{1}) ;
    s = [s obj ' ' lapack] ;
    fprintf ('.') ;
    % fprintf (' %s\n', s) ;
    k = k + 1 ;
    if (mod (k,50) == 0)
        fprintf ('\n') ;
    end
    eval (s) ;
end

 % clean up
eval (['delete ' obj]) ;

fprintf ('\nNow compiling the AMD, CCOLAMD, and COLAMD mexFunctions:\n') ;
cholmod_path = pwd ;
addpath (cholmod_path)
cd ../../AMD/MATLAB
amd_make
amd_path = pwd ;
addpath (amd_path)
cd ../../COLAMD
colamd_make
colamd_path = pwd ;
addpath (colamd_path)
cd ../CCOLAMD
ccolamd_make
ccolamd_path = pwd ;
addpath (ccolamd_path)
cd (cholmod_path)

fprintf ('\nThe following paths have been added.  You may wish to add them\n') ;
fprintf ('permanently, using the MATLAB pathtool command.\n') ;
fprintf ('%s\n', cholmod_path) ;
fprintf ('%s\n', amd_path) ;
fprintf ('%s\n', colamd_path) ;
fprintf ('%s\n', ccolamd_path) ;

fprintf ('\nTo try your new mexFunctions, cut-and-paste this command:\n') ;
fprintf ('amd_demo, colamd_demo, ccolamd_demo, graph_demo, cholmod_demo\n') ;
