/*!****************************************************************************
*
*    exrap.c
*    ========
*
*    TODO Info to write:
*    -   ONC RPC, rpcgen etc.
*    -   EX makefile.EX: (adjust before make to include RAP)
*    -   c-pointer must be same size or smaller than DBint
*    -   ref to ABB documentation
*    -   ref to rpc documentation
*
*    EXrpcclntcr();       Create client for Remote Procedure calls (RPC)
*    EXrpcclntde();       Destroy RPC client
*    EXrapoutfloat();     Write varkon float to rapid 'num' variable
*    EXrapinfloat();      Read rapid 'num' variable to varkon float
*    EXrapoutint();       Write varkon int to rapid 'num' variable
*    EXrapinint();        Read rapid 'num' variable to varkon int
*    EXrapoutrobtarget(); Write varkon data to rapid 'robtarget' structure
*    EXrapinrobtarget();  Read rapid 'robtarget' structure to varkon data 
*    EXrapoutspeed();     Write varkon data to rapid 'speeddata' structure
*    EXrapinspeed();      Read rapid 'speeddata' structure to varkon data 
*    EXrapoutrobtarr();   Write varkon data to array of rapid 'robtarget'
*    EXrapinrobtarr();    Read rapid 'robtarget' array to varkon data
*    EXrapoutfloatarr();  Write varkon float array to rapid 'num' array
*    EXrapinfloatarr();   Read rapid 'num' array to varkon float array
*
*    This file is part of the VARKON Execute  Library.
*
*    This library is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    This library is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    (C) 2004-04-26 Sren Larsson, rebro University                             
*
*******************************************************************************/

#include "../../DB/include/DB.h"
#include "../../IG/include/IG.h"
#include "../include/EX.h"

extern short erpush();

#ifdef V3RAP

#include "rap.h"   /* rap.h is generated by rpcgen from rap.x */

/* Transfer type numbers for some RAPID variables */
#define NUM_TRANSFER_TYPE       1
#define STRING_TRANSFER_TYPE    3
#define BOOL_TRANSFER_TYPE      4
#define CONFDATA_TRANSFER_TYPE  7
#define EXTJOINT_TRANSFER_TYPE  8
#define ORIENT_TRANSFER_TYPE    11
#define POS_TRANSFER_TYPE       12
#define POSE_TRANSFER_TYPE      13
#define ROBTARGET_TRANSFER_TYPE 15
#define SPEEDDATA_TRANSFER_TYPE 16
#define TOOLDATA_TRANSFER_TYPE  17
#define WOBJDATA_TRANSFER_TYPE  18
                
#endif





/*!****************************************************************************/

        short  EXrpcclntcr(
        char   *server,
        DBint   program,
        DBint   version,
        DBint  *Pclient,
        DBint  *pstat)
 
/*      Create client for Remote Procedure calls (RPC)
 *
 *      In:  server     = Name or ip-adress of RPC server
             program    = Program number 
             version    = Program version 
 *
 *      Out: *Pclient   = Handle to created client (DBint)
 *           *pstat     = Status 
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP
CLIENT *clnt = NULL;

clnt = clnt_create (server, program, version, "tcp");
if (clnt == NULL) {
   *pstat=-1;
   return(0);
}
*Pclient=(DBint)(void*)clnt;  /* asume size of pointer == size of DBint) */
*pstat = 0;
return(0);
  
#else
return(erpush("EX6012",""));
#endif
}



/*!****************************************************************************/

        short  EXrpcclntde(
        DBint  Pclient,
        DBint  *pstat)

/*      Destroy RPC client
 *
 *      In:   Pclient   =  Handle to RPC client (DBint)
 *      Out: *pstat     =  Status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP
CLIENT *clnt;
clnt=(CLIENT*)(void*)Pclient;
clnt_destroy (clnt);
*pstat = 0;
return(0);

#else
return(erpush("EX6012",""));  
#endif
}



/*!****************************************************************************/

        short   EXrapoutfloat(
        DBint   Pclient,
        char    *name,
        DBfloat data,
        DBint   *pstat)

/*      Write varkon float to rapid 'num' variable
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            data      =  Float data to write
 *
 *      Out: *pstat     =  Status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_WRITE_NAME_RESP_TYPE *result;
RAPVAR_WRITE_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="num";
request.var.domain=1;                    /* see RAP service specifacation, page 43 */
request.var.number1=0;                   /*              - " -                     */
request.var.number2=NUM_TRANSFER_TYPE;   /*              - " -                     */   
request.data.var_type=NUM_TRANSFER_TYPE;        
request.data.RAPVAR_DATA_TYPE_u.num=(RAPVAR_NUM_TYPE)data;

result = rapvar_write_name_1(&request, clnt);
if (result == (RAPVAR_WRITE_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;  /*status >= 0 is ok */
xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));  
#endif
}


/*!****************************************************************************/

        short   EXrapinfloat(
        DBint   Pclient,
        char    *name,
        DBfloat *data,
        DBint   *pstat)

/*      Read rapid 'num' variable to varkon float
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *
 *      Out:  *data      =  Float data to read
 *            *pstat     =  Status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/

{
#ifdef V3RAP
CLIENT *clnt;
RAPVAR_READ_NAME_RESP_TYPE *result;
RAPVAR_READ_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="num";
request.var.domain=1;
request.var.number1=0;
request.var.number2=NUM_TRANSFER_TYPE;          

result = rapvar_read_name_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}

*pstat = result->resp.status;  /*status >= 0 is ok */
if (*pstat >= 0){
*data =(DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.num;
}
xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012","")); 
#endif
}




/*!****************************************************************************/

        short   EXrapoutint(
        DBint   Pclient,
        char    *name,
        DBint   data,
        DBint   *pstat)

/*      Write varkon int to rapid 'num' variable
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            data      =  Integer data to write
 *
 *      Out: *pstat     =  Status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP


CLIENT *clnt;
RAPVAR_WRITE_NAME_RESP_TYPE *result;
RAPVAR_WRITE_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="num";
request.var.domain=1;
request.var.number1=0;
request.var.number2=NUM_TRANSFER_TYPE;          
request.data.var_type=NUM_TRANSFER_TYPE;        
request.data.RAPVAR_DATA_TYPE_u.num=(RAPVAR_NUM_TYPE)data;

result = rapvar_write_name_1(&request, clnt);
if (result == (RAPVAR_WRITE_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;  /*status >= 0 is ok */
xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));  
#endif
}


/*!****************************************************************************/

        short   EXrapinint(
        DBint   Pclient,
        char    *name,
        DBint   *data,
        DBint   *pstat)

/*      Read rapid 'num' variable to varkon int
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *
 *      Out:  *data      =  Float data to read
 *            *pstat     =  Status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_READ_NAME_RESP_TYPE *result;
RAPVAR_READ_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="num";
request.var.domain=1;
request.var.number1=0;
request.var.number2=NUM_TRANSFER_TYPE;          

result = rapvar_read_name_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->resp.status;  /*status >= 0 is ok */
if (*pstat >= 0){
*data =(DBint) result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.num;
}
xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));  
#endif
}


/*!****************************************************************************/

        short    EXrapoutrobtarget(
        DBint    Pclient,
        char     *name,
        DBVector *trans,
        DBfloat  *rot,   
        DBint    *conf,  
        DBfloat  *extax, 
        DBint    *pstat)

/*      Write varkon data to rapid 'robtarget' structure
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            trans     =  translation
 *            rot[4]    =  orientation q1,q2,q3,q4 (quaternions)
 *            conf[4]   =  robot configuration cf1, cf4, cf6, cfx
 *            extax[6]  =  external axes
 *
 *      Out:  *pstat    =  status
 *
 *      (C) 2004-04-26 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_WRITE_NAME_RESP_TYPE *result;
RAPVAR_WRITE_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="robtarget";
request.var.domain=1;
request.var.number1=0;
request.var.number2=ROBTARGET_TRANSFER_TYPE;          

request.data.var_type=ROBTARGET_TRANSFER_TYPE;        
request.data.RAPVAR_DATA_TYPE_u.robtarget.trans.x=(RAPVAR_NUM_TYPE) trans->x_gm ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.trans.y=(RAPVAR_NUM_TYPE) trans->y_gm ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.trans.z=(RAPVAR_NUM_TYPE) trans->z_gm ;

request.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q1=(RAPVAR_NUM_TYPE) rot[0] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q2=(RAPVAR_NUM_TYPE) rot[1] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q3=(RAPVAR_NUM_TYPE) rot[2] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q4=(RAPVAR_NUM_TYPE) rot[3] ;

request.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf1=(RAPVAR_NUM_TYPE) conf[0] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf4=(RAPVAR_NUM_TYPE) conf[1] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf6=(RAPVAR_NUM_TYPE) conf[2] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cfx=(RAPVAR_NUM_TYPE) conf[3] ;

request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_a=(RAPVAR_NUM_TYPE) extax[0] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_b=(RAPVAR_NUM_TYPE) extax[1] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_c=(RAPVAR_NUM_TYPE) extax[2] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_d=(RAPVAR_NUM_TYPE) extax[3] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_e=(RAPVAR_NUM_TYPE) extax[4] ;
request.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_f=(RAPVAR_NUM_TYPE) extax[5] ;

result = rapvar_write_name_1(&request, clnt);

if (result == (RAPVAR_WRITE_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;

xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));  
#endif
}



/*!****************************************************************************/

        short    EXrapinrobtarget(
        DBint    Pclient,
        char     *name,
        DBVector *trans,
        DBfloat  *rot,  
        DBint    *conf,  
        DBfloat  *extax, 
        DBint    *pstat)

/*      Read rapid 'robtarget' structure to varkon data  
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *
 *      Out:  *trans    =  translation
 *            *rot      =  [4] orientation q1,q2,q3,q4 (quaternions)
 *            *conf     =  [4] robot configuration (cf1, cf4, cf6, cfx)
 *            *extax    =  [6] external axes (eax_a, ... eax_f)
 *            *pstat    =  status
 *
 *      (C) 2004-04-27 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_READ_NAME_RESP_TYPE *result;
RAPVAR_READ_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="robtarget";
request.var.domain=1;
request.var.number1=0;
request.var.number2=ROBTARGET_TRANSFER_TYPE;          

result = rapvar_read_name_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->resp.status;  /*status >= 0 is ok */

if (*pstat >= 0){

trans->x_gm = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.trans.x;
trans->y_gm = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.trans.y;
trans->z_gm = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.trans.z;

rot[0] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q1;
rot[1] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q2;
rot[2] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q3;
rot[3] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.rot.q4;

conf[0] = (DBint)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf1;
conf[1] = (DBint)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf4;
conf[2] = (DBint)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cf6;
conf[3] = (DBint)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.robconf.cfx;

extax[0] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_a;
extax[1] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_b;
extax[2] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_c;
extax[3] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_d;
extax[4] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_e;
extax[5] = (DBfloat)result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.robtarget.extax.eax_f;
}

xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_RESP_TYPE, (char*)result);  
return(0);
#else
return(erpush("EX6012",""));  
#endif
}


/*!****************************************************************************/

        short   EXrapoutspeed(
        DBint   Pclient,
        char    *name,
        DBfloat *speeddata,
        DBint   *pstat)

/*      Write varkon data to rapid 'speeddata' structure
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            speeddata =  [4]Data to write v_tcp, v_ori, v_leax, v_reax
 *
 *      Out: *pstat     =  Status
 *
 *      (C) 2004-04-27 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_WRITE_NAME_RESP_TYPE *result;
RAPVAR_WRITE_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="speeddata";
request.var.domain=1;
request.var.number1=0;
request.var.number2=SPEEDDATA_TRANSFER_TYPE;          
request.data.var_type=SPEEDDATA_TRANSFER_TYPE;        

request.data.RAPVAR_DATA_TYPE_u.speeddata.v_tcp =(RAPVAR_NUM_TYPE) speeddata[0] ;
request.data.RAPVAR_DATA_TYPE_u.speeddata.v_ori =(RAPVAR_NUM_TYPE) speeddata[1] ;
request.data.RAPVAR_DATA_TYPE_u.speeddata.v_leax=(RAPVAR_NUM_TYPE) speeddata[2] ;
request.data.RAPVAR_DATA_TYPE_u.speeddata.v_reax=(RAPVAR_NUM_TYPE) speeddata[3] ;

result = rapvar_write_name_1(&request, clnt);
if (result == (RAPVAR_WRITE_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;  /*status >= 0 is ok */
xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_RESP_TYPE, (char*)result);
return(0);

#else
return(erpush("EX6012",""));  
#endif
}


/*!****************************************************************************/

        short   EXrapinspeed(
        DBint   Pclient,
        char    *name,
        DBfloat *speeddata,
        DBint   *pstat)

/*      Read rapid 'speeddata' structure to varkon data
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *
 *      Out:  *speeddata =  Data to read: v_tcp, v_ori, v_leax, v_reax
 *            *pstat     =  Status
 *
 *      (C) 2004-04-27 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_READ_NAME_RESP_TYPE *result;
RAPVAR_READ_NAME_REQ_TYPE  request;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="speeddata";
request.var.domain=1;
request.var.number1=0;
request.var.number2=SPEEDDATA_TRANSFER_TYPE;          

result = rapvar_read_name_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->resp.status;  /*status >= 0 is ok */
if (*pstat >= 0){
speeddata[0] =(DBfloat) result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.speeddata.v_tcp;
speeddata[1] =(DBfloat) result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.speeddata.v_ori;
speeddata[2] =(DBfloat) result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.speeddata.v_leax;
speeddata[3] =(DBfloat) result->resp.RAPVAR_READ_NAME_RESPX_TYPE_u.data.RAPVAR_DATA_TYPE_u.speeddata.v_reax;
}
xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012","")); 
#endif
}



/*!****************************************************************************/

        short    EXrapoutrobtarr(
        DBint    Pclient,
        char     *name,
        DBint    first,
        DBint    last,
        DBVector *trans,
        DBfloat  *rot,
        DBint    *conf,
        DBfloat  *extax,
        DBint    *pstat)

/*      Write varkon data to array of rapid 'robtarget'
 *
 *      In:   Pclient        =  Handle to client (DBint)
 *            name           =  Variable name
 *            first          =  first element in robtarget array to write
 *            last           =  last element in robtarget array to write
 *            trans[size]    =  translation
 *            rot[size,4]    =  orientation q1,q2,q3,q4 (quaternions)
 *            conf[size,4]   =  robot configuration cf1, cf4, cf6, cfx
 *            extax[size,6]  =  external axes
 *
 *      Out:  *pstat    =  status
 *
 *      (C) 2004-05-05 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE *result;
RAPVAR_WRITE_NAME_ARRAY_REQ_TYPE  request;
DBint asize;
int i;

asize=last-first+1;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;

request.var.name=name;
request.var.type="robtarget";
request.var.domain=0;
request.var.number1=0;
request.var.number2=0;

request.dim.dimension=1;
request.dim.elements[0]=first;
request.dim.elements[1]=last;
request.dim.elements[2]=0;
request.dim.elements[3]=0;
request.dim.elements[4]=0;
request.dim.elements[5]=0;

request.data.RAPVAR_ARRAY_DATA_TYPE_len=asize;

/*
***Allocate memory.
*/
request.data.RAPVAR_ARRAY_DATA_TYPE_val=v3mall(asize*sizeof(RAPVAR_XDATA_TYPE),"EXrapoutrobtarr"); 
for ( i=0; i< asize; ++i )
  {
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val=v3mall(17*sizeof(RAP_FLOAT_TYPE),"EXrapoutrobtarr"); 
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val=v3mall(17*sizeof(RAP_FLOAT_TYPE),"EXrapoutrobtarr"); 
  }
if (request.data.RAPVAR_ARRAY_DATA_TYPE_val[asize-1].numdata.numdata_val==NULL)
  {
  return(erpush("EX6032",""));
  }

/*
***Copy values to request structure
*/
    
for ( i=0; i< asize; ++i )
  {                    
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_len=17;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_len=17;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].strdata.strdata_len=0;

  /* trans */  
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[0]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[0]=trans[i].x_gm;

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[1]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[1]=trans[i].y_gm;

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[2]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[2]=trans[i].z_gm;

  /* rot */
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[3]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[3]=rot[i*4+0];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[4]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[4]=rot[i*4+1];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[5]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[5]=rot[i*4+2];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[6]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[6]=rot[i*4+3];  

  /* conf */
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[7]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[7]=conf[i*4+0];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[8]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[8]=conf[i*4+1];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[9]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[9]=conf[i*4+2];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[10]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[10]=conf[i*4+3];
 
  /* extax */
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[11]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[11]=extax[i*6+0];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[12]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[12]=extax[i*6+1];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[13]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[13]=extax[i*6+2];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[14]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[14]=extax[i*6+3];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[15]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[15]=extax[i*6+2];

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[16]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[16]=extax[i*6+3];
  }

result = rapvar_write_name_array_1(&request, clnt);



/*
***Deallocate memory.
*/
for ( i=0; i< asize; ++i )
{
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val,"EXrapoutrobtarr");
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val,"EXrapoutrobtarr");
}
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val,"EXrapoutrobtarr");


if (result == (RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;

xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));
#endif
}

    
/*!****************************************************************************/

        short    EXrapinrobtarr(
        DBint    Pclient,
        char     *name,
        DBint    first,
        DBint    last,        
        DBVector *trans,
        DBfloat  *rot,
        DBint    *conf,
        DBfloat  *extax,
        DBint    *pstat)

/*      Read rapid 1-dimensional array of 'robtarget' to varkon data
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            first     =  first element in robtarget array to read
 *            last      =  last element in robtarget array to read
 *
 *      Out:  *trans    =  [asize]translation
 *            *rot      =  [(last-first+1)*4] orientation q1,q2,q3,q4 (quaternions)
 *            *conf     =  [(last-first+1)*4] robot configuration (cf1, cf4, cf6, cfx)
 *            *extax    =  [(last-first+1)*6] external axes (eax_a, ... eax_f)
 *            *pstat    =  status
 *
 *      (C) 2004-04-27 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_READ_NAME_ARRAY_RESP_TYPE *result;
RAPVAR_READ_NAME_ARRAY_REQ_TYPE  request;
DBint asize;
int i;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;
request.var.name=name;
request.var.type="robtarget";
request.var.domain=0;          /* see RAP service specifacation, page 44 */
request.var.number1=0;         /*                -   "   -               */
request.var.number2=0;         /*                -   "   -               */
request.dim.dimension=1;
request.dim.elements[0]=first;
request.dim.elements[1]=last;
request.dim.elements[2]=0;
request.dim.elements[3]=0;
request.dim.elements[4]=0;
request.dim.elements[5]=0;

result = rapvar_read_name_array_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_ARRAY_RESP_TYPE *) NULL)
  {
  *pstat = -1;
  return(0);
  }
*pstat = result->resp.status;  /*status >= 0 is ok */

if (*pstat >= 0)
  {
  asize=result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.RAPVAR_ARRAY_DATA_TYPE_len;

  if (asize != (last-first+1))
    {
    xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_ARRAY_RESP_TYPE, (char*)result);
    return(erpush("EX6022",""));
    }
    
  for ( i=0; i< asize; ++i )
    {
    trans[i].x_gm = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[0];
    trans[i].y_gm = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[1];
    trans[i].z_gm = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[2];

    rot[i*4+0] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[3];
    rot[i*4+1] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[4];
    rot[i*4+2] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[5];
    rot[i*4+3] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[6];

    conf[i*4+0] = (DBint)result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[7];
    conf[i*4+1] = (DBint)result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[8];
    conf[i*4+2] = (DBint)result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[9];
    conf[i*4+3] = (DBint)result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[10];

    extax[i*6+0] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[11];
    extax[i*6+1] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[12];
    extax[i*6+2] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[13];
    extax[i*6+3] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[14];
    extax[i*6+4] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[15];
    extax[i*6+5] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[16];
    }
  }
 
xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_ARRAY_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));
#endif
}
              




/*!****************************************************************************/

        short    EXrapoutfloatarr(
        DBint    Pclient,
        char     *name,
        DBint    first,
        DBint    last,
        DBfloat  *data,
        DBint    *pstat)

/*      Write varkon data to array of rapid 'robtarget'
 *
 *      In:   Pclient        =  Handle to client (DBint)
 *            name           =  Variable name
 *            first          =  first element in robtarget array to write
 *            last           =  last element in robtarget array to write
 *            data[size]     =  float data
 *
 *      Out:  *pstat    =  status
 *
 *      (C) 2004-05-13 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE *result;
RAPVAR_WRITE_NAME_ARRAY_REQ_TYPE  request;
DBint asize;
int i;

asize=last-first+1;

clnt=(CLIENT*)(void*)Pclient;

request.head.userdef=1;

request.var.name=name;
request.var.type="num";
request.var.domain=0;
request.var.number1=0;
request.var.number2=0;

request.dim.dimension=1;
request.dim.elements[0]=first;
request.dim.elements[1]=last;
request.dim.elements[2]=0;
request.dim.elements[3]=0;
request.dim.elements[4]=0;
request.dim.elements[5]=0;

request.data.RAPVAR_ARRAY_DATA_TYPE_len=asize;

/*
***Allocate memory.
*/
request.data.RAPVAR_ARRAY_DATA_TYPE_val=v3mall(asize*sizeof(RAPVAR_XDATA_TYPE),"EXrapoutfloatarr");
for ( i=0; i< asize; ++i )
  {
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val=v3mall(sizeof(RAP_FLOAT_TYPE),"EXrapoutfloatarr");
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val=v3mall(sizeof(RAP_FLOAT_TYPE),"EXrapoutfloatarr");
  }
if (request.data.RAPVAR_ARRAY_DATA_TYPE_val[asize-1].numdata.numdata_val==NULL)
  {
  return(erpush("EX6032",""));
  }

/*
***Copy values to request structure
*/

for ( i=0; i< asize; ++i )
  {
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_len=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_len=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].strdata.strdata_len=0;

  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val[0]=1;
  request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[0]=data[i];
  }

result = rapvar_write_name_array_1(&request, clnt);




/*
***Deallocate memory.
*/
for ( i=0; i< asize; ++i )
{
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].datacfg.datacfg_val,"EXrapoutfloatarr");
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val,"EXrapoutfloatarr");
}
v3free(request.data.RAPVAR_ARRAY_DATA_TYPE_val,"EXrapoutfloatarr");


if (result == (RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE *) NULL) {
   *pstat = -1;
   return(0);
}
*pstat = result->status;

xdr_free((xdrproc_t)xdr_RAPVAR_WRITE_NAME_ARRAY_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));
#endif
}


/*!****************************************************************************/

        short    EXrapinfloatarr(
        DBint    Pclient,
        char     *name,
        DBint    first,
        DBint    last,
        DBfloat  *data,
        DBint    *pstat)

/*      Read rapid array of 'num' to varkon float array
 *
 *      In:   Pclient   =  Handle to client (DBint)
 *            name      =  Variable name
 *            first     =  first element in robtarget array to read
 *            last      =  last element in robtarget array to read
 *
 *      Out:  *data     =  [asize] float array
 *            *pstat    =  status
 *
 *      (C) 2004-05-13 Sren Larsson, rebro University
 *
 ****************************************************************************!*/
{
#ifdef V3RAP

CLIENT *clnt;
RAPVAR_READ_NAME_ARRAY_RESP_TYPE *result;
RAPVAR_READ_NAME_ARRAY_REQ_TYPE  request;
DBint asize;
int i;

clnt=(CLIENT*)(void*)Pclient;


request.head.userdef=1;
request.var.name=name;
request.var.type="num";
request.var.domain=0;          /* see RAP service specifacation, page 44 */
request.var.number1=0;         /*                -   "   -               */
request.var.number2=0;         /*                -   "   -               */
request.dim.dimension=1;
request.dim.elements[0]=first;
request.dim.elements[1]=last;
request.dim.elements[2]=0;
request.dim.elements[3]=0;
request.dim.elements[4]=0;
request.dim.elements[5]=0;

result = rapvar_read_name_array_1(&request, clnt);

if (result == (RAPVAR_READ_NAME_ARRAY_RESP_TYPE *) NULL)
  {
  *pstat = -1;
  return(0);
  }
*pstat = result->resp.status;  /*status >= 0 is ok */

if (*pstat >= 0)
  {
  asize=result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.RAPVAR_ARRAY_DATA_TYPE_len;

  if (asize != (last-first+1))
    {
    xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_ARRAY_RESP_TYPE, (char*)result);
    return(erpush("EX6022",""));
    }

  for ( i=0; i< asize; ++i )
    {

    data[i] = result->resp.RAPVAR_READ_NAME_ARRAY_RESPX_TYPE_u.data.
                    RAPVAR_ARRAY_DATA_TYPE_val[i].numdata.numdata_val[0];
    }
  }

xdr_free((xdrproc_t)xdr_RAPVAR_READ_NAME_ARRAY_RESP_TYPE, (char*)result);
return(0);
#else
return(erpush("EX6012",""));
#endif
}






              

