#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <gtk/gtk.h>

#include "../include/strexp.h"
#include "../include/string.h"

#include "v3dmp.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fprompt.h"

#include "editor.h"

#include "scratchpad.h"
#include "scratchpadcb.h"
#include "scratchpaddnd.h"

#include "vmacfg.h"
#include "vmacfglist.h"
#include "vma.h"
#include "config.h"


static void ScratchPadRecordPositions(vma_scratch_pad_struct *sp);

void ScratchPadRowDeleteCB(gpointer data);
void ScratchPadTargetComboTextInsertCB(
	GtkEditable *editable,
	const gchar *text, gint length, gint *position,
	gpointer data
);
void ScratchPadTargetComboButtonEnterCB(GtkButton *button, gpointer data);

gint ScratchPadEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint ScratchPadMenuMapCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

static void ScratchPadFPromptApplyCB(gpointer data, const gchar *value);
static void ScratchPadFPromptCancelCB(gpointer data);
static void ScratchPadFPromptMap(
	vma_scratch_pad_struct *sp, gint column, gint row
);

void ScratchPadDestroyCB(GtkObject *object, gpointer data);
gint ScratchPadCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

void ScratchPadCloseButtonCB(
	GtkWidget *widget, gpointer data
);

gint ScratchPadMenuItemEnterCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

void ScratchPadInsertCB(GtkWidget *widget, gpointer data);
void ScratchPadEditCB(GtkWidget *widget, gpointer data);
void ScratchPadDeleteCB(GtkWidget *widget, gpointer data);

void ScratchPadSetVertexCB(GtkWidget *widget, gpointer data);
void ScratchPadSetNormalCB(GtkWidget *widget, gpointer data);
void ScratchPadSetTexCoordCB(GtkWidget *widget, gpointer data);
void ScratchPadSetAllCB(GtkWidget *widget, gpointer data);

void ScratchPadCListSelectCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void ScratchPadCListUnselectCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,  
	gpointer data            
);

void ScratchPadTextChangeCB(
	GtkEditable *editable, gpointer data  
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Records positions to the global configuration options list with respect
 *	to the given scratch pad.
 */
static void ScratchPadRecordPositions(vma_scratch_pad_struct *sp)
{
	GtkWidget *w;
	vma_cfg_item_struct *opt = option;
	u_int32_t ui32;


	/* Check if we are suppose to record positions and sizes. */
	if(!VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_RECORD_LAST_POS_AND_SIZE
	))
	    return;

	if(sp == NULL)
	    return;

	if(!sp->initialized)
	    return;

	/* Record size of toplevel window. */
	w = sp->toplevel;
	if(w != NULL)  
	{
	    ui32 = w->allocation.width; 
	    VMACFGItemListMatchSetValue(
		opt, VMA_CFG_PARM_SCRATCHPAD_WIDTH,
		&ui32, 0
	    );
	    ui32 = w->allocation.height;
	    VMACFGItemListMatchSetValue(
		opt, VMA_CFG_PARM_SCRATCHPAD_HEIGHT,
		&ui32, 0
	    );
	}

	/* CList's hbox height. */
	w = sp->clist_toplevel;
	if(w != NULL)
	{
	    ui32 = w->allocation.height;
	    VMACFGItemListMatchSetValue(
		opt, VMA_CFG_PARM_SCRATCHPAD_CLIST_HEIGHT,
		&ui32, 0
	    );
	}

	/* Column widths. */
	w = sp->clist;
	if(w != NULL)
	{
	    gint i;
	    GtkCList *clist = GTK_CLIST(w);
	    GtkCListColumn *column_heading;

	    for(i = 0; i < clist->columns; i++)
	    {
		column_heading = &clist->column[i];
		switch(i)
		{
		  case 0:
		    ui32 = column_heading->width;
		    VMACFGItemListMatchSetValue(
			opt, VMA_CFG_PARM_SCRATCHPAD_CH0,
			&ui32, 0
		    );
		    break;

		  case 1:
		    ui32 = column_heading->width;
		    VMACFGItemListMatchSetValue(
			opt, VMA_CFG_PARM_SCRATCHPAD_CH1,
			&ui32, 0
		    );
		    break;

		  case 2:
		    ui32 = column_heading->width;
		    VMACFGItemListMatchSetValue(
			opt, VMA_CFG_PARM_SCRATCHPAD_CH2,
			&ui32, 0
		    );
		    break;
		}
	    }
	}

	return;
}

/*
 *	Row data delete callback.
 */
void ScratchPadRowDeleteCB(gpointer data)
{
	vma_scratch_pad_row_struct *rd = (vma_scratch_pad_row_struct *)data;
	if(rd == NULL)
	    return;

	rd->core_ptr = NULL;
	rd->scratch_pad_ptr = NULL;

	free(rd->comment);
	rd->comment = NULL;

	free(rd);
	rd = NULL;

	return;
}


/*
 *	Target editor combo text insert callback.
 *
 *	Used to check when a new item has been selected on the 
 *	target_combo.
 */
void ScratchPadTargetComboTextInsertCB(
	GtkEditable *editable,
	const gchar *text, gint length, gint *position,
	gpointer data
)
{
	gchar *name = NULL;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if((text == NULL) || (sp == NULL))
	    return;

	/* If given text is valid, then make a copy of it (since the
	 * given text is not null terminated.
	 */
	if((text != NULL) && (length >= 0))
	{
	    name = (gchar *)malloc((length + 1) * sizeof(gchar));
	    memcpy(name, text, length * sizeof(gchar));
	    name[length] = '\0';
	}
	if(name == NULL)
	    return;

	if(1)
	{
	    /* Update the editor index number on the scratch pad. */
	    sp->editor_num = ScratchPadGetEditorIndex(sp, name);
	}


	/* Deallocate our locally coppied name. */
	if(name != NULL)
	{
	    free(name);
	    name = NULL;
	}

	return;
}

/*
 *	Scratch pad target_combo button enter callback.
 *
 *	Updates the list of target editors on the target_combo whenever
 *	an enter event is recieved.
 */
void ScratchPadTargetComboButtonEnterCB(GtkButton *button, gpointer data)
{
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	ScratchPadUpdateTargetEditors(sp);

	return;
}

/*
 *	Scratch pad general event callback.
 */
gint ScratchPadEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if((event == NULL) || (sp == NULL))
	    return(FALSE);



	return(TRUE);
}


/*
 *	Menu map from button press callback.
 */
gint ScratchPadMenuMapCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	GtkWidget *w;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if((widget == NULL) || (button == NULL) || (sp == NULL))
	    return(FALSE);


	/* Clist? */
	if(widget == sp->clist)
	{
	    GtkCList *clist = GTK_CLIST(widget);
	    gint row, column;

	    if(!gtk_clist_get_selection_info(
		clist, button->x, button->y, &row, &column
	    ))
	    {
		row = -1;
		column = -1;
	    }

	    /* Update DND icon. */
	    ScratchPadDNDSetIcon(sp, button->x, button->y);

	    /* Handle by which button was pressed. */
	    switch(button->button)
	    {
	      case 2:
		/* Forward to clist select callback. */
		if((row >= 0) && (column >= 0))
		{
		    if((row != sp->selected_row) ||
		       (column != sp->selected_column)
		    )
			gtk_clist_select_row(clist, row, column);

		    ScratchPadCListSelectCB(
			clist, row, column, (GdkEvent *)button,
			(gpointer)sp
		    );
		}
		return(TRUE);
		break;

	      case 3:
		w = sp->clist_menu;
		if(w != NULL)
		{
		    /* Map menu. */
		    gtk_menu_popup(
			GTK_MENU(w),
			NULL, NULL, NULL, NULL,
			button->button, button->time
		    );
		    return(TRUE);
		}
		return(TRUE);
		break;
	    }
	}

	return(FALSE);
}

/*
 *	Floating prompt apply callback.
 *
 *	This applies the given value to the selected row.
 */
static void ScratchPadFPromptApplyCB(gpointer data, const gchar *value)
{
	gint column, row;
	GtkWidget *w;
	GtkCList *clist;
	gchar **strv;
	gint strc;
	const gchar *cstrptr;
	vma_scratch_pad_row_struct *rd;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if((value == NULL) || (sp == NULL))
	    return;

	w = sp->clist;
	if(w == NULL)
	    return;

	clist = GTK_CLIST(w);

	column = sp->selected_column;
	row = sp->selected_row;

	if((row < 0) || (row >= clist->rows))
	    return;

	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(rd == NULL)
	    return;

	/* Handle by which column is selected. */
	switch(column)
	{
	  case 0:	/* Vertex. */
	    cstrptr = value;
	    while(ISBLANK(*cstrptr))
		cstrptr++;
	    rd->v.x = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->v.y = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->v.z = atof(cstrptr);
	    break;

	  case 1:	/* Normal. */
	    cstrptr = value;
	    while(ISBLANK(*cstrptr))
		cstrptr++;
	    rd->n.x = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->n.y = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->n.z = atof(cstrptr);
	    break;

	  case 2:	/* Texcoord. */
	    cstrptr = value;
	    while(ISBLANK(*cstrptr))
		cstrptr++;
	    rd->tc.x = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->tc.y = atof(cstrptr);

	    while(!ISBLANK(*cstrptr) && ((*cstrptr) != ',') &&
		  ((*cstrptr) != '\0')
	    )
		cstrptr++;
	    while(ISBLANK(*cstrptr) || ((*cstrptr) == ','))
		cstrptr++;
	    rd->tc.z = atof(cstrptr);
	    break;
	}

	/* Get allocated strings for clist row text. */
	strv = ScratchPadAllocRowText(
	    sp, &rd->v, &rd->n, &rd->tc, &strc
	);

	/* Update clist row text. */
	if(strc > 0)
	    gtk_clist_set_text(clist, row, 0, strv[0]);
	if(strc > 1)
	    gtk_clist_set_text(clist, row, 1, strv[1]);
	if(strc > 2)
	    gtk_clist_set_text(clist, row, 2, strv[2]);

	/* Deallocate row text, we don't need them anymore. */
	strlistfree(strv, strc);
	strv = NULL;
	strc = 0;

	return;
}

/*
 *	Floating prompt cancel callback.
 */
static void ScratchPadFPromptCancelCB(gpointer data)
{


	return;
}

/*
 *	Maps the floating prompt relative to the given scratch pad's
 *	clist at the specified column and row.
 */
static void ScratchPadFPromptMap(    
	vma_scratch_pad_struct *sp, gint column, gint row
)
{
	GtkCList *clist;
	vma_scratch_pad_row_struct *rd;
	gint status, cx, cy, px, py, pwidth, pheight;


	if((sp == NULL) || (column < 0) || (row < 0))
	    return;

	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;

	/* Get clist cell geometry. */
	status = GUICListGetCellGeometry(
	    clist, column, row,
	    &cx, &cy, &pwidth, &pheight
	);
	GUIGetPositionRoot(clist->clist_window, cx, cy, &px, &py);

	/* Get row data. */
	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);


	if((rd != NULL) && (!status))
	{
	    gchar **strv, *strptr = NULL;
	    gint strc;

	    /* Center the prompt along Y. */
	    py = py + ((pheight / 2) - (FPROMPT_DEF_HEIGHT / 2));
	    pheight = FPROMPT_DEF_HEIGHT;
	    if(pwidth < 200)
		pwidth = 200;


	    /* Allocate an array of current value strings. */
	    strv = ScratchPadAllocRowText(
		sp,
		&rd->v, &rd->n, &rd->tc, &strc
	    );

	    /* Check which string to use by which column. */
	    if(strc > column)
		strptr = strv[column];

	    /* Map floating prompt to change values. */
	    FPromptBreakQuery();
	    FPromptSetTransientFor(sp->toplevel);
	    FPromptSetPosition(px, py);
	    FPromptMapQuery(
		NULL,			/* No label. */
		strptr,
		NULL,			/* No tooltip message. */
		FPROMPT_MAP_NO_MOVE,	/* Map code. */
		pwidth, pheight,
		FPROMPT_FLAG_OK_BTN | FPROMPT_FLAG_CANCEL_BTN,
		(gpointer)sp,
		NULL,                       /* Browse callback. */
		ScratchPadFPromptApplyCB,
		ScratchPadFPromptCancelCB
	    );

	    /* Deallocate array of current value strings. */
	    strlistfree(strv, strc);
	}

	return;
}




/*
 *	Destroy callback.
 */
void ScratchPadDestroyCB(GtkObject *object, gpointer data)
{

	return;
}

/*
 *      Scratch pad close callback.
 */
gint ScratchPadCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	ScratchPadRecordPositions((vma_scratch_pad_struct *)data);
	ScratchPadUnmap((vma_scratch_pad_struct *)data);
	return(TRUE);
}

/*
 *	Close button callback.
 */
void ScratchPadCloseButtonCB(
	GtkWidget *widget, gpointer data
)
{
	ScratchPadCloseCB(widget, NULL, data);
	return;
}

/*
 *	Menu item enter event callback.
 */
gint ScratchPadMenuItemEnterCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return(FALSE);



	return(TRUE);
}

/*
 *	Insert new row callback.
 */
void ScratchPadInsertCB(GtkWidget *widget, gpointer data)
{
	mp_vertex_struct v, n, tc;

	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	ScratchPadSyncData(sp);

	memset(&v, 0x00, sizeof(mp_vertex_struct));
	memset(&n, 0x00, sizeof(mp_vertex_struct));
	memset(&tc, 0x00, sizeof(mp_vertex_struct));

	ScratchPadRowAppend(sp, &v, &n, &tc, NULL);

	ScratchPadUpdateMenus(sp);

	return;
}

/*
 *	Edit row callback.
 */
void ScratchPadEditCB(GtkWidget *widget, gpointer data)
{
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	ScratchPadFPromptMap(
	    sp, sp->selected_column, sp->selected_row
	);

	return;
} 

/*
 *	Delete selected row callback.
 */
void ScratchPadDeleteCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkCList *clist;

	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;

	row = sp->selected_row;
	if((row >= 0) && (row < clist->rows))
	{
	    gtk_clist_remove(clist, row);
	}

	ScratchPadUpdateMenus(sp);

	return;
}


/*
 *      Set vertex callback.
 */     
void ScratchPadSetVertexCB(GtkWidget *widget, gpointer data)
{
	gint editor_num, status, row;
	GtkCList *clist;
	const gchar *mesg_ptr;
	vma_scratch_pad_row_struct *rd;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(!sp->initialized)
	    return;

	row = sp->selected_row;
	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;

	if((row < 0) || (row >= clist->rows))
	    return;

	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(rd == NULL)
	    return;

	/* Get currently selected target editor number. */
	sp->editor_num = editor_num = ScratchPadGetEditorIndex(
	    sp, NULL
	);
	if(editor_num < 0)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"No Editor Selected!",
"You have not selected an editor to set values to.",
"You have attempted to set values to an invalid editor\n\
or you have not selected an editor at all. To select\n\
an editor, click on the target editor combo widget's\n\
button to select from a list of editors.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}

	/* Set vertex. */
	mesg_ptr = NULL;
	status = ScratchPadSetVertex(sp, &rd->v);
  	switch(status)
	{
	  case -1:
	    mesg_ptr = "\
A general error was encountered while attempting\n\
to set the new value.";
	    break;

	  case -2:
	    mesg_ptr = "\
The selected primitive does not have vertices.";
	    break;

	  case -3:
	    mesg_ptr = "\
You have selected an invalid editor to set values to.";
	    break;

	  case -4:
	    mesg_ptr = "\
No model or primitives selected on the specified editor.";
	    break;

	  case -5:
	    mesg_ptr = "\
The editor's write protect is enabled, cannot set values\n\
to read-only editor.";
	    break;
	}
	if(mesg_ptr != NULL)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"Error Setting Value!",
		mesg_ptr,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	return;
} 

/*
 *	Set normal callback.
 */
void ScratchPadSetNormalCB(GtkWidget *widget, gpointer data)
{
	gint editor_num, status, row;
	GtkCList *clist;
	const gchar *mesg_ptr;
	vma_scratch_pad_row_struct *rd;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(!sp->initialized)
	    return;   

	row = sp->selected_row;
	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;
	
	if((row < 0) || (row >= clist->rows))
	    return;
	 
	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(rd == NULL)
	    return;

	/* Get currently selected target editor number. */
	sp->editor_num = editor_num = ScratchPadGetEditorIndex(
	    sp, NULL
	);
	if(editor_num < 0)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"No Editor Selected!",
"You have not selected an editor to set values to.",
"You have attempted to set values to an invalid editor\n\
or you have not selected an editor at all. To select\n\
an editor, click on the target editor combo widget's\n\
button to select from a list of editors.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}   


	/* Set normal. */
	mesg_ptr = NULL;
	status = ScratchPadSetNormal(sp, &rd->n);
	switch(status) 
	{
	  case -1:
	    mesg_ptr = "\
A general error was encountered while attempting\n\
to set the new value.";
	    break;

	  case -2:
	    mesg_ptr = "\
The selected primitive does not have vertices.";
	    break;

	  case -3:
	    mesg_ptr = "\
You have selected an invalid editor to set values to.";
	    break;

	  case -4:
	    mesg_ptr = "\
No model or primitives selected on the specified editor.";
	    break;

	  case -5:
	    mesg_ptr = "\
The editor's write protect is enabled, cannot set values\n\
to read-only editor.";
	    break;
	}
	if(mesg_ptr != NULL)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"Error Setting Value!",
		mesg_ptr,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	return;
} 

/*
 *	Set texcoord callback.
 */
void ScratchPadSetTexCoordCB(GtkWidget *widget, gpointer data)
{
	gint editor_num, status, row;
	GtkCList *clist; 
	const gchar *mesg_ptr;
	vma_scratch_pad_row_struct *rd;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(!sp->initialized)
	    return;

	row = sp->selected_row;
	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;

	if((row < 0) || (row >= clist->rows))
	    return;  

	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(rd == NULL)
	    return;

	/* Get currently selected target editor number. */
	sp->editor_num = editor_num = ScratchPadGetEditorIndex(
	    sp, NULL
	);
	if(editor_num < 0)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"No Editor Selected!",
"You have not selected an editor to set values to.",
"You have attempted to set values to an invalid editor\n\
or you have not selected an editor at all. To select\n\
an editor, click on the target editor combo widget's\n\
button to select from a list of editors.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}   

	/* Set texcoord. */
	mesg_ptr = NULL;
	status = ScratchPadSetTexCoord(sp, &rd->tc);
	switch(status)
	{
	  case -1:
	    mesg_ptr = "\
A general error was encountered while attempting\n\
to set the new value.";
	    break;

	  case -2:
	    mesg_ptr = "\
The selected primitive does not have vertices.";
	    break;

	  case -3:
	    mesg_ptr = "\
You have selected an invalid editor to set values to.";
	    break;

	  case -4:
	    mesg_ptr = "\
No model or primitives selected on the specified editor.";
	    break;

	  case -5:
	    mesg_ptr = "\
The editor's write protect is enabled, cannot set values\n\
to read-only editor.";
	    break;
	}
	if(mesg_ptr != NULL)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"Error Setting Value!",
		mesg_ptr,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	return;
}

/*
 *      Set all callback.
 */     
void ScratchPadSetAllCB(GtkWidget *widget, gpointer data)
{
	gint editor_num, status, row;
	GtkCList *clist;
	const gchar *mesg_ptr;
	vma_scratch_pad_row_struct *rd;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(!sp->initialized)
	    return; 

	row = sp->selected_row;
	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return;

	if((row < 0) || (row >= clist->rows))
	    return;

	rd = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(rd == NULL)
	    return;

	/* Get currently selected target editor number. */
	sp->editor_num = editor_num = ScratchPadGetEditorIndex(
	    sp, NULL
	);
	if(editor_num < 0)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"No Editor Selected!",
"You have not selected an editor to set values to.",
"You have attempted to set values to an invalid editor\n\
or you have not selected an editor at all. To select\n\
an editor, click on the target editor combo widget's\n\
button to select from a list of editors.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}

	/* Set vertex, normal, and texcoord. */
	mesg_ptr = NULL;
	status = ScratchPadSetAll(sp, &rd->v, &rd->n, &rd->tc);
	switch(status)
	{
	  case -1:
	    mesg_ptr = "\
A general error was encountered while attempting\n\
to set the new value.";
	    break;

	  case -2:
	    mesg_ptr = "\
The selected primitive does not have vertices.";
	    break;

	  case -3:
	    mesg_ptr = "\
You have selected an invalid editor to set values to.";
	    break;

	  case -4:
	    mesg_ptr = "\
No model or primitives selected on the specified editor.";
	    break;

	  case -5:
	    mesg_ptr = "\
The editor's write protect is enabled, cannot set values\n\
to read-only editor.";
	    break;
	}
	if(mesg_ptr != NULL)
	{
	    CDialogSetTransientFor(sp->toplevel);
	    CDialogGetResponse(
"Error Setting Value!",
		mesg_ptr,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	return;
} 


/*
 *	Scratch pad clist select_row callback.
 */
void ScratchPadCListSelectCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	static gbool reenterant = FALSE;
	GdkEventButton *button = NULL;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if((clist == NULL) || (sp == NULL))
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Check if event is valid and if it is a variation of a button
	 * press. If it is then we set the GdkEventButton pointer to it.
	 */
	if(event != NULL)
	{
	    if((event->type == GDK_BUTTON_PRESS) ||
	       (event->type == GDK_2BUTTON_PRESS) ||
	       (event->type == GDK_3BUTTON_PRESS)
	    )
		button = (GdkEventButton *)event;
	}

	/* Change in row selection? */
	if(sp->selected_row != row)
	{
	    /* Break floating prompt query unconditionally whenever
	     * the selection has changed.
	     */
	    FPromptBreakQuery();

	    /* Sync data with respect to the previously selected row. */
	    ScratchPadSyncData(sp);

	    /* Update selected row and column. */
	    sp->selected_row = row;
	    sp->selected_column = column;

	    /* Get comments from newly selected row. */
	    ScratchPadCommentsTextFetch(sp, NULL);
	}

	/* Update selected column unconditionally (it might have already
	 * been updated when the row change was checked).
	 */
	sp->selected_column = column;

	/* Double clicked on button 1? */
	if((button == NULL) ? 0 : (button->button == 1))
	{
	    if(button->type == GDK_2BUTTON_PRESS)
	    {
		switch(column)
		{
		  case 0:
		    ScratchPadSetVertexCB(NULL, data);
		    break;

		  case 1:
		    ScratchPadSetNormalCB(NULL, data);
		    break;

		  case 2:
		    ScratchPadSetTexCoordCB(NULL, data);
		    break;
		}
	    }
	}

	/* Pressed button 2? */
	if((button == NULL) ? 0 : (button->button == 2))
	{
	    ScratchPadFPromptMap(sp, column, row);
	}

	/* Update menus on scratchpad. */
	ScratchPadUpdateMenus(sp);

	reenterant = FALSE;
	return;
}

/*
 *      Scratch pad clist unselect_row callback.
 */
void ScratchPadCListUnselectCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	static gbool reenterant = FALSE;
	GdkEventButton *button = NULL;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Check if event is valid and if it is a variation of a button
	 * press. If it is then we set the GdkEventButton pointer to it.
	 */
	if(event != NULL)
	{
	    if((event->type == GDK_BUTTON_PRESS) ||
	       (event->type == GDK_2BUTTON_PRESS) ||
	       (event->type == GDK_3BUTTON_PRESS)   
	    )
		button = (GdkEventButton *)event;
	}

	/* Break floating prompt query unconditionally whenever a row
	 * is unselected.
	 */
	FPromptBreakQuery();

	/* Sync data before unselecting. */
	ScratchPadSyncData(sp);

	/* Mark row as unselected. */
	sp->selected_column = -1;
	sp->selected_row = -1;

	/* Update menus on scratchpad. */
	ScratchPadUpdateMenus(sp);

	reenterant = FALSE;
	return;
}

/*
 *	Comments text changed callback.
 */
void ScratchPadTextChangeCB(
	GtkEditable *editable, gpointer data  
)
{
	static gbool reenterant = FALSE;
	vma_scratch_pad_struct *sp = (vma_scratch_pad_struct *)data;
	if(sp == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Update has changes mark on scratch pad. */
	if(!sp->comments_text_has_changes)
	    sp->comments_text_has_changes = TRUE;

	/* Update menus on scratchpad. */
	ScratchPadUpdateMenus(sp);

	reenterant = FALSE;
	return;
}
