#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "vpi.h"
#include "vpiinternal.h"
#include "vpiinternalfio.h"

#include "config.h"


static void STR_END_BLANK(char *s);
static char *STR_SEEK_PAST_BLANK(char *s);
static int MATCH_PLUGIN_NAME(
	const char *filename, const char *s
);
static vma_plugin_struct *MATCH_PLUGIN(
	vma_plugin_struct **plugin, int total_plugins,
	const char *filename
);

int VPILoadConfigurationFromFile(
	vma_plugin_struct ***plugin, int *total_plugins,
	const char *filename
);
int VPISaveConfigurationToFile(
	vma_plugin_struct **plugin, int total_plugins,
	const char *filename
);


/*
 *	Sets the first blank char in s to '\0',
 */
static void STR_END_BLANK(char *s)
{
	if(s == NULL)
	    return;

	while((*s) != '\0')
	{
	    if(ISBLANK(*s))
	    {
		(*s) = '\0';
		break;
	    }
	    s++;
	}
	return;
}

/*
 *	Seeks the given pointer s past the next blank char if any.
 */
static char *STR_SEEK_PAST_BLANK(char *s)
{
	if(s == NULL)
	    return(NULL);

	while(!ISBLANK(*s) && ((*s) != '\0'))
	    s++;
	while(ISBLANK(*s))
	    s++;

	return(s);
}

/*
 *	Returns true if the string s (a plug-in name) matches
 *	the given file name.
 */
static int MATCH_PLUGIN_NAME(
	const char *filename, const char *s
)
{
	if((filename == NULL) || (s == NULL))
	    return(0);

	if(strcmp(filename, s))
	    return(0);
	else
	    return(1);
}

/*
 *	Returns pointer to plug-in that matches the given filename from
 *	the list of plug-ins.
 */
static vma_plugin_struct *MATCH_PLUGIN(
	vma_plugin_struct **plugin, int total_plugins,
	const char *filename
)
{
	int i;
	vma_plugin_struct *plugin_ptr;

	if((plugin == NULL) || (total_plugins <= 0) || (filename == NULL))
	    return(NULL);

	for(i = 0; i < total_plugins; i++)
	{
	    plugin_ptr = plugin[i];
	    if(plugin_ptr == NULL)
		continue;

	    if(plugin_ptr->filename == NULL)
		continue;

	    if(MATCH_PLUGIN_NAME(plugin_ptr->filename, filename))
		return(plugin_ptr);
	}

	return(NULL);
}

/*
 *	Loads configuration from file and modifies the given
 *	plug-in's list as needed.
 *
 *	Returns non-zero on error.
 */
int VPILoadConfigurationFromFile(
	vma_plugin_struct ***plugin, int *total_plugins,
	const char *filename
)
{
	FILE *fp;
	struct stat stat_buf;
	char *buf, *buf_ptr;
	vma_plugin_struct *plugin_ptr;
	char plugin_filename[PATH_MAX + NAME_MAX];
	int enabled;


	if((filename == NULL) || (plugin == NULL) || (total_plugins == NULL))
	    return(-1);
	if((*total_plugins) < 0)
	    (*total_plugins) = 0;

	if(stat(filename, &stat_buf))
	{
/*
	    fprintf(
		stderr,
		"%s: No such file.\n",
		filename
	    );
 */
	    return(-1);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
	    fprintf(
		stderr,
		"%s: Not a file.\n",
		filename
	    );
	    return(-1);
	}

	/* Open file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open.\n",
		filename
	    );
 */
	    return(-1);
	}

	/* Begin reading file. */
	buf = NULL;
	while(1)
	{
	    free(buf);
	    buf = FGetString(fp);
	    if(buf == NULL)
		break;

	    /* Begin parsing line. */
	    buf_ptr = buf;
	    while(ISBLANK(*buf_ptr))
		buf_ptr++;

	    /* Skip comments. */
	    if((*buf_ptr) == VMA_CFG_COMMENT_CHAR)
		continue;

	    /* Parse line of the following format:
	     *
	     *	<plugin name> <enabled|disabled> <create_time> <last_used_time>
	     *
	     */
	    strncpy(plugin_filename, buf_ptr, PATH_MAX + NAME_MAX);
	    plugin_filename[PATH_MAX + NAME_MAX - 1] = '\0';
	    STR_END_BLANK(plugin_filename);

	    /* Match plug-in from the given list with the parsed plug-in
	     * name.
	     */
	    plugin_ptr = MATCH_PLUGIN(
		*plugin, *total_plugins, plugin_filename
	    );
	    /* Plug-in exist? */
	    if(plugin_ptr != NULL)
	    {
		time_t time_last_used;


		buf_ptr = STR_SEEK_PAST_BLANK(buf_ptr);
		enabled = atoi(buf_ptr);
		if(enabled)
		    VPIEnable(plugin_ptr);
		else
		    VPIDisable(plugin_ptr);

		buf_ptr = STR_SEEK_PAST_BLANK(buf_ptr);
		plugin_ptr->time_created = atol(buf_ptr);

		buf_ptr = STR_SEEK_PAST_BLANK(buf_ptr);
		time_last_used = atol(buf_ptr);
		if(time_last_used > plugin_ptr->time_last_used)
		    plugin_ptr->time_last_used = time_last_used;
	    }
	}


	/* Close file. */
	FClose(fp);
	fp = NULL;

	return(0);
}

/*
 *      Loads configuration information based on the given list to the
 *	specified configuration file.
 *
 *      Returns non-zero on error.
 */
int VPISaveConfigurationToFile(
	vma_plugin_struct **plugin, int total_plugins,
	const char *filename
)
{
	int i;
	FILE *fp;
	vma_plugin_struct *plugin_ptr;


	if((filename == NULL) || (plugin == NULL) || (total_plugins <= 0))
	    return(-1);

	/* Open file for writing. */
	fp = FOpen(filename, "wb");
	if(fp == NULL)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open.\n",
		filename
	    );
 */
	    return(-1);
	}

	/* Itterate through each plugin. */
	for(i = 0; i < total_plugins; i++)
	{
	    plugin_ptr = plugin[i];
	    if(plugin_ptr == NULL)
		continue;

	    /* Can't write anything if filename is NULL. */
	    if(plugin_ptr->filename == NULL)
		continue;

	    fprintf(
		fp,
		"%s %i %ld %ld\n",
		plugin_ptr->filename,
		((plugin_ptr->handle == NULL) ? 0 : 1),
		plugin_ptr->time_created,
		plugin_ptr->time_last_used
	    );
	}

	FClose(fp);
	fp = NULL;

	return(0);
}
