// Copyright 2005 Ben Hutchings <ben@decadent.org.uk>.
// See the file "COPYING" for licence details.

#include "style_sheets.hpp"

#include <nsContentCID.h>
#if MOZ_VERSION_MAJOR > 1 || (MOZ_VERSION_MAJOR == 1 && MOZ_VERSION_MINOR >= 8)
#   include <nsIStyleSheetService.h>
#   include <nsServiceManagerUtils.h>
#else
#   include <nsICSSLoader.h>
#   include <nsICSSStyleSheet.h>
#   include <nsIPresShell.h>
#   include <nsIServiceManagerUtils.h>
#endif
#include <nsIURI.h>
#include <nsNetUtil.h>

#include "xpcom_support.hpp"

using xpcom_support::check;

#if MOZ_VERSION_MAJOR > 1 || (MOZ_VERSION_MAJOR == 1 && MOZ_VERSION_MINOR >= 8)

// We just have to load and register a style-sheet as a user
// style-sheet.  There is no need to do anything for each page.

agent_style_sheet_holder init_agent_style_sheet(const char * uri)
{
    nsCOMPtr<nsIURI> style_sheet_uri;
    check(NS_NewURI(getter_AddRefs(style_sheet_uri), nsCString(uri)));

    nsCOMPtr<nsIStyleSheetService> style_sheet_service;
    static const nsCID style_sheet_service_cid = {
	// NS_STYLESHEETSERVICE_CID copied from
	// layout/base/nsStyleSheetService.cpp
	0xfcca6f83, 0x9f7d, 0x44e4,
	{0xa7, 0x4b, 0xb5, 0x94, 0x33, 0xe6, 0xc8, 0xc3}
    };
    check(CallGetService<nsIStyleSheetService>(
	      style_sheet_service_cid, getter_AddRefs(style_sheet_service)));
    check(style_sheet_service->LoadAndRegisterSheet(
	      style_sheet_uri, nsIStyleSheetService::USER_SHEET));

    return agent_style_sheet_holder();
}

#else // Mozilla version < 1.8

already_AddRefed<nsIStyleSheet> init_agent_style_sheet(const char * uri)
{
    nsCOMPtr<nsICSSLoader> css_loader;
    static const nsCID css_loader_cid = NS_CSS_LOADER_CID;
    check(CallGetService<nsICSSLoader>(css_loader_cid,
				       getter_AddRefs(css_loader)));

    nsCOMPtr<nsIURI> style_sheet_uri;
    check(NS_NewURI(getter_AddRefs(style_sheet_uri), nsCString(uri)));

    nsICSSStyleSheet * style_sheet;
    check(css_loader->LoadAgentSheet(style_sheet_uri, &style_sheet));
    return style_sheet;
}

// Apply a style-sheet to a given presentation shell as the top-priority
// agent style-sheet and disable the preferences-derived style rules.
void apply_agent_style_sheet(nsIStyleSheet * style_sheet,
			     nsIPresShell * pres_shell)
{
    nsCOMArray<nsIStyleSheet> style_sheets;
    check(pres_shell->GetAgentStyleSheets(style_sheets));
    check(style_sheets.InsertObjectAt(style_sheet, 0));
    check(pres_shell->SetAgentStyleSheets(style_sheets));

    check(pres_shell->EnablePrefStyleRules(false));

    // Update the display
    check(pres_shell->ReconstructStyleData());
    check(pres_shell->FlushPendingNotifications(true));
}

#endif // Mozilla version >=/< 1.8
