/*
 * The 3D Studio File Format Library
 * Copyright (C) 1996-2001 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mainwnd.cpp,v 1.4 2001/01/15 10:56:12 jeh Exp $
 */
#include "mainwnd.h"
#include <lib3ds/file.h>
#include <lib3ds/camera.h>
#include "viewport.h"
#include <qapplication.h>
#include <qslider.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qframe.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qstatusbar.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qaccel.h>
#include <qiconset.h>
#include <config.h>


static QApplication *app=0;


#include "filenew.xpm"
#include "fileopen.xpm"
#include "filesave.xpm"
#include "timectrls_begin.xpm"
#include "timectrls_prev.xpm"
#include "timectrls_play.xpm"
#include "timectrls_stop.xpm"
#include "timectrls_next.xpm"
#include "timectrls_end.xpm"


/*!
 *
 */
MainWnd::MainWnd(Document *doc, QWidget* parent, const char *name)
{
  int id;

  if (doc) {
    d_doc=doc;
  }
  else {
    d_doc=new Document();
  }
  setCaption("View3ds - 3D Studio File Viewer " VERSION );
  connect(d_doc, SIGNAL(documentChanged()), SLOT(documentChanged()));

  
  QPixmap newIcon((const char**)filenew_xpm);
  QPixmap openIcon((const char**)fileopen_xpm);
  QPixmap saveIcon((const char**)filesave_xpm);

  QToolBar* fileTools;
  fileTools = new QToolBar( this, "file operations" );
  fileTools->setLabel( tr( "File Operations" ) );
  QToolButton * fileOpen = new QToolButton( openIcon, "Open File", QString::null,
    this, SLOT(load()), fileTools, "open file" );
  QToolButton * fileSave = new QToolButton( saveIcon, "Save File", QString::null,
    this, SLOT(save()), fileTools, "save file" );
  d_cameras=new QComboBox(fileTools);
  d_doc->connect(d_cameras, SIGNAL(activated(const QString&)), 
    SLOT(setCamera(const QString&)));

  QPopupMenu *file = new QPopupMenu( this );
  menuBar()->insertItem( "&File", file );
  file->insertItem( newIcon, "&New", this, SLOT(newDoc()), CTRL+Key_N );
  file->insertItem( openIcon, "&Open", this, SLOT(load()), CTRL+Key_O );
  file->insertItem( saveIcon, "&Save", this, SLOT(save()), CTRL+Key_S );
  file->insertItem( "Save &as...", this, SLOT(saveAs()) );
  file->insertSeparator();
  file->insertItem( "&Close", this, SLOT(close()), CTRL+Key_W );
  file->insertItem( "&Quit", app, SLOT( closeAllWindows() ), CTRL+Key_Q );

  QPopupMenu * help = new QPopupMenu( this );
  menuBar()->insertSeparator();
  menuBar()->insertItem( "&Help", help );
  help->insertItem( "&About", this, SLOT(about()), Key_F1 );
  help->insertItem( "About &Qt", this, SLOT(aboutQt()) );

  QWidget *cw=new QWidget(this);
  setCentralWidget(cw);
  QVBoxLayout *vl=new QVBoxLayout(cw);

  {
    QFrame *frame=new QFrame(cw);
    vl->addWidget(frame);
    frame->setFrameStyle(QFrame::WinPanel | QFrame::Raised);

    QHBoxLayout *hl=new QHBoxLayout(frame);
    hl->setMargin(2);
    QGLFormat f=QGLFormat::defaultFormat();
    d_viewport=new Viewport(d_doc, f, frame);
    hl->addWidget(d_viewport);
    d_viewport->connect(d_doc, SIGNAL(documentChanged()), SLOT(updateGL()));
  }
  {
    QFrame *frame=new QFrame(cw);
    vl->addWidget(frame);
    frame->setFrameStyle(QFrame::WinPanel | QFrame::Raised);
    QHBoxLayout *hl=new QHBoxLayout(frame);
    hl->setMargin(2);

    d_slider=new QSlider(0, 1, 10, 0, QSlider::Horizontal, frame);
    hl->addWidget(d_slider);
    d_slider->setFixedHeight(d_slider->sizeHint().height());
    d_slider->setTickmarks(QSlider::Below);
    d_slider->setTracking(true);
    d_doc->connect(d_slider, SIGNAL(valueChanged(int)), SLOT(setCurrent(int)));

    d_edit=new QLineEdit(frame);
    d_edit->setText("0");
    d_edit->setFixedSize(d_edit->sizeHint());
    hl->addWidget(d_edit);
    connect(d_edit, SIGNAL(returnPressed()), SLOT(currentChanged()));

    {
      QSize bsize(d_edit->sizeHint().height()*2, d_edit->sizeHint().height()*2);
      QPushButton *b;

      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_begin_xpm));
      hl->addWidget(b);
      d_doc->connect(b, SIGNAL(clicked()), SLOT(begin()));

      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_prev_xpm));
      hl->addWidget(b);
      d_doc->connect(b, SIGNAL(clicked()), SLOT(prev()));

      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_play_xpm));
      hl->addWidget(b);
      d_doc->connect(b, SIGNAL(clicked()), SLOT(play()));
      
      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_stop_xpm));
      d_doc->connect(b, SIGNAL(clicked()), SLOT(stop()));
      hl->addWidget(b);

      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_next_xpm));
      hl->addWidget(b);
      d_doc->connect(b, SIGNAL(clicked()), SLOT(next()));

      b=new QPushButton(frame);
      b->setPixmap(QPixmap((const char**)timectrls_end_xpm));
      hl->addWidget(b);
      d_doc->connect(b, SIGNAL(clicked()), SLOT(end()));
    }
    frame->setFixedHeight(frame->sizeHint().height());
  }
  {
    QAccel *a = new QAccel( this );
    a->connectItem( a->insertItem(Key_Escape), d_doc, SLOT(stop()));
  }
  statusBar()->message( "Ready", 2000 );
}


/*!
 *
 */
MainWnd::~MainWnd()
{
}


/*!
 *  
 */
void 
MainWnd::about()
{
  QMessageBox::about( this, 
    "View3ds - 3DS File Viewer " VERSION,
    "The 3D Studio File Format Library\n"
    "Copyright (C) 1996-2001 by J.E. Hoffmann <je-h@gmx.net>\n"
    "All rights reserved."
  );
}


/*!
 *  
 */
void 
MainWnd::aboutQt()
{
  QMessageBox::aboutQt(this);
}


/*!
 *  
 */
void 
MainWnd::newDoc()
{
  MainWnd *wnd = new MainWnd(0);
  wnd->resize(500,500);
  wnd->show();
}


/*!
 *  
 */
void 
MainWnd::load()
{
  QString fn = QFileDialog::getOpenFileName(
    QString::null,
    "3D-Studio Files (*.3ds)",
    this
  );
  if ( fn.isEmpty() ) {
    return;
  }
 
  d_filename = fn;
  setCaption(QString("View3ds: %1" ).arg( d_filename ));
  d_doc->open(fn);
}


/*!
 *  
 */
void 
MainWnd::save()
{
  if ( d_filename.isEmpty() ) {
    saveAs();
    return;
  }
  
  d_doc->save(d_filename);
  setCaption(QString("View3ds: %1" ).arg( d_filename ));
  statusBar()->message( QString( "File %1 saved" ).arg( d_filename ), 2000 );
}


/*!
 *  
 */
void 
MainWnd::saveAs()
{
  QString fn = QFileDialog::getSaveFileName( QString::null, QString::null, this );
  if ( !fn.isEmpty() ) {
    d_filename = fn;
    save();
  } 
  else {
    statusBar()->message( "Saving aborted", 2000 );
  }
}


/*!
 *  
 */
void 
MainWnd::documentChanged()
{
  Lib3dsFile *f=d_doc->file();
  if (f) {
    d_slider->setRange(0, f->frames);
    d_slider->setValue(f->current_frame);
    d_slider->setTickmarks(QSlider::Below);
    {
      d_edit->setText(QString("%1").arg(f->current_frame));
    }

    {
      d_cameras->clear();
      Lib3dsCamera *c;
      for (c=f->cameras; c; c=c->next) {
        d_cameras->insertItem(c->name);
      }
    }
  }
}


/*!
 *  
 */
void 
MainWnd::currentChanged()
{
  d_doc->setCurrent(d_edit->text().toInt());
}


/*!
 * Program entry point.
 */
int 
main(int argc, char **argv)
{
  app=new QApplication(argc, argv);
  MainWnd *mainWnd=new MainWnd(0);
  app->setMainWidget(mainWnd);
  mainWnd->resize(400,400);
  mainWnd->show();
  app->exec();
  delete app;
  return(0);
}
