// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        svbsptree.h
//
// Purpose:     Interface to class BSPTree
//
// Created:     24 May 1996  Georg Meszaros
//
// Modified:    13 Jun 1996  Georg Meszaros
//
//
// Description:
//
// Header of the class SVBSPTree 
// 
//
//</file>


//<class>
//
// Name:       SVBSPTree
//
// Purpose:    
//
//
// Public Interface:
//
/*
   - SVBSPTree(); 
   - SVBSPTree(const point3D& eye, float margin, int use_x_faces, BSPTree* bsp_root); 
     > constructor for the root of the shadow volume BSP-tree
       eye ... position of the camera in world coordinates
       margin ...  0   :  use all polygons for visibility calculations
                   0.5 :  use all polygons which projection is > 0.5 * largest polygon
                   1   :  no visibility calculation at all
       use_x_faces ... use only the first "use_x_faces" polygons for vis calc
       bsp_root ... the BSP-Tree that is used to build the SVBSP-Tree 

   - SVBSPTree(NodeType type);
     > for the internal building of the SVBSP-Tree: VISIBLE, SHADOW, PLANE 
       VISIBLE: the side of a shadow plane that is visible
       SHADOW:  the side of a shadow plane that is not visible
       PLANE:   the node contains a shadow plane equation

   - ~SVBSPTree();
     > destroy the SVBSP-Tree recursively

   - void insertViewVolume(Camera* camera);
     > include the view volume planes in the SVBSP-Tree, needs to be
       done with the empty root

   - void replace(Face* face, Face* original_face);
     > actually puts a new shadow volume into the tree - only instead
       of a VISIBLE node useful 
   
   - void filter(Face* face, Face* original_face);
     > filters a face down the SVBSP-Tree until a visible part extends
       the SVBSP-Tree using replace

   - point3D& eye() const;
   - unsigned long nodeNumber() const;
 
   counters for the whole SVBSP-Tree
   - static unsigned long visible_face_count_;
   - static unsigned long hidden_face_count_;
   - static unsigned long back_face_count_;
   - static unsigned long visible_split_face_count_;
 
   - static void calcPlaneEqu(const point3D& vertex1, 
                      const point3D& vertex2, 
                      const point3D& vertex3,
                      Plane& plane_equation);
     > simple plane equation calculation out of 3 vertices

*/
// Description:
//
// Implementation of the SVBSP-tree algorithm by Chin, Feiner
// "Near real time shadow generation using BSP-trees" as a visibility
// approach only.
//
//</class>


#ifndef harmony_scene_svbsptree_h
#define harmony_scene_svbsptree_h

#include "ge3d/vectors.h"

struct Plane
{
  vector3D normal;
  float d;
};

#include "polygon.h"

class BSPTree;
class Camera;

enum NodeType { VISIBLE, SHADOW, PLANE }; 


class SVBSPTree
{
  public:
    
    SVBSPTree(); 
    SVBSPTree(const point3D& eye, float margin, int use_x_faces, BSPTree* bsp_root); 
    SVBSPTree(NodeType type);
    ~SVBSPTree();

    void insertViewVolume(Camera* camera);

    void replace(Face* face, Face* original_face);   
    void filter(Face* face, Face* original_face);

    point3D& eye() const;
    unsigned long nodeNumber() const;
 
    static unsigned long visible_face_count_;
    static unsigned long hidden_face_count_;
    static unsigned long back_face_count_;
    static unsigned long visible_split_face_count_;
 
    static void calcPlaneEqu(const point3D& vertex1, 
                      const point3D& vertex2, 
                      const point3D& vertex3,
                      Plane& plane_equation);


  private:

    void calcPlaneEqu(const point3D& vertex1, 
                      const point3D& vertex2, 
                      const point3D& vertex3);
      
    void setPlaneEqu(const Plane& plane_equation); 
   
    static unsigned long node_number_;
    static unsigned long face_number_;
    static point3D eye_;  
    static BSPTree* bsp_root_;    
    static float margin_;
    static int use_x_faces_;

    Plane plane_; // shadow plane;
    NodeType type_;
    SVBSPTree* visible_;
    SVBSPTree* shadow_;

}; // SVBSPTree


inline point3D& SVBSPTree::eye() const { return eye_; }
inline unsigned long SVBSPTree::nodeNumber() const { return node_number_; }
inline void SVBSPTree::setPlaneEqu(const Plane& plane_equation) { plane_ = plane_equation; } 
   

#endif
