#ifndef WARPED_UNIT_TEST_H
#define WARPED_UNIT_TEST_H

#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>

#include "SimulationObjectAdapter.h"
#include "TimeWarpSimulationManagerAdapter.h"
#include "warped/DefaultEvent.h"
#include "warped/DefaultTimeWarpEventSet.h"

/**
   An abstract class that other tests can derive from.  Tries to set up a
   minimal but "sane enough" environment for unit testing chunks of the
   warped kernel.
*/

class WarpedUnitTest : public CppUnit::TestFixture {
public:
  WarpedUnitTest() : simObj1(0), simMgr(0) {}

  /**
     Set up our minimal environment.  Automagically called prior to each
     test case by CPPUnit.
  */
  void setUp(){
    simObj1 = allocateSimObject( getSimObj1Name() );
    simMgr =  allocateTimeWarpSimulationManager( simObj1 );
  }
  
  /**
     Destructs our minimal environment.  Automagically called after each
     test case by CPPUnit.
  */
  void tearDown(){
    // Note that simMgr kills the object(s) for us.
    delete simMgr;
    simMgr = 0;
  }

  /**
     Factory method that test cases can use to override the construction of
     the SimulationObjects...
  */
  virtual SimulationObject *allocateSimObject( const string &name ){
    return new UnitTestSimulationObject( name );
  }

  /**
     Factory method that test cases can use to override the construction of
     TimeWarpSimulationManagers.
  */
  virtual TimeWarpSimulationManager *
  allocateTimeWarpSimulationManager( SimulationObject *initObject ){
    return new UnitTestSimulationManager( initObject );
  }

  SimulationObject *getSimObj1(){
    return simObj1;
  }

  TimeWarpSimulationManager *getSimManager(){
    return simMgr;
  }
  
  static const string &getDefaultSender(){
    static const string defaultSender = getSimObj1Name();
    return defaultSender;
  }
  
  class UnitTestEvent : public DefaultEvent {
  public:
    UnitTestEvent( const VTime &recvTime,
		   const string &receiver ) :
      DefaultEvent( IntVTime::getIntVTimeZero(),
		    recvTime,
		    getDefaultSender(),
		    receiver ){}
    const string &getDataType() const { static string type = "UnitTestEvent"; return type; }
    bool eventCompare( const Event *compareTo ){ return compareEvents( this, compareTo ); }
  };

  class UnitTestSimulationManager : public TimeWarpSimulationManagerAdapter {
  public:
    ~UnitTestSimulationManager(){
      delete obj1;
      obj1 = 0;
      delete eventSet;
    }

    UnitTestSimulationManager( SimulationObject *initObj1 ) :
      obj1( initObj1 ),
      eventSet( new DefaultTimeWarpEventSet(this) ){
      obj1->setSimulationManager( this );
    }

    SimulationObject *getObjectHandle( const string &objectName ) const {
      SimulationObject *retval = 0;
      if( obj1->getName() == objectName ){
	retval = obj1;
      }
      else{
	std::cerr << "SimulationObject " << objectName << " unknown, aborting!" << std::endl;
	abort();
      }
      return retval;
    }

    TimeWarpEventSet *getEventSetManager(){ return eventSet; }

    vector<string> *getSimulationObjectNames(){
      vector<string> *retval = new vector<string>;
      retval->push_back( obj1->getName() );
      return retval;
    }

  private:
    SimulationObject *obj1;
    TimeWarpEventSet *eventSet;
  };

  class UnitTestSimulationObject : public SimulationObjectAdapter {
  public:
    UnitTestSimulationObject( const string &initName ) : myName( initName ){}

    const string &getName() const { return myName; }

  private:
    const string myName;
  };

  static const string &getSimObj1Name(){ static string retval = "SimObj1"; return retval; }

  /**
     An event with zero receive time.
  */
  static Event *getEvent1();
  /**
     An event with zero receive time.
  */
  static Event *getEvent2();
  /**
     An event with 1 receive time.
  */
  static Event *getEvent3();
  /**
     An event with zero receive time.
  */
  static Event *getEvent4();

  static vector<Event *>  &getDefaultEvents();
  static void checkDefaultEvents( const Event *event1,
				  const Event *event2,
				  const Event *event3,
				  const Event *event4 );

  static void              doDefaultInsert( TimeWarpEventSet *toTest );

private:
  SimulationObject *simObj1;
  TimeWarpSimulationManager *simMgr;
  
};

#endif
