/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="cisco_pix"
%option outfile="cisco_pix.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#define IPLEN 16

#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "cisco_pix.h"
#include "common.h"
#include "defs.h"

enum {
  CP_OPT_NONE,
  CP_OPT_HOST,
  CP_OPT_TCP,
  CP_OPT_TCP_S,
  CP_OPT_UDP,
  CP_OPT_UDP_S,
  CP_OPT_ICMP,
  CP_OPT_DST,
  CP_OPT_DST_S
};

#define BUFSIZE 1024
#define BUFSIZE_S "1024"

static void cisco_pix_parse_date(char* input, unsigned char mode);
static void cisco_pix_parse_src(char* input, unsigned char mode);
static void cisco_pix_parse_dst(char* input, unsigned char mode);
static void cisco_pix_parse_group(char* input);

static wf_logentry* logentry;
%}

MONTH	"Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec"
STRING	[a-zA-Z-][a-zA-Z0-9._-]*
LOGHOST	[0-9.a-zA-Z_-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
HEXDIGIT	[0-9a-fA-F]
HEXNUM	"0x"{HEXDIGIT}+
SYSID	"%PIX"|"%FWSM"
PIX	{SYSID}"-"[1-6]"-"[0-9]{6}":"

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}		cisco_pix_parse_date(cisco_pixtext, CP_OPT_HOST);
{PIX}	/* ignore */
{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{4}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}":"		cisco_pix_parse_date(cisco_pixtext, CP_OPT_NONE);

" Inbound TCP connection denied from "{IPADDR}"/"{PORT}	cisco_pix_parse_src(cisco_pixtext + 36, CP_OPT_TCP);
" Deny TCP (no connection) from "{IPADDR}"/"{PORT}	cisco_pix_parse_src(cisco_pixtext + 31, CP_OPT_TCP);
" Deny inbound UDP from "{IPADDR}"/"{PORT}		cisco_pix_parse_src(cisco_pixtext + 23, CP_OPT_UDP);
" Deny udp src "{STRING}":"{IPADDR}"/"{PORT}		cisco_pix_parse_src(cisco_pixtext + 14, CP_OPT_UDP_S);
" Deny tcp src "{STRING}":"{IPADDR}"/"{PORT}		cisco_pix_parse_src(cisco_pixtext + 14, CP_OPT_TCP_S);
" Deny inbound tcp src "{STRING}":"{IPADDR}"/"{PORT}	cisco_pix_parse_src(cisco_pixtext + 22, CP_OPT_TCP_S);
" Deny inbound (No xlate) tcp src "{STRING}":"{IPADDR}"/"{PORT}	cisco_pix_parse_src(cisco_pixtext + 33, CP_OPT_TCP_S);

{IPADDR}" attempted to ping "	cisco_pix_parse_src(cisco_pixtext, CP_OPT_ICMP);
"to "{IPADDR}"/"{PORT}		cisco_pix_parse_dst(cisco_pixtext + 3, CP_OPT_DST);
"dst "{STRING}":"{IPADDR}"/"{PORT}	cisco_pix_parse_dst(cisco_pixtext + 4, CP_OPT_DST_S);
{IPADDR}				cisco_pix_parse_dst(cisco_pixtext, CP_OPT_NONE);

"ICMP echo request"	{
  logentry->protocol = IPPROTO_ICMP;
  logentry->sport = 8; /* type echo-request */
}
"from "{IPADDR}" "	cisco_pix_parse_src(cisco_pixtext + 5, CP_OPT_ICMP);

"flags"		/* ignore */
"FIN"			logentry->tcpflags |= TCP_FIN;
"SYN"			logentry->tcpflags |= TCP_SYN;
"RST"			logentry->tcpflags |= TCP_RST;
"PSH"			logentry->tcpflags |= TCP_PSH;
"ACK"			logentry->tcpflags |= TCP_ACK;
"URG"			logentry->tcpflags |= TCP_URG;

"on interface "{STRING}		logentry->input_iface = cisco_pixtext + 13;
"by access-group \""{STRING}"\""	cisco_pix_parse_group(cisco_pixtext + 17);
"due to DNS "("Query"|"Response")							/* ignore */
"("{IPADDR}")"						/* ignore */
[ ]+		/* ignore whitespace */
[\n]		/* ignore */
{STRING}	if (verbose) fprintf(stderr, _("Unrecognized token: %s\n"), cisco_pixtext);
.		if (verbose) fprintf(stderr, _("Unrecognized character: %s\n"), cisco_pixtext);

%%

static void
cisco_pix_parse_date(char *input, unsigned char mode) {
  int day, hour, minute, second;
  char smonth[4], hostname[33];
#ifdef IRIX
  char tmp[SHOSTLEN];
#endif
#ifdef LOGDOTS
  char *remove_dot;
#endif
  
  if (mode == CP_OPT_HOST) {
    if (sscanf(input, "%3s %2d %2d:%2d:%2d %32s",
	       smonth, &day, &hour, &minute, &second,
#ifndef IRIX
		    hostname) != 6)
     return;
   logentry->hostname = hostname;
#else
		    tmp) != 6)
     return;
   if (tmp[2] == ':')
     logentry->hostname = tmp + 3;
#endif
     
#ifdef LOGDOTS
    remove_dot = strstr(opt.line->hostname, ".");
    if (remove_dot != NULL)
      *remove_dot = '\0';
#endif
  }
  else if (mode == CP_OPT_NONE) {
    int year;
    if (sscanf(input, "%3s %2d %4d %2d:%2d:%2d:",
      smonth, &day, &year, &hour, &minute, &second) != 5)
     return;
  }

  logentry->start_time = build_time(smonth, day, hour, minute, second);

  // |CISCO_PIX_DATE;
}

static void
cisco_pix_parse_src(char* input, unsigned char mode) {
  char ip[IPLEN];
  int shost1, shost2, shost3, shost4, sport;

  if (mode == CP_OPT_TCP || mode == CP_OPT_UDP) {
    if (sscanf(input, "%3d.%3d.%3d.%3d/%5d",
	       &shost1, &shost2, &shost3, &shost4, &sport) != 5)
      return;
    logentry->sport = sport;
	if (mode == CP_OPT_TCP)
      logentry->protocol = IPPROTO_TCP;
    else
      logentry->protocol = IPPROTO_UDP;
  }
  else if (mode == CP_OPT_TCP_S || mode == CP_OPT_UDP_S) {
    char buf[BUFSIZE + 1], *pnt;
    int sport;
    pnt = strstr(input, ":");
    *pnt = ' ';
    if (sscanf(input, "%" BUFSIZE_S "s %3d.%3d.%3d.%3d/%5d",
	       buf, &shost1, &shost2, &shost3, &shost4, &sport) != 6)
      return;
    logentry->sport = sport;
    if (mode == CP_OPT_TCP_S)
      logentry->protocol = IPPROTO_TCP;
    else
      logentry->protocol = IPPROTO_UDP;
  }
  else if (mode == CP_OPT_ICMP) {
    logentry->protocol = IPPROTO_ICMP;
    if (sscanf(input, "%3d.%3d.%3d.%3d ",
	       &shost1, &shost2, &shost3, &shost4) != 4)
      return;
  }

  snprintf(ip, IPLEN, "%d.%d.%d.%d", shost1, shost2, shost3, shost4);
  if (logentry->sipaddr.set(ip) == false)
    return;

  //  |CISCO_PIX_SRC;
}

static void
cisco_pix_parse_dst(char *input, unsigned char mode) {
  char ip[IPLEN];
  int dhost1, dhost2, dhost3, dhost4, dport;

  if (mode == CP_OPT_DST) {
    if (sscanf(input, "%3d.%3d.%3d.%3d/%5d",
	       &dhost1, &dhost2, &dhost3, &dhost4, &dport) != 5)
      return;
    logentry->dport = dport;
  }
  else if (mode == CP_OPT_DST_S) {
    char buf[BUFSIZE + 1], *pnt;
    pnt = strstr(input, ":");
    *pnt = ' ';
    if (sscanf(input, "%" BUFSIZE_S "s %3d.%3d.%3d.%3d/%5d",
	       buf, &dhost1, &dhost2, &dhost3, &dhost4, &dport) != 6)
      return;
    logentry->dport = dport;
  }
  else if (mode == CP_OPT_NONE) {
    if (sscanf(input, "%3d.%3d.%3d.%3d",
	       &dhost1, &dhost2, &dhost3, &dhost4) != 4)
      return;
  }

  snprintf(ip, IPLEN, "%d.%d.%d.%d", dhost1, dhost2, dhost3, dhost4);
  if (logentry->dipaddr.set(ip) == false)
    return;

  // |CISCO_PIX_DST;
}

static void
cisco_pix_parse_group(char* input) {
  char *pnt = strstr(input, "\"");
  *pnt = '\0';
  logentry->chainlabel = input;
}


wf_inmodule_cisco_pix::wf_inmodule_cisco_pix() {
}

bool
wf_inmodule_cisco_pix::match(const string& line) const {
  return (line.find(" %PIX-") != string::npos ||
          line.find(" %FWSM-") != string::npos);
}

enum wf_logentry_parsing_result
wf_inmodule_cisco_pix::parse(wf_logentry** retlogentry,
                             const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
    return WF_LOGENTRY_PARSING_ERROR;
     
  cisco_pix_scan_string(line.c_str());
  cisco_pixlex();

  logentry->count = 1;
  logentry->format = "cisco_pix";
  logentry->chainlabel = "-";
  logentry->branchname = "Deny";

  /*
  if (parser == (CISCO_PIX_DATE|CISCO_PIX_SRC|CISCO_PIX_DST))
  */
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (verbose)
    fprintf(stderr, _("cisco_pix parse error in line %d, ignoring.\n"),
	    linenum);

  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_cisco_pix_init() {
  return new wf_inmodule_cisco_pix();
}
