/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <list>

#include "wflogentries.h"
#include "wf_sort.h"
#include "output_sort.h"
#include "list1.h"
#include "defs.h"


wf_logentries::wf_logentries() :
  elems()
{}

wf_logentries::wf_logentries(const wf_logentries& logentries) {
  list1_deepcopy(elems, logentries.elems);
}

wf_logentries&
wf_logentries::operator=(const wf_logentries& logentries) {
  if (this != &logentries)
    list1_deepcopy(elems, logentries.elems);
  return *this;
}


bool
wf_logentries::add(const wf_logentry& logentry) {
  elems.push_back((wf_logentry*)&logentry); /* MV@@7 what shall I do ? */
  return true;
}

static void
entry_merge(wf_logentry& entry1, const wf_logentry& entry2) {
  time_t entry2_end_time; /* because entry2 is const */

  /* end_time values are not always defined: define them */

  if (entry1.end_time == 0)
    entry1.end_time = entry1.start_time;

  if (entry2.end_time == 0)
    entry2_end_time = entry2.start_time;
  else
    entry2_end_time = entry2.end_time;

  if (entry2.start_time < entry1.start_time)
    entry1.start_time = entry2.start_time;
  if (entry2_end_time > entry1.end_time)
    entry1.end_time = entry2_end_time;

  entry1.count += entry2.count;
}

static inline bool
approxequality1(const wf_logentry& entry1, const wf_logentry& entry2) {
  if (entry1.protocol != entry2.protocol)
    return false;

  if (entry1.protocol == IPPROTO_ICMP) /* types and codes must match exactly */
    return (entry1.sport == entry2.sport && /* type */
	    entry1.dport == entry2.dport && /* code */
	    entry1.sipaddr == entry2.sipaddr &&
	    entry1.dipaddr == entry2.dipaddr);

  /* compare only destination port */
  return (entry1.dport == entry2.dport &&
	  entry1.sipaddr == entry2.sipaddr &&
	  entry1.dipaddr == entry2.dipaddr);
}

static inline bool
approxequality2(const wf_logentry& entry1, const wf_logentry& entry2) {
  if (entry1.protocol != entry2.protocol)
    return false;

  if (entry1.protocol == IPPROTO_ICMP) /* types and codes must match exactly */
    return (entry1.sport == entry2.sport && /* type */
	    entry1.dport == entry2.dport && /* code */
	    entry1.sipaddr == entry2.sipaddr &&
	    entry1.dipaddr == entry2.dipaddr);

  /* compare only source port */
  return (entry1.sport == entry2.sport &&
	  entry1.sipaddr == entry2.sipaddr &&
	  entry1.dipaddr == entry2.dipaddr);
}

bool
wf_logentries::summary(unsigned int* deleted_entries) {
  wf_sort mysort;
  if (mysort.available_set(format_array, format_array_count) == false)
    return false;

  if (deleted_entries != NULL)
    *deleted_entries = 0;
  list<wf_logentry*>::iterator first, last, next;

  mysort.parse("protocol,dipaddr,dport,sipaddr");
  elems.sort(mysort);

  first = elems.begin();
  last = elems.end();
  if (first == last)
    return true;
  next = first;
  while (++next != last) {
    if (approxequality1(**first, **next)) {
      entry_merge(**first, **next);
      elems.erase(next);
      if (deleted_entries != NULL)
	++*deleted_entries;
    }
    else
      first = next;
    next = first;
  }

  mysort.criterias_clear();
  mysort.parse("protocol,sipaddr,sport,dipaddr");
  elems.sort(mysort);

  first = elems.begin();
  last = elems.end();
  if (first == last)
    return true;
  next = first;
  while (++next != last) {
    if (approxequality2(**first, **next)) {
      entry_merge(**first, **next);
      elems.erase(next);
      if (deleted_entries != NULL)
	++*deleted_entries;
    }
    else
      first = next;
    next = first;
  }

  return true;
}

bool
wf_logentries::filter(wflogs_filter& myfilter, unsigned int* deleted_entries) {
  if (deleted_entries != NULL) {
    *deleted_entries = 0;

    /* Unfortunately T.remove_if returns void, so we cannot gather information
       about the number of deleted entries (through myfilter object).
       But as for_each(iter, iter, pred) returns pred, we can use it for this
       purpose, but we have to iterate twice, which is ugly.
    */

    myfilter.count_reset();
    wflogs_filter F = for_each(elems.begin(), elems.end(), myfilter);
    *deleted_entries =  F.count_get();
  }

  elems.remove_if(myfilter);
  return true;
}

bool
wf_logentries::obfuscate(const wflogs_obfuscator& myobfuscator) {
  for_each(elems.begin(), elems.end(), myobfuscator);
  return true;
}

ostream&
wf_logentries::debugprint(ostream& os) const {
  list<wf_logentry*>::const_iterator first = elems.begin(), last = elems.end();
  for (; first != last; ++first)
    (*first)->debugprint(os) << endl;
  return os;
}
