/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <arpa/inet.h> /* for inet_[an]to[na]() */
#include <netinet/in.h> /* for struct in_addr on FreeBSD */
#include <netdb.h> /* for gethostbyaddr() */
#include <sys/socket.h> /* for AF_INET on NetBSD */

#include "wfipaddr.h"
#include "wfindent.h"
#include "defs.h"


wf_ipaddr::wf_ipaddr() :
  _addr(0),
  _defined(false)
{}

wf_ipaddr::wf_ipaddr(uint32_t address) {
  set(address);
}

wf_ipaddr::wf_ipaddr(const string& str) {
  set(str);
}

bool
wf_ipaddr::set(uint32_t address) {
  _addr = address;
  _defined = true;
  return true;
}

bool
wf_ipaddr::set(const string& str) {
  if (this->check(str) == false) /* inet_aton checks are not sufficient */
    return false;

  struct in_addr saddr;
  if (inet_aton(str.c_str(), &saddr) == 0)
    return false;

  return set(saddr.s_addr);
}

bool
wf_ipaddr::isdefined() const {
  return _defined;
}

uint32_t
wf_ipaddr::get() const {
  if (_defined == false) {
    return 0;
    throw "error"; // RV@@6
  }
  return _addr;
}

string
wf_ipaddr::tostr() const {
  string str;
  if (_defined) {
    struct in_addr saddr;
  
    saddr.s_addr = _addr;
    str = inet_ntoa(saddr);
  }
  return str;
}

bool
wf_ipaddr::isnull() const {
  return _defined && (_addr == 0);
}

bool
wf_ipaddr::isloopback() const {
  uint32_t address = htonl(_addr);
  return ((address & 0xFF000000) == 0x7F000000);
}

bool
wf_ipaddr::isroutable() const {
  if (_defined == false)
    return false; /* invalid */

  uint32_t address = htonl(_addr);
  if ((address & 0xFF000000) == 0x0A000000 || /* 10.0.0.0/8      */
      (address & 0xFFF00000) == 0xAC100000 || /* 172.16.0.0/12   */
      (address & 0xFFFF0000) == 0xC0A80000 || /* 192.168.0.0/16  */
      (address & 0xFF000000) == 0x7F000000 || /* 127.0.0.0/8     */
      address == 0xFFFFFFFF)                  /* 255.255.255.255 */
    return false; /* not routable */

  return true; /* routable */
}

bool
wf_ipaddr::dnsname(string& name) const {
  if (_defined == false)
    return false;

  struct hostent* reverse;
  struct in_addr ip;
  ip.s_addr = _addr;
  reverse = gethostbyaddr((char*)&ip.s_addr, sizeof(struct in_addr), AF_INET);
  if (reverse == NULL)
    return false;
  if (reverse->h_name != NULL)
    name = reverse->h_name;
  else
    name.erase();
  return true;
}


ostream&
wf_ipaddr::print(ostream& os) const {
  string str = tostr();
  if (str.empty())
    os << _("(undefined)");
  else
    os << str;
  return os;
}

ostream&
wf_ipaddr::output_xml(ostream& os, unsigned int indent_level) const {
  return os << wf_indent(indent_level) << "<ipaddr>" << tostr() <<
    "</ipaddr>" << endl;
}

ostream&
wf_ipaddr::debugprint(ostream& os) const {
  return os << _("IP address:\t") << *this << endl;
}

ostream&
operator<<(ostream& os, const wf_ipaddr& ipaddr) {
  return ipaddr.print(os);
}

bool
operator==(const wf_ipaddr& ipaddr1, const wf_ipaddr& ipaddr2) {
  return (ipaddr1._defined == ipaddr2._defined &&
	  ipaddr1._addr == ipaddr2._addr);
}

bool
operator!=(const wf_ipaddr& ipaddr1, const wf_ipaddr& ipaddr2) {
  return !(ipaddr1 == ipaddr2);
}

bool
operator<(const wf_ipaddr& ipaddr1, const wf_ipaddr& ipaddr2) {
  if (ipaddr1._defined == false || ipaddr2._defined == false) {
    return false;
    throw "error"; // RV@@6
  }
  return (htonl(ipaddr1._addr) < htonl(ipaddr2._addr));
}

bool
operator>(const wf_ipaddr& ipaddr1, const wf_ipaddr& ipaddr2) {
  if (ipaddr1._defined == false || ipaddr2._defined == false) {
    return false;
    throw "error"; // RV@@6
  }
  return (htonl(ipaddr1._addr) > htonl(ipaddr2._addr));
}


/* make it less C_ish RV@@2 */
bool
wf_ipaddr::check(const char* str) {
  const char *delim = ".";
  char *number, *ptr, *dup;
  int i;

  if (str == NULL || *str == '\0')
    return false;

  dup = ptr = strdup(str);

  for (i = 0; i < 4; i++) {
    char *p;
    int num;

    if ((number = strsep(&ptr, delim)) == NULL)
      goto check_destr;

    num = strtol(number, &p, 10);
    if (*p != '\0' || num < 0 || num > 255)
      goto check_destr;
  }

  if (ptr == NULL) {
    free(dup);
    return true;
  }

 check_destr:
  free(dup);
  return false;
}

bool
wf_ipaddr::check(const string& str) {
  struct in_addr saddr;
  return (wf_ipaddr::check(str.c_str()) &&
	  inet_aton(str.c_str(), &saddr) != 0);
}
