/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "listbox.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "pushbutton.h"
#include "filler.h"
#include "box.h"

#include <sigc++/object_slot.h>

namespace wftk {

//the width is fixed, so that the buttons fit nicely ...
ListBox::ListBox() :
  SingleContainer(0, 0, 0, 0),
  lock_(false),
  selected_(0)
{
  getResourceBackground("listbox");

  Color::Resource *listbox_selected_color = Color::registry.get("listbox_selected_color");
  selectedCol_ = listbox_selected_color ? listbox_selected_color->res()
    : "white";

  Box *mainBox = new Box(Box::LEFT_TO_RIGHT);
  pack(mainBox);

  // do the scrollbar

  Box *scrollbarBox =  new Box(Box::TOP_TO_BOTTOM);
  mainBox->packBack(scrollbarBox);

  scrollUp_ = new PushButton();
  scrollbarBox->packFront(scrollUp_);

  scrollUp_->setUpPic("arrow_up");
  scrollUp_->setDownPic("arrow_up_pressed");
  //  scrollUp_->setRepeat(true);

  scrollbarBox->packBack(new Filler(Filler::VERTICAL)); // filler

  scrollDown_ = new PushButton();
  scrollbarBox->packBack(scrollDown_);

  scrollDown_->setUpPic("arrow_down");
  scrollDown_->setDownPic("arrow_down_pressed");
  //scrollDown_->setRepeat(true);

  scrollUp_->pressed.connect(slot(*this, &ListBox::scrollUp));
  scrollDown_->pressed.connect(slot(*this, &ListBox::scrollDown));

  // create the button list

  buttonBox_ = new Box(Box::TOP_TO_BOTTOM);
  mainBox->packFront(buttonBox_);
  // make sure any extra space doesn't go into the scrollbar, also
  // handle the case where we have no buttons gracefully
  buttonBox_->packFront(new Filler(Filler::HORIZONTAL, 0,
	PackingInfo::Expander::FILL / 2));

  //disable the scrollers
  scrollUp_->disable();
  scrollDown_->disable();
}

void
ListBox::setSelectedColor(const Color& color)
{ 
  selectedCol_ = color;
  for(Elements::iterator I = elements_.begin(); I != elements_.end(); ++I) {
    if(I->second == selected_)
      I->second->setColor(selectedCol_);
    else
      I->second->setTextColor(selectedCol_);
  }

  invalidate();
}

void
ListBox::setColor(const Color& color)
{ 
  Widget::setColor(color);

  if(selected_)
    selected_->setTextColor(color);
}

void
ListBox::setSelected(const std::string& item)
{
  Elements::iterator I = !item.empty() ? elements_.find(item) : elements_.end();

  setSelectedLabel(I != elements_.end() ? I->second : 0);
}

void
ListBox::setSelectedElement(ScreenArea* sa)
{
    if(sa) {
      Button* button = dynamic_cast<Button*>(sa);
      assert(button);
      Label* label = dynamic_cast<Label*>(button->child());
      assert(label);
      setSelectedLabel(label);
    }
    else
      setSelectedLabel(0);
}

void
ListBox::setSelectedLabel(Label* new_selection)
{
  // if it's the current selection, don't do anything
  if(new_selection == selected_)
    return;

  if(selected_) {
    selected_->setColor("transparent");
    selected_->setTextColor(selectedCol_);
  }

  selected_ = new_selection;

  if(selected_) {
    selected_->setColor(selectedCol_);
    selected_->setTextColor(Widget::color());
  }

  selectionChanged(selected_ ? selected_->getText() : "");
}

void ListBox::addElementImpl(const std::string& data)
{
  if(data == "")
    return;

  std::pair<Elements::iterator,bool> result =
    elements_.insert(Elements::value_type(data, 0));

  if(!result.second) // duplicate label
    return;

  Elements::iterator I = result.first;

  I->second = new Label(data);
  I->second->setTextColor(selectedCol_);
  Button* button = new Button();
  button->pack(I->second);
  buttonBox_->packBack(button);
  button->pressed.connect(SigC::slot(*this, &ListBox::selected));
}

void ListBox::addElement(const std::string& data)
{
  addElementImpl(data);

  packingUpdate();
  invalidate();
}

void ListBox::addElements(const std::vector<std::string>& data)
{
  for(unsigned int i=0; i < data.size(); i++)
    addElementImpl(data[i]);

  packingUpdate();
  invalidate();
}

void ListBox::clear()
{
  elements_.clear();
  selected_ = 0;
}

void ListBox::scrollUp()
{
  if(!selected_)
    return;

  Box::iterator I = buttonBox_->find(*selected_);
  assert(I != buttonBox_->end());
  setSelectedElement(I == buttonBox_->begin() ? 0 : &*(--I));
}

void ListBox::scrollDown()
{
  if(!selected_)
    return;

  Box::iterator I = buttonBox_->find(*selected_);
  assert(I != buttonBox_->end());
  setSelectedElement(++I == buttonBox_->end() ? 0 : &*I);
}

void 
ListBox::selected(Button* button)
{
  if(!button)
    return;

  Debug out(Debug::GENERIC);

  Label* new_selection = (button->child() == selected_) ? selected_
	: dynamic_cast<Label*>(button->child());
  assert(new_selection);
  
  out << "ListBox: selected(" << new_selection->getText() << ")" << Debug::endl;

  if(!lock_ && button->child() != selected_) {
    lock_ = true; // avoid recursive calls to selected()
    setSelectedLabel(new_selection);
    lock_ = false;
  }
}

}

