/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "table.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include <set>
//#include <iostream>

bool
wftk::Table::pack(ScreenArea* sa, unsigned x, unsigned y, unsigned w, unsigned h)
{
  if(!sa || sa->parent() == this || !w || !h)
    return false;

  if(!left_) { // we haven't started the grid yet
    assert(!top_);
    left_ = new GridEdge;
    top_ = new GridEdge;
    top_->elems = left_->elems = new GridElem;
  }
  else
    assert(top_);

  GridEdge *edge = left_;

  while(y && edge->next) {
    edge = edge->next;
    --y;
  }

  GridElem *corner = edge->elems;

  unsigned need_rows = (h - 1) + y;
  while(need_rows) {
    if(!edge->next) { // add a row
      edge->next = new GridEdge;
      GridElem *elem = edge->elems, *new_elem = 0;
      while(elem) {
        elem->down = new GridElem;
        if(new_elem)
          new_elem->right = elem->down;
        else
          edge->next->elems = elem->down;
        new_elem = elem->down;
        elem = elem->right;
      }
    }
    edge = edge->next;
    --need_rows;
  }

  while(y) { // advance corner down into the new rows if necessary
    corner = corner->down;
    --y;
  }

  while(x && corner->right) {
    corner = corner->right;
    --x;
  }

  unsigned need_columns = (w - 1) + x;

  if(x == 0) { // corner is in the right place, check if we have enough space
    GridElem *tmp = corner->right;
    while(tmp && need_columns) {
      tmp = tmp->right;
      --need_columns;
    }
  }

  if(need_columns) {
    // make 'edge' be the top right corner
    edge = top_;
    while(edge->next)
      edge = edge->next;

    // and add the columns

    while(need_columns) {
    edge->next = new GridEdge;
    GridElem *elem = edge->elems, *new_elem = 0;
    while(elem) {
      elem->right = new GridElem;
      if(new_elem)
        new_elem->down = elem->right;
      else
        edge->next->elems = elem->right;
      new_elem = elem->right;
      elem = elem->down;
    }
    edge = edge->next;
    --need_columns;
    }
  }

  while(x) { // advance corner into the new columns if necessary
    corner = corner->right;
    --x;
  }

  // We now have the upper left hand corner, and enough space
  // in the grid for the widget. Check for overlap with existing
  // children.

  GridElem *side = corner;
  for(unsigned tmp_h = 0; tmp_h < h; ++tmp_h) {
    assert(side);
    GridElem *elem = side;
    for(unsigned tmp_w = 0; tmp_w < w; ++tmp_w) {
      assert(elem);
      if(elem->content) // there's already a child here
        return false;
      elem = elem->right;
    }
    side = side->down;
  }

  // okay, we're clear, insert the child

  side = corner;
  for(unsigned tmp_h = 0; tmp_h < h; ++tmp_h) {
    assert(side);
    GridElem *elem = side;
    for(unsigned tmp_w = 0; tmp_w < w; ++tmp_w) {
      assert(elem && !elem->content);
      elem->content = sa;
      elem = elem->right;
    }
    side = side->down;
  }

  Debug::channel(Debug::PACKING) << "Added child " << sa->name() << " " << sa
	<< " to Table" << Debug::endl << Debug::endl << Debug::endl;

  sa->setParent(this);

  return true;
}

void
wftk::Table::remove(ScreenArea* sa)
{
  if(!sa)
    return;

  GridElem *corner = 0;
  GridEdge *edge = left_;
  while(edge) {
    GridElem *elem = edge->elems;
    while(elem) {
      if(elem->content == sa) {
        corner = elem;
        break;
      }
      elem = elem->right;
    }
    edge = edge->next;
  }

  if(!corner)
    return;

  // we have the upper left corner of the widget,
  // remove it from the grid

  do {
    GridElem* elem = corner;
    do {
      elem->content = 0;
      elem = elem->right;
    } while(elem && elem->content == sa);
    corner = corner->down;
  } while(corner && corner->content == sa);

  sa->setParent(0);

  packingUpdate();
}

void
wftk::Table::clear()
{
  // keep track of children to remove

  std::set<ScreenArea*> children;

  GridEdge *edge = left_;
  while(edge) {
    GridElem *elem = edge->elems;
    while(elem) {
      if(elem->content)
        children.insert(elem->content);
      elem = elem->right;
    }
    edge = edge->next;
  }

  std::set<ScreenArea*>::iterator I;
  for(I = children.begin(); I != children.end(); ++I)
    (*I)->setParent(0);

  freeGrid();

  packingUpdate();
}

wftk::Table::GridElem*
wftk::Table::gridPos(unsigned x, unsigned y) const
{
  if(!top_ || !left_)
    return 0;

  return top_->elems->traverse(x, y);
}

// The complier gets confused if this isn't here, even though
// screenarea.h is included. Weird.
struct wftk::ScreenArea::PackingInfo::Expander;

wftk::ScreenArea::PackingInfo::Expander
wftk::Table::getRowPackingInfo(unsigned i) const
{
  GridEdge* edge = left_ ? left_->traverse(i) : 0;
  return edge ? edge->packing : PackingInfo::Expander();
}

wftk::ScreenArea::PackingInfo::Expander
wftk::Table::getColumnPackingInfo(unsigned i) const
{
  GridEdge* edge = top_ ? top_->traverse(i) : 0;
  return edge ? edge->packing : PackingInfo::Expander();
}

void
wftk::Table::handleResize(Uint16 w, Uint16 h)
{
  ScreenArea::handleResize(w, h);

  // calculate row/column sizes

  x_weight_.setExpand(packing_info_.x.pref, w);
  setPixels(top_, x_weight_);
  y_weight_.setExpand(packing_info_.y.pref, h);
  setPixels(left_, y_weight_);

  // resize the children

  // keep track of which children we've resized
  std::set<ScreenArea*> handled;

  Uint16 y_pos = 0;
  for(GridEdge* left_edge = left_; left_edge; left_edge = left_edge->next) {
    Uint16 x_pos = 0;
    GridElem *elem = left_edge->elems;
    ScreenArea *current = 0; // don't check repeatedly for multicolumn widgets

    for(GridEdge* top_edge = top_; top_edge; top_edge = top_edge->next) {
      assert(elem);
      if(!elem->content || elem->content == current
        || !handled.insert(elem->content).second) {
        current = elem->content;
        x_pos += top_edge->pixels;
        elem = elem->right;
        continue;
      }

      // we haven't resized this child yet

      current = elem->content;

      // need to find width and height

      Uint16 child_w = 0, child_h = 0;

      GridEdge *new_left = left_edge;
      GridElem *new_elem = elem;

      do {
        assert(new_left);
        child_h += new_left->pixels;
        new_elem = new_elem->down;
        new_left = new_left->next;
      } while(new_elem && new_elem->content == current);

      // for h, we don't need to make copies of top_edge and elem,
      // since we need to skip past the area contained by this widget
      // to find the next widget anyway

      while(true) {
        assert(top_edge);
        child_w += top_edge->pixels;
        elem = elem->right;
        if(!elem || elem->content != current)
          break;
        top_edge = top_edge->next;
      }

      Rect size(x_pos, y_pos, child_w, child_h);

      const PackingInfo& info = current->getPackingInfo();

      if(!info.x.expand && child_w > info.x.pref) {
        size.x += (child_w - info.x.pref) / 2;
        size.w = info.x.pref;
      }
      if(!info.y.expand && child_h > info.y.pref) {
        size.y += (child_h - info.y.pref) / 2;
        size.h = info.y.pref;
      }

      current->resize(size);

      x_pos += child_w;

      // already advanced elem to elem->right in while loop
    }
    y_pos += left_edge->pixels;
  }
}

void
wftk::Table::setPackingInfo()
{
  // get the per-row and per-column info

  Debug out(Debug::PACKING);

  out << "Called Table::setPackingInfo()" << Debug::endl;

  SpanList left_spans, top_spans;

  std::set<ScreenArea*> handled;

  for(GridEdge *top_edge = top_; top_edge; top_edge = top_edge->next) {
    top_edge->packing = PackingInfo::Expander();
    top_edge->packing.expand = false;
  }

  for(GridEdge *left_edge = left_; left_edge; left_edge = left_edge->next) {
    left_edge->packing = PackingInfo::Expander();
    left_edge->packing.expand = false;

    out << "Got a left edge" << Debug::endl;

    GridElem *elem = left_edge->elems;
    for(GridEdge *top_edge = top_; top_edge; top_edge = top_edge->next) {
      assert(elem);

      out << "Got a table element" << Debug::endl;

      if(!elem->content || !handled.insert(elem->content).second) {
        elem = elem->right;
        continue;
      }

      Debug::channel(Debug::PACKING) << "Adding packing info for "
	<< elem->content->name() << " to Table" << Debug::endl;

      const PackingInfo& info = elem->content->getPackingInfo();

      if(!elem->down || elem->down->content != elem->content) {
        // only span one row
        left_edge->packing.contain(info.y);
        // Don't subtract PackingInfo::Expander::SUBTRACT here,
        // we do that when we combine the edges to get the total
        // packing info.
        if(info.y.expand && info.y.filler > left_edge->packing.filler)
          left_edge->packing.filler = info.y.filler;
      }
      else {
// FIXME
#if 0
        left_spans.push_back(GridSpan());
        left_spans.back().min = info.y.min;
        left_spans.back().pref = info.y.pref;
        left_spans.back().start = left_edge;
        left_spans.back().length = 1;

        GridElem *copy = elem;
        GridEdge *left_copy = left_edge;
        while(true) {
          if(info.y.expand) {
            left_copy->packing.expand = true;
            if(info.y.filler > left_copy->packing.filler)
              left_copy->packing.filler = info.y.filler;
          }
          if(!copy->down || elem->content != copy->down->content)
            break;
          ++left_spans.back().length;
          copy = copy->down;
          left_copy = left_copy->next;
          assert(left_copy);
        }
#endif
      }

      if(!elem->right || elem->right->content != elem->content) {
        // only span one row
        top_edge->packing.contain(info.x);
        // Don't subtract PackingInfo::Expander::SUBTRACT here,
        // we do that when we combing the edges to get the total
        // packing info.
        if(info.x.expand && info.x.filler > top_edge->packing.filler)
          top_edge->packing.filler = info.x.filler;
      }
      else {
// FIXME
#if 0
        top_spans.push_back(GridSpan());
        top_spans.back().min = info.x.min;
        top_spans.back().pref = info.x.pref;
        top_spans.back().start = top_edge;
        top_spans.back().length = 1;

        // advance elem, top_edge here; we need to skip over
        // this part of the grid anyway because it's all the same widget

        while(true) {
          if(info.x.expand) {
            top_edge->packing.expand = true;
            if(info.x.filler > top_edge->packing.filler)
              top_edge->packing.filler = info.x.filler;
          }
          if(!elem->right || elem->content != elem->right->content)
            break;
          ++top_spans.back().length;
          elem = elem->right;
          top_edge = top_edge->next;
          assert(top_edge);
        }
#endif
      }

      elem = elem->right;
    }
  }

  // incorporate the information from spans into the edge prefered sizes
  incorporateSpans(top_, top_spans);
  incorporateSpans(left_, left_spans);

  // set packing info based on per-row/per-column stuff

  packingFromEdge(top_, packing_info_.x, x_weight_);
  packingFromEdge(left_, packing_info_.y, y_weight_);
}

void
wftk::Table::freeGrid()
{
  // free the edge elements on the top edge

  while(top_) {
    GridEdge *tmp = top_;
    top_ = top_->next;
    delete tmp;
  }

  // and free the rest, going down the left

  while(left_) {
    GridEdge *tmp = left_;
    left_ = left_->next;
    GridElem *elem = tmp->elems;
    delete tmp;
    while(elem) {
      GridElem *tmp2 = elem;
      elem = elem->right;
      delete tmp2;
    }
  }

  // and left_ and top_ are automagically zero 
}

bool
wftk::Table::insertEdge(unsigned index, bool down)
{
  GridEdge *edge = down ? top_ : left_, *prev = 0;

  while(edge && index) {
    prev = edge;
    edge = edge->next;
    --index;
  }

  // insert at end doesn't have any effect
  if(!edge)
    return false;

  GridEdge *new_edge = new GridEdge;
  if(prev)
    prev->next = new_edge;
  else if(down)
    top_ = new_edge;
  else
    left_ = new_edge;
  new_edge->next = edge;

  GridElem *prev_elem = prev ? prev->elems : 0, *next_elem = edge->elems, *new_elem = 0;

  while(next_elem) {
    GridElem* tmp = new GridElem;
    if(!new_elem)
      new_edge->elems = tmp;
    else {
      new_elem->down = down ? tmp : next_elem;
      new_elem->right = down ? next_elem : tmp;
    }

    if(prev_elem) {
      if(down)
        prev_elem->right = new_elem;
      else
        prev_elem->down = new_elem;
    }

    new_elem = tmp;

    // check for widgets spanning this row/column
    if(prev_elem && prev_elem->content == next_elem->content)
      new_elem->content = next_elem->content;

    // advance bordering elements
    if(prev_elem)
      prev_elem = down ? prev_elem->down : prev_elem->right;
    next_elem = down ? next_elem->down : next_elem->right;
  }

  packingUpdate();

  return true;
}

void
wftk::Table::packingFromEdge(GridEdge* edge,
	PackingInfo::Expander& info, PackingInfo::Weights& weights)
{
  weights = PackingInfo::Weights();
  info = PackingInfo::Expander();

  info.expand = false;

  while(edge) {
    Debug::channel(Debug::PACKING) << "Expanding table info for an edge of min size "
 	<< edge->packing.min << " and pref size " << edge->packing.pref << Debug::endl;
    info.extend(edge->packing);
    weights.extend(edge->packing);
    edge = edge->next;
  }
}

void
wftk::Table::setPixels(GridEdge* edge, PackingInfo::Weights& weights)
{
  double space_filled = 0;
  Uint16 pixels_filled = 0;

  while(edge) {
    space_filled += edge->packing.pref + weights.padding(edge->packing);
    Uint16 new_pixels_filled = (Uint16) (space_filled + 0.5); // round
    edge->pixels = new_pixels_filled - pixels_filled;
    pixels_filled = new_pixels_filled;
    edge = edge->next;
  }
}

void
wftk::Table::incorporateSpans(GridEdge* edge, SpanList& spans)
{
#if 0
  std::cerr << "Got " << spans.size() << " spans" << std::endl;

  for(SpanList::iterator I = spans.begin(); I != spans.end(); ++I) {
    std::cerr << "One of length " << (unsigned) I->length << ", with min size " << I->min
	<< " and pref size " << I->pref << std::endl;
    I->calcOverage();
    std::cerr << " and overage of (" << I->min_overage << ',' << I->pref_overage
	<< ')' << std::endl;
  }
#endif

#if 0
  while(!spans.empty()) {
    for(SpanList::iterator I = spans.begin(); I != spans.end(); ++I)
      I->calcOverage();

    sort(spans.begin(), spans.end());

    // get rid of any spans whose requirements we already satisfy
    while(spans.front().min_overage == 0 && spans.front().pref_overage == 0) {
      spans.erase(spans.begin());
      if(spans.empty())
        return;
    }

    // heuristic, add enough space to satisfy the largest one of
    // the spans' requirements

    if(spans.back().min_overage != 0) {
      GridEdge *edge = spans.back().start;
      Uint16 space = spans.back().min_overage;
      unsigned char edges_left = spans.back().length;
      assert(edges_left);
      do {
        Uint16 padding = space / edges_left;
        space -= padding;
        assert(edge);
        edge->packing.min += padding;
        if(edge->packing.pref < edge->packing.min)
          edge->packing.pref = edge->packing.min;
        edge = edge->next;
      } while(--edges_left);
    }
    else {
      assert(spans.back().pref_overage != 0);
      GridEdge *edge = spans.back().start;
      Uint16 space = spans.back().pref_overage;
      unsigned char edges_left = spans.back().length;
      assert(edges_left);
      do {
        Uint16 padding = space / edges_left;
        space -= padding;
        assert(edge);
        edge->packing.pref += padding;
        edge = edge->next;
      } while(--edges_left);
      spans.pop_back(); // we've satisfied this one
    }
  }
#endif
}

wftk::Table::GridElem*
wftk::Table::GridElem::traverse(unsigned x, unsigned y)
{
  GridElem* elem = this;

  while(x-- != 0 && elem)
    elem = elem->right;

  while(y-- != 0 && elem)
    elem = elem->down;

  return elem;
}

wftk::Table::GridEdge*
wftk::Table::GridEdge::traverse(unsigned i)
{
  GridEdge* elem = this;

  while(i-- != 0 && elem)
    elem = elem->next;

  return elem;
}

void
wftk::Table::GridSpan::calcOverage()
{
  GridEdge* edge = start;

  Uint16 calc_min = 0, calc_pref = 0;

  for(unsigned char i = 0; i < length; ++i) {
    assert(edge);
    calc_min += edge->packing.min;
    calc_pref += edge->packing.pref;
    edge = edge->next;
  }

  min_overage = min > calc_min ? min - calc_min : 0;
  pref_overage = pref > calc_pref ? pref - calc_pref : 0;
}

// want to sort by overage
bool
wftk::Table::GridSpan::operator<(const GridSpan& g) const
{
  if(min_overage != g.min_overage)
    return min_overage < g.min_overage;

  if(pref_overage != g.pref_overage)
    return pref_overage < g.pref_overage;

  // given equal overage, take the longest sequence
  if(length != g.length)
    return length < g.length;

  // tiebreaker
  return start < g.start;
}
