/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

// writtten May 1999



#ifndef _VIDEO_H
#define _VIDEO_H

#include <wftk/sound.h>
#include <wftk/point.h>
#include <wftk/region.h>
#include <wftk/mutex.h>
#include <wftk/surface.h>

#include <string>
#include <map>

namespace wftk {

class ScreenSurface;

/// Wrapper class for loading and playing mpeg video
class Video : public Sound, public Surface
{
 public:

  /// create a Video instance
  Video(const Point& screen_loc = Point(0, 0), ScreenSurface* screen = 0);
  ///
  ~Video();

  // Don't call any of these three when screen_->mutex is locked
  // (i.e., in any draw() or drawAfter() widget function)

  /// set the MPEG drawing mask
  void setMask(const Region& r);
  /// clear the MPEG drawing mask
  void clearMask();
  /// move the video to a different location on the screen
  void move(const Point&);

  /// overrides Sound::load() pure virtual
  virtual bool load(const std::string& file) {return load(file, 1);}
  /// load MPEG video, and configure it to display resized by factor 'scale'
  virtual bool load(const std::string&, double scale);  
  /// play the video
  virtual bool play(int loops = 0) const;

  /// set volume (0..100)
  virtual int setVolume(int vol);  
  /// cease playback
  virtual void stop() const;
  /// true if music is playing
  virtual bool playing() const;
  /// true if music is paused
  virtual bool paused() const {return paused_;}
  /// pause playback
  virtual void pause();
  /// rewind song to beginning
  virtual void rewind() const; 
  /// resume song (if paused)
  virtual void resume();

 private:
  /// looks up video from surf, calls update()
  static void screenUpdate(SDL_Surface* surf, int x, int y, unsigned w, unsigned h);
  /// blit video to screen
  void update(const Rect&);

  ///
  typedef std::map<SDL_Surface*,Video*> SurfaceMap;
  ///
  static SurfaceMap map_;
  /// need to lock mutex_ before locking this
  static Mutex map_mutex_;

  /// guards access to loc_, mask_, Surface
  Mutex mutex_;
  ///
  bool use_mask_;
  ///
  Region mask_;
  /// screen location
  Point loc_;

  /// This will make life easier when we get multiple root windows
  ScreenSurface* screen_;

  ///
  void* mpeg; // SMPEG*
  ///
  bool paused_;
};

} // namespace

#endif
