#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

#include "common.h"
#include "config.h"

void DynString::add(char* c) {
  while(*c) add(*(c++));
}

// A class for reading a line or secition contest from a file:
class FileRead {
  DynString curLine;
  DynString curSection;
  char* savedline;
  int Lineno;
  
  // File handling:  
  FILE* f;

  public:
  FileRead() : savedline(0), f(0), Lineno(0) {}
  ~FileRead() { if(f) fclose(f); }  
  
  // Returns 0 on succes and -1 on error:
  int init(char* filename);
  
  char* readLine();  // Reads a single line - returns 0 on error/EOF
  char* readLines(); // Read all lines up to and the next '[' starting line - returns 0 on error/EOF
  
  int lineno() { return Lineno; }
};    

int FileRead::init(char* filename) {
  f=fopen(filename,"r");  
  if(f==NULL) return -1;  
  Lineno=1;
  return 0;
}

char* FileRead::readLine() {
  if(savedline) { char* c=savedline; savedline=0; return c; }
  for(;;) {
    // Read line:
    curLine.setEmpty();
    int c=0;
    for(;;) {
      c=fgetc(f);
      if(c=='\n') Lineno++;
      if(c==EOF) break;
      if(c=='\\') {
        int d=fgetc(f);
	if(d=='\n') { Lineno++; continue; } // escaped newline
	ungetc(d,f);
      } else if(c=='\n') {
        break;
      }
      curLine.add(c);
    }
    curLine.add('\x0');
  
    // Remove leading and trailing spaces:
    char* s=curLine.getString();
    while(isspace(*s)) s++;
    int l; for(l=strlen(s)-1; l>=0 && isspace(s[l]); l--);
    s[l+1]='\x0';
  
    // Read new line if line is a comment or is empty
    if(s[0]=='\x0' || s[0]==';' || s[0]=='#' || (s[0]=='/' && s[1]=='/')) {
      if(c==EOF) return 0;
      continue;
    }
  
    return s;
  } 
}  

char* FileRead::readLines() {
  curSection.setEmpty();
  for(;;) {
    char* r=readLine();
    if((r==0) || (r[0]=='[')) { 
      savedline=r;
      curSection.add('\x0');
      return curSection.getString(); 
    }
    curSection.add(r);
    curSection.add('\n');
  }
}

// Finally define the readConfFile class:
char* Config::readConfFile(char* filename) {
  FileRead fr;
  if(fr.init(filename)!=0) { snprintf(errmsg,sizeof(errmsg),"%s: Can't open file",filename); return errmsg; }    
  
  char* l=fr.readLine();
  while(l) {    
    if(!strcmp(l,"[counters]")) {
      for(;;) {
        l=fr.readLine();
        if(l==0 || l[0]=='[') break;
        if(Counterc==maxcounterc) { snprintf(errmsg,sizeof(errmsg),"%s: Too many counters defined",filename); return errmsg; }
        strncpy(Counternames[Counterc],l,MaxCounterNameLen);
        Counternames[Counterc][MaxCounterNameLen]='\x0';
        Counterc++;
      }
    } else if(!strcmp(l,"[options]")) {
      for(;;) {
        l=fr.readLine();
	if(l==0 || l[0]=='[') break;
	char* opt=strchr(l,':');	
	if(opt==0) { snprintf(errmsg,sizeof(errmsg),"%s: No colon found in option line",filename); return errmsg; }
	*opt='\x0';
	opt++; while(isspace(*opt)) opt++;
	if(!strcmp(l,"logaddrchange")) { Logaddrchange=atoi(opt); }
	else if(!strcmp(l,"logbadpackets")) { Logbadpackets=atoi(opt); }
	else if(!strcmp(l,"logtablefull")) { Logtablefull=atoi(opt); }
	else if(!strcmp(l,"promiscmode"))   { Promiscmode=atoi(opt); }
	else if(!strcmp(l,"maxentries")) { Maxcardc=atoi(opt); }
	else if(!strcmp(l,"logfile")) { Logfile=new char[strlen(opt)+1]; strcpy(Logfile,opt); }
	else if(!strcmp(l,"pidfile")) { Pidfile=new char[strlen(opt)+1]; strcpy(Pidfile,opt); }
	else if(!strcmp(l,"daemonfile")) { Daemonfile=new char[strlen(opt)+1]; strcpy(Daemonfile,opt); }
	else if(!strcmp(l,"netdev")) { Netdev=new char[strlen(opt)+1]; strcpy(Netdev,opt); }
	else if(!strcmp(l,"filter")) { Filter=new char[strlen(opt)+1]; strcpy(Filter,opt); }
	else if(!strcmp(l,"proxymaxcon")) { Proxymaxcon=atoi(opt); }
	else if(!strcmp(l,"addrmode")) { 
	  if(!strcmp(opt,"ip"))
   	    Cit=idIP;
	  else if(!strcmp(opt,"mac"))
   	    Cit=idMAC;
	  else {
	    snprintf(errmsg,sizeof(errmsg),"%s: Unknown addressing mode: %s",filename,opt); return errmsg; 
	  }	    
	}
	else if(!strcmp(l, "user")) {
	  struct passwd * pwd;

	  pwd=getpwnam(opt);
	  if(pwd == NULL)
	    Uid=atoi(opt);
	  else
	    Uid=pwd->pw_uid;

	  if(Uid == 0) {
	    snprintf(errmsg,sizeof(errmsg),"%s: Invalid username or uid (0 is not accepted): %s",filename,opt); 
	    return errmsg;
	  }
	} 
	else if(!strcmp(l, "group")) {
	  struct group * grp;

	  grp=getgrnam(opt);
	  if(grp == NULL)
	    Gid=atoi(opt);
	  else
	    Gid=grp->gr_gid;

	  if(Gid == 0) {
	    snprintf(errmsg,sizeof(errmsg),"%s: Invalid group name or gid (0 is not accepted): %s",filename,opt); 
	    return errmsg;
	  }
	}
	else { snprintf(errmsg,sizeof(errmsg),"%s: Unknown option: %s",filename,l); return errmsg; }
      }	
    } else if(!strcmp(l,"[program]")) {
      Prgline=fr.lineno();
      char* p=fr.readLines();
      Program=new char[strlen(p)+1];
      strcpy(Program,p);
      l=fr.readLine();
    }
  }
  return 0;    
}

