//------------------------------------------------------------------------------
// This module maintains a shared memory area of type:

#define ShMemT void

// between different processes. More precisely a server process creates
// the shared memory area and the area is only valid as long as the server
// process lives. Different client processes can then get access to the memory.
//
// The module maintains a lock machanisme on the area to ensure mutive access.
//
//------------------------------------------------------------------------------

#include <assert.h>

// See ShMemClient::Lock:
#define LockErrString "Server terminated or permission denied"

// Interface for client processes:
class ShMemClient {
  ShMemT* mem;
  
  protected:
  int shmid;
  int semid;
  int locked;
  int readonly;

  // Returns -1 on error
  int wait();   
  int signal(); 

  public:
  ShMemClient()  { locked=0; }
  ~ShMemClient() { assert(!locked); }
  
  // file    : Should be the same as given to the server
  // readonly: Should be true if the memory is only going to be read.
  // Returns an error message on failure and 0 on succes:
  char* init(const char* file, int readonly);
  
  // Locks the memory for exclusive access. 
  // Returns 0 on failure - for example if the server is died.
  // On failure LockErrString contains an approprite error message
  virtual ShMemT* Lock();
  
  // Unlocks the locked memory:
  virtual void UnLock(ShMemT*);
};

// Interface for the server process. Note that the server is also a client
// to the memory area.
class ShMemServer:public ShMemClient {
  ShMemT* memory;
  
  static void signalHandler(void*);
  char* int_init(const char* file, int size, ShMemT*& Lock);
  void releaseipc();
  
  public:
  virtual ShMemT* Lock();
  virtual void UnLock(ShMemT*);

  ShMemServer();
  ~ShMemServer();
  
  // This inits the shared memory area.
  // file: A filename which is used to identify the memory area.
  //       The permissons of the file determines the permissons of the
  //       shared memory area. Note that a process with read access will
  //       not be able to modify the area. But it can lock it forever...
  // size: Size in bytes of the shared memory area
  //
  // Returns an error message on failere and 0 on success.
  //
  // On success the area is automatically locked and the lock location
  // is saved in the last parameter. This makes it possible for the server
  // to initialize the memory before any clients can access it.
  char* init(const char* file, int size, ShMemT*& Lock);
};
